﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.XPath;

namespace GitExternalRepository.Repository.Svn
{
    /// <summary>
    /// 外部 svn コマンドを利用して機能を提供するクラスです。
    /// </summary>
    [RepositoryDefinition(RepositoryType.Svn)]
    public class SvnRepositoryCommand : SvnRepositoryBase
    {
        private string workingDirectory;

        public SvnRepositoryCommand(string workingDirectory)
        {
            this.workingDirectory = workingDirectory;
        }

        #region SvnRepositoryBase の抽象メソッド実装

        public override void Init()
        {
            SvnCommandUtils.RunSvnAdmin(".", string.Format("create \"{0}\"", this.workingDirectory));
        }

        public override bool CanSeparateMetaDirectory()
        {
            return false;
        }

        public override void Clone(string url, string revision)
        {
            var revisionInfo = ParseRevisionString(revision);
            var checkoutUrl = string.Format("{0}/{1}", url, revisionInfo.DirectoryPath);

            SvnCommandUtils.RunSvn(".", string.Format("checkout --non-interactive {0} \"{1}\" --revision {2}", checkoutUrl, this.workingDirectory, revisionInfo.Revision));
        }

        public override void Clone(string url, string revision, string gitDirectory)
        {
            throw new NotImplementedException();
        }

        public override void Clone(string url, CloneCheckoutOption cloneCheckoutOption)
        {
            if (cloneCheckoutOption == CloneCheckoutOption.None)
            {
                throw new NotImplementedException();
            }

            SvnCommandUtils.RunSvn(".", string.Format("checkout --non-interactive {0} \"{1}\"", url, this.workingDirectory));
        }

        public override void Clone(string url, CloneCheckoutOption cloneCheckoutOption, string gitDirectory)
        {
            throw new NotImplementedException();
        }

        public override void Open()
        {
            // do nothing
        }

        public override string GetMetaDirectory()
        {
            return PathUtility.Combine(this.GetRepositoryRoot(), ".svn");
        }

        public override string GetRepositoryRoot()
        {
            var xml = SvnCommandUtils.GetSvnOutput(this.workingDirectory, "info --non-interactive --xml");
            var xdocument = XDocument.Parse(xml);

            return xdocument.XPathSelectElement("//wcroot-abspath").Value;
        }

        public override string GetRepositoryRootUrl()
        {
            var xml = SvnCommandUtils.GetSvnOutput(this.workingDirectory, "info --non-interactive --xml");
            var xdocument = XDocument.Parse(xml);

            return xdocument.XPathSelectElement("//root").Value;
        }

        public override bool IsInsideRepository()
        {
            try
            {
                // svn info を実行して、例外が発生しなければリポジトリ内にいるものとみなす。
                // 余計な出力を出さないよう、RunSvn ではなく、GetSvnOutput を用います。
                SvnCommandUtils.GetSvnOutput(this.workingDirectory, "info --non-interactive");

                return true;
            }
            catch (SvnRepositoryCommandFailedException)
            {
                return false;
            }
        }

        public override bool IsRepositoryRoot()
        {
            var metaPath = Path.Combine(this.workingDirectory, ".svn");

             // 直下に .svn ディレクトリを持つかどうか判断したのち、正式なチェックを行います。
            if (Directory.Exists(metaPath))
            {
                return this.IsInsideRepository() && PathUtility.AreSame(this.GetRepositoryRoot(), this.workingDirectory);
            }

            return false;
        }

        public override string GetConfig(string name)
        {
            throw new NotImplementedException();
        }

        public override void SetConfig(string name, string value)
        {
            throw new NotImplementedException();
        }

        public override string GetRevision(string reference)
        {
            var xml = SvnCommandUtils.GetSvnOutput(this.workingDirectory, string.Format("info --non-interactive --revision \"{0}\" --xml", reference));
            var xdocument = XDocument.Parse(xml);
            var revisionInfo = new RevisionInfo
            {
                DirectoryPath = this.GetCheckoutDirectoryPath(),
                Revision = xdocument.XPathSelectElement("//entry").Attribute("revision").Value,
            };

            return this.GetRevisionString(revisionInfo);
        }

        public override string GetHeadRevision()
        {
            var revisionString = SvnCommandUtils.GetSvnVersionOutput(this.workingDirectory, "--no-newline");
            var revisionInfo = new RevisionInfo
            {
                DirectoryPath = this.GetCheckoutDirectoryPath(),
                Revision = this.ParseSvnVersionString(revisionString),
            };

            return this.GetRevisionString(revisionInfo);
        }

        public override void Add(string filepath)
        {
            SvnCommandUtils.RunSvn(this.workingDirectory, string.Format("add --non-interactive \"{0}\"", filepath));
        }

        public override void Commit(string message)
        {
            SvnCommandUtils.RunSvn(this.workingDirectory, string.Format("commit --non-interactive -m \"{0}\"", message));
        }

        public override void Checkout(string revision, bool force)
        {
            var revisionInfo = this.ParseRevisionString(revision);

            var forceOptionString = string.Empty;
            if (force)
            {
                forceOptionString = "--force";

                // svn switch では変更点を破棄してくれないので、ここでリバートします。
                SvnCommandUtils.RunSvn(this.workingDirectory, string.Format("revert --recursive ."));
            }

            SvnCommandUtils.RunSvn(this.workingDirectory, string.Format("switch --non-interactive \"^{0}\" --revision \"{1}\" {2}", revisionInfo.DirectoryPath, revisionInfo.Revision, forceOptionString));
        }

        public override void Clean()
        {
            // 無視対象も含めて列挙する
            var output = SvnCommandUtils.GetSvnOutput(this.workingDirectory, "status --non-interactive --no-ignore");
            // 行にパースする
            var lines = output.Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries).ToList();
            // 無視または管理対象外の行をピックアップする
            var targets = lines.Where(x => x.StartsWith("I") || x.StartsWith("?"));
            // パス部分を取得
            var paths = targets.Select(x => PathUtility.Combine(this.workingDirectory, x.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries)[1]));

            foreach (var path in paths)
            {
                if (File.Exists(path))
                {
                    File.Delete(path);
                }
                else if (Directory.Exists(path))
                {
                    Directory.Delete(path, true);
                }
            }
        }

        public override IEnumerable<string> GetHeadTree()
        {
            // ディレクトリ、ファイルを含むツリーの出力
            var revisionInfo = this.ParseRevisionString(this.GetHeadRevision());
            var output = SvnCommandUtils.GetSvnOutput(this.workingDirectory, string.Format("list --non-interactive -R -r {0}", revisionInfo.Revision));
            // 行にパースする
            var lines = output.Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries).ToList();
            // ディレクトリ('/' で終わる) を取り除く
            var files = lines.Where(x => !x.EndsWith("/"));

            return files;
        }

        public override bool IsMetaDirectory()
        {
            throw new NotImplementedException();
        }

        #endregion

        /// <summary>
        /// svn のチェックアウトに用いた URL を取得します。
        /// </summary>
        /// <returns>チェックアウトに用いた URL</returns>
        private string GetRepositoryCheckoutUrl()
        {
            var xml = SvnCommandUtils.GetSvnOutput(this.workingDirectory, "info --non-interactive --xml");
            var xdocument = XDocument.Parse(xml);

            return xdocument.XPathSelectElement("//url").Value;
        }

        /// <summary>
        /// リポジトリ内のチェックアウトしたディレクトリのパスを取得します。
        /// </summary>
        /// <returns>ディレクトリパス</returns>
        private string GetCheckoutDirectoryPath()
        {
            var rootUrl = this.GetRepositoryRootUrl();
            var checkoutUrl = this.GetRepositoryCheckoutUrl();
            var directoryPath = checkoutUrl.Replace(rootUrl, string.Empty);

            return directoryPath != string.Empty ? directoryPath : "/";
        }

        private static readonly Regex SvnVersionPatternRegex = new Regex("^([0-9]+)[MSP]*$");
        private static readonly Regex SvnVersionPatternMixedRegex = new Regex("^([0-9]+):([0-9]+)[MSP]*$");

        /// <summary>
        /// svnversion が出力するバージョン文字列をパースし、現在のリビジョンを取得します。
        /// </summary>
        /// <param name="versionString">バージョン文字列</param>
        /// <returns>現在のリビジョン</returns>
        private string ParseSvnVersionString(string versionString)
        {
            {
                var match = SvnVersionPatternRegex.Match(versionString);
                if (match.Success)
                {
                    return match.Groups[1].Value;
                }
            }

            {
                var match = SvnVersionPatternMixedRegex.Match(versionString);
                if (match.Success)
                {
                    return match.Groups[2].Value;
                }
            }

            throw new SvnRepositoryCommandFailedException(string.Format("'{0}' is invalid svn version format.", versionString));
        }

        private static readonly Regex RevisionPatternRegex = new Regex("^(.*):([0-9]+)$");

        /// <summary>
        /// リビジョン情報
        /// </summary>
        private struct RevisionInfo
        {
            /// <summary>
            /// チェックアウトしたサブディレクトリ
            /// </summary>
            public string DirectoryPath;

            /// <summary>
            /// リビジョン番号
            /// </summary>
            public string Revision;
        }

        /// <summary>
        /// リビジョン文字列をパースして、リビジョン情報を取得します。
        /// </summary>
        /// <param name="revisionString">リビジョン文字列</param>
        /// <returns>リビジョン情報</returns>
        private RevisionInfo ParseRevisionString(string revisionString)
        {
            var match = RevisionPatternRegex.Match(revisionString);
            if (match.Success)
            {
                return new RevisionInfo
                {
                    DirectoryPath = match.Groups[1].Value,
                    Revision = match.Groups[2].Value,
                };
            }

            throw new SvnRepositoryCommandFailedException(string.Format("'{0}' is invalid svn format.", revisionString));
        }

        /// <summary>
        /// リビジョン情報からリビジョン文字列を生成します。
        /// </summary>
        /// <param name="revisionInfo">リビジョン情報</param>
        /// <returns>リビジョン文字列</returns>
        private string GetRevisionString(RevisionInfo revisionInfo)
        {
            return string.Format("{0}:{1}", revisionInfo.DirectoryPath, revisionInfo.Revision);
        }
    }
}
