﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using Nintendo.Foundation.IO;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 外部リポジトリの初期化処理を行うコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class UpdateReferenceCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineValue(0, ValueName = "path", Description = "Specify an external repository directory.")]
            public string Path { get; set; }
        }

        public UpdateReferenceCommand()
        {
        }

        public UpdateReferenceCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;

            var repositoryList = Environments.RepositoryList;

            if (!repositoryList.Exists())
            {
                throw new GitExternalRepositoryException(string.Format("The repository list '{0}' does not exist.", repositoryList.FilePath));
            }

            var entries = repositoryList.Read();

            // 与えられた引数
            var path = commandParameters.Path.TrimSlash();
            // 親リポジトリのルートからの相対パス
            var relatedPathFromRoot = PathUtility.GetRelativePath(Environments.GitRepository.GetRepositoryRoot() + @"\", path);

            var entry = entries.FirstOrDefault(x => x.TargetDirectory == relatedPathFromRoot);
            if (entry != null)
            {
                this.UpdateReference(entry);
            }
            else
            {
                throw new GitExternalRepositoryException(string.Format("'{0}' does not exist in the repository list '{1}'.", relatedPathFromRoot, repositoryList.FilePath));
            }

            return true;
        }

        /// <summary>
        /// 指定のエントリのリビジョンを現在のリビジョンで更新します。
        /// </summary>
        /// <param name="entry"></param>
        private void UpdateReference(RepositoryListEntry entry)
        {
            this.CommandConsole.WriteLineToOut("Update the reference revision of '{0}'", entry.TargetDirectory);

            // ディレクトリパス
            var path = PathUtility.Combine(Environments.GitRepository.GetRepositoryRoot(), entry.TargetDirectory);

            // 外部リポジトリの GitRepository インスタンス
            var exrepo = Environments.RepositoryFactory.Create(entry.RepositoryType, path);
            // 外部リポジトリの現在のリビジョン
            var exrepoRevision = exrepo.GetHeadRevision();

            this.CommandConsole.WriteLineToOut("Updating to the new revision '{0}'", exrepoRevision);

            // リポジトリリストのエントリを現在のリビジョンで更新します。
            var repositoryList = Environments.RepositoryList;
            repositoryList.Add(new RepositoryListEntry()
            {
                TargetDirectory = entry.TargetDirectory,
                Url = entry.Url,
                Revision = exrepoRevision,
                RepositoryType = exrepo.GetRepositoryDefinitionAttribute().RepositoryType,
            });

            this.CommandConsole.WriteLineToOut("Done.");
        }
    }
}
