﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/util/util_FormatString.h>
#include <algorithm>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/diag/diag_LogTypes.h>
#include <nn/diag/detail/diag_DetailLog.h>

/**
* @brief utilTool 用のログ定義
*
*   SDK のライブラリは NN_LOG を使えないが、
*   ここではツールとしてのログを定義したいので
*   NN_LOG の定義をコピーしてある
*/
#define NN_UTILTOOL_LOG_RAW(...)                                    \
do                                                                  \
{                                                                   \
    ::nn::diag::LogMetaData logMetaData;                            \
    logMetaData.sourceInfo.lineNumber = __LINE__;                   \
    logMetaData.sourceInfo.fileName = __FILE__;                     \
    logMetaData.sourceInfo.functionName = NN_CURRENT_FUNCTION_NAME; \
    logMetaData.moduleName = "";                                    \
    logMetaData.severity = ::nn::diag::LogSeverity_Info;            \
    logMetaData.verbosity = 0;                                      \
    logMetaData.useDefaultLocaleCharset = false;                    \
    logMetaData.pAdditionalData = nullptr;                          \
    logMetaData.additionalDataBytes = 0;                            \
    ::nn::diag::detail::LogImpl(logMetaData, __VA_ARGS__);          \
} while (NN_STATIC_CONDITION(0))

namespace nn {
    namespace utilTool {

// diag_DebugPrintf.cpp からコピーした。
// 実装で NN_LOG を使っているため、更に内部で同じコードが走るため無駄がある。
// diag か、その付近で適切なログ出力インタフェースが定義されたらそれを使う。
        namespace {
            // VPrintf, Printf が内部で使用するバッファサイズです。
            const int DebugPrintfBufferLength = 128;

            class FormatStringOutputBufferedDebugPrint
            {
            public:
                FormatStringOutputBufferedDebugPrint(char* pBuffer, int length) NN_NOEXCEPT
                    : m_pBegin(pBuffer),
                    // 出力時に '\0' を入れるため pBuffer + length にはしない
                    m_pEnd(pBuffer + length - 1),
                    m_pCurrent(pBuffer)
                {
                }

                void Flush() NN_NOEXCEPT
                {
                    *m_pCurrent = '\0';
                    NN_UTILTOOL_LOG_RAW("%s", m_pBegin);
                    m_pCurrent = const_cast<char*>(m_pBegin);
                }

                static void Output(uintptr_t arg, const char* pCharacters, int count) NN_NOEXCEPT
                {
                    reinterpret_cast<FormatStringOutputBufferedDebugPrint*>(arg)->PutCharacters(pCharacters, count);
                }

            private:
                void PutCharacters(const char* pCharacters, size_t count) NN_NOEXCEPT
                {
                    while (count > 0)
                    {
                        const size_t bufferLeft = m_pEnd - m_pCurrent;
                        size_t copyCount = std::min(bufferLeft, count);
                        std::memcpy(m_pCurrent, pCharacters, copyCount);

                        m_pCurrent += copyCount;
                        pCharacters += copyCount;
                        count -= copyCount;

                        if (m_pCurrent == m_pEnd)
                        {
                            Flush();
                        }
                    }
                }

                // 出力用バッファの先頭
                char const* m_pBegin;
                // 出力用バッファの末尾
                char const* m_pEnd;
                // 出力用バッファの現在位置
                char* m_pCurrent;
            };

            void VPrintf(const char* pFormat, std::va_list formatArg) NN_NOEXCEPT
            {
                char outputBuffer[DebugPrintfBufferLength];
                FormatStringOutputBufferedDebugPrint out(outputBuffer, DebugPrintfBufferLength);
                nn::util::VFormatString(out.Output, reinterpret_cast<uintptr_t>(&out), pFormat, formatArg);
                out.Flush();
            }

            void Printf(const char* pFormat, ...) NN_NOEXCEPT
            {
                std::va_list formatArg;

                va_start(formatArg, pFormat);
                VPrintf(pFormat, formatArg);
                va_end(formatArg);
            }
        } // anonymous namespace

        namespace
        {
            nn::util::IntrusiveList<ICommandLogObserver, nn::util::IntrusiveListBaseNodeTraits<ICommandLogObserver>> s_CommandLogObserverHolder;
        } // anonymous namespace

        bool NeedCommandLog(CommandLogLevel level) NN_NOEXCEPT
        {
            for (auto observer = s_CommandLogObserverHolder.begin(); observer != s_CommandLogObserverHolder.end(); observer++)
            {
                if (observer->NeedLog(level))
                {
                    return true;
                }
            }

            return false;
        }

        void SendCommandLogV(const CommandLogMetaInfo& logMetaInfo, const char* format, std::va_list formatArg) NN_NOEXCEPT
        {
            for (auto observer = s_CommandLogObserverHolder.begin(); observer != s_CommandLogObserverHolder.end(); observer++)
            {
                std::va_list formatArgCopy;

                NN_CSTD_VA_COPY(formatArgCopy, formatArg);

                const CommandLogMessage message = { format, &formatArgCopy };

                observer->SendLog(logMetaInfo, message);

                va_end(formatArgCopy);
            }
        }

        void SendCommandLog(const CommandLogMetaInfo& logMetaInfo, const char* format, ...) NN_NOEXCEPT
        {
            std::va_list list;
            va_start(list, format);
            SendCommandLogV(logMetaInfo, format, list);
            va_end(list);
        }

        void RegisterCommandLogObserver(ICommandLogObserver &observer)
        {
            s_CommandLogObserverHolder.push_back(observer);
        }

        void UnregisterCommandLogObserver(ICommandLogObserver &observer)
        {
            auto node = std::find_if(s_CommandLogObserverHolder.begin(), s_CommandLogObserverHolder.end(), [&observer](ICommandLogObserver &i){ return &observer == &i; });
            NN_ABORT_UNLESS(node != s_CommandLogObserverHolder.end());
            s_CommandLogObserverHolder.erase(node);
        }

        bool DefaultCommandLogObserver::NeedLog(CommandLogLevel level)
        {
            return m_CurrentLevel <= level;
        }
        void DefaultCommandLogObserver::SendLog(const CommandLogMetaInfo& metaInfo, const CommandLogMessage& message)
        {
            if (m_IsEnabledTagDisplay)
            {
                for (int i = 0; i < metaInfo.numTags; i++)
                {
                    Printf("[%s] ", metaInfo.tags[i]);
                }
            }

            VPrintf(message.format, *message.args);

            if (m_IsEnabledPositionDisplay)
            {
                const char *nameonly = (strrchr(metaInfo.fileName, '\\') ? strrchr(metaInfo.fileName, '\\') + 1 : metaInfo.fileName);
                Printf(" (%s:%d)\n", nameonly, metaInfo.lineNumber);
            }
            else
            {
                Printf("\n");
            }
        }

        void DefaultCommandLogObserver::SetLevel(CommandLogLevel level)
        {
            m_CurrentLevel = level;
        }

        void DefaultCommandLogObserver::SetPositionDisplay(bool isEnabled)
        {
            m_IsEnabledPositionDisplay = isEnabled;
        }

        void DefaultCommandLogObserver::SetTagDisplay(bool isEnabled)
        {
            m_IsEnabledTagDisplay = isEnabled;
        }
    }
}
