﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  This code is copied from CTR's source tree.
//  The following header comment is original copyright of CTR.
//-----------------------------------------------------------------------------

/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     math_TinyMt.cpp

  Copyright (C)2009-2013 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 50823 $
 *---------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_TinyMt.h>

namespace nn { namespace util {

namespace
{
    inline Bit32 MixMsb2(Bit32 v) NN_NOEXCEPT
    {
        return v ^ (v >> 30);
    }

    inline Bit32 MixMsb5(Bit32 v) NN_NOEXCEPT
    {
        return v ^ (v >> 27);
    }
}
// anonymouse namespace



/* ------------------------------------------------------------------------
        public
   ------------------------------------------------------------------------ */

void TinyMt::Initialize(Bit32 seed) NN_NOEXCEPT
{
    m_State[0] = seed;
    m_State[1] = ParameterMat1;
    m_State[2] = ParameterMat2;
    m_State[3] = ParameterTmat;;

    {
        for( int i = 1; i < MinLoop; ++i )
        {
            const Bit32 v = MixMsb2(m_State[(i - 1) % ParameterN]);
            m_State[i % ParameterN] ^= v * 0x6c078965 + i;
        }
    }

    FinalizeInitialization();
}

void TinyMt::Initialize(const Bit32* pSeed, int numSeed) NN_NOEXCEPT
{
    m_State[0] = 0;
    m_State[1] = ParameterMat1;
    m_State[2] = ParameterMat2;
    m_State[3] = ParameterTmat;;

    {
        const int minLoop = MinLoop;
        const int numLoop = std::max(numSeed + 1, minLoop) - 1;

        GenerateInitialValuePlus(m_State, 0, numSeed);

        int offset = 1;
        for( int i = 0; i < numLoop; ++i )
        {
            const Bit32 seed = (i < numSeed) ? pSeed[i]: 0;
            GenerateInitialValuePlus(m_State, (offset + i) % ParameterN, seed);
        }

        offset = numLoop + 1;
        for( int i = 0; i < ParameterN; ++i )
        {
            GenerateInitialValueXor(m_State, (offset + i) % ParameterN);
        }
    }

    FinalizeInitialization();
}

void TinyMt::SaveState(TinyMt::State* pStateBuffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pStateBuffer);

    std::memcpy(pStateBuffer->state, m_State, sizeof(pStateBuffer->state));
}

void TinyMt::RestoreState(const TinyMt::State* pStateBuffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pStateBuffer);

    std::memcpy(m_State, pStateBuffer->state, sizeof(m_State));
}

uint32_t TinyMt::GenerateRandomU32() NN_NOEXCEPT
{
    // 状態更新
    const Bit32 a = (m_State[0] & Bit31Mask) ^ m_State[1] ^ m_State[2];
    const Bit32 b = m_State[3];
    const Bit32 c = a ^ (a << 1);
    const Bit32 d = b ^ (b >> 1) ^ c;

    const Bit32 s0 = m_State[1];
          Bit32 s1 = m_State[2];
          Bit32 s2 = c ^ (d << 10);
    const Bit32 s3 = d;

    if( (d & 1) != 0 )
    {
        s1 ^= ParameterMat1;
        s2 ^= ParameterMat2;
    }

    m_State[0] = s0;
    m_State[1] = s1;
    m_State[2] = s2;
    m_State[3] = s3;

    // tempering
    Bit32 t = s0 + (s2 >> 8);
    Bit32 v = s3 ^ t;

    if( (t & 1) != 0 )
    {
        v ^= ParameterTmat;
    }

    return v;
}

void TinyMt::GenerateRandomBytes(void* p, size_t size) NN_NOEXCEPT
{
    uintptr_t begin  = reinterpret_cast<uintptr_t>(p);
    uintptr_t end    = begin + size;
    uintptr_t begin4 = util::align_up(begin, 4);
    uintptr_t end4   = util::align_down(end, 4);

    if( begin < begin4 )
    {
        Bit32 v = GenerateRandomU32();
        std::memcpy(reinterpret_cast<void*>(begin), &v, begin4 - begin);
    }

    {
        Bit32* p32  = reinterpret_cast<Bit32*>(begin4);
        Bit32* pEnd = reinterpret_cast<Bit32*>(end4);

        while( p32 < pEnd )
        {
            *p32++ = GenerateRandomU32();
        }
    }

    if( end4 < end )
    {
        Bit32 v = GenerateRandomU32();
        std::memcpy(reinterpret_cast<void*>(end4), &v, end - end4);
    }
}



/* ------------------------------------------------------------------------
        private
   ------------------------------------------------------------------------ */

void TinyMt::GenerateInitialValuePlus(Bit32* p, int d, Bit32 k) NN_NOEXCEPT
{
    Bit32& s0 = p[d];
    Bit32& s1 = p[(d + 1) % ParameterN];
    Bit32& s2 = p[(d + 2) % ParameterN];
    Bit32& s3 = p[(d + 3) % ParameterN];

    const Bit32 a = MixMsb5(s0 ^ s1 ^ s3) * 0x0019660d;
    const Bit32 b = a + d + k;

    s0 = b;
    s1 += a;
    s2 += b;
}

void TinyMt::GenerateInitialValueXor(Bit32* p, int d) NN_NOEXCEPT
{
    Bit32& s0 = p[d];
    Bit32& s1 = p[(d + 1) % ParameterN];
    Bit32& s2 = p[(d + 2) % ParameterN];
    Bit32& s3 = p[(d + 3) % ParameterN];

    const Bit32 a = MixMsb5(s0 + s1 + s3) * 0x5d588b65;
    const Bit32 b = a - d;

    s0 = b;
    s1 ^= a;
    s2 ^= b;
}

void TinyMt::FinalizeInitialization() NN_NOEXCEPT
{
    const Bit32 state031 = (m_State[0] & Bit31Mask);

    if( (state031   == 0) && (m_State[1] == 0)
     && (m_State[2] == 0) && (m_State[3] == 0) )
    {
        m_State[0] = 'T';
        m_State[1] = 'I';
        m_State[2] = 'N';
        m_State[3] = 'Y';
    }

    for( int i = 0; i < NumSkip; ++i )
    {
        GenerateRandomU32();
    }
}

}} // namespace nn::util
