﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_BinaryFormat.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

namespace nn { namespace util {

typedef util::BitPack16::Field< 0, 1, bool> FlagRelocationBit;
typedef util::BitPack16::Field< 0, 16, uint16_t> FlagField;

bool BinaryFileHeader::IsValid( int64_t packedSignature, int majorVersion, int minorVersion, int microVersion) const NN_NOEXCEPT
{
    bool result = true;

    if ( !IsSignatureValid( packedSignature ) )
    {
        result = false;
    }

    if ( !IsVersionValid( majorVersion, minorVersion, microVersion) )
    {
        result = false;
    }

#if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_BIG )
    if( !IsEndianReverse() )
#elif defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE )
    // nothing to do
#else
    #error Unknown NN_BUILD_TARGET_PLATFORM_ENDIAN
#endif
    {
        if ( !IsEndianValid() )
        {
            result = false;
        }
    }

    if ( !IsAlignmentValid() )
    {
        result = false;
    }

    return result;
}

bool BinaryFileHeader::IsSignatureValid( int64_t packedSignature ) const NN_NOEXCEPT
{
    if ( this->signature.IsValid( packedSignature ) )
    {
        return true;
    }
    else
    {
        BinFileSignature bin;
        bin.SetPacked( packedSignature );
        NN_SDK_LOG( "Signature check failed ('%02x %02x %02x %02x %02x %02x %02x %02x' must be '%02x %02x %02x %02x %02x %02x %02x %02x').\n",
            this->signature._str[ 0 ], this->signature._str[ 1 ],
            this->signature._str[ 2 ], this->signature._str[ 3 ],
            this->signature._str[ 4 ], this->signature._str[ 5 ],
            this->signature._str[ 6 ], this->signature._str[ 7 ],
            bin._str[ 0 ], bin._str[ 1 ], bin._str[ 2 ], bin._str[ 3 ],
            bin._str[ 4 ], bin._str[ 5 ], bin._str[ 6 ], bin._str[ 7 ] );

        return false;
    }
}

bool BinaryFileHeader::IsVersionValid(int majorVersion, int minorVersion, int microVersion) const NN_NOEXCEPT
{
    if ( this->version.IsValid( majorVersion, minorVersion, microVersion) )
    {
        return true;
    }
    else
    {
        NN_SDK_LOG( "Version check failed (bin:'%d.%d.%d', lib:'%d.%d.%d').\n",
            this->version.major, this->version.minor, this->version.micro,
            majorVersion, minorVersion, microVersion);
        return false;
    }
}

bool BinaryFileHeader::IsEndianValid() const NN_NOEXCEPT
{
    if ( this->_byteOrderMark == static_cast< uint16_t >( nn::util::ByteOrderMark_Normal ) )
    {
        return true;
    }
    else
    {
        NN_SDK_LOG( "Endian check failed ('0x%X' must be '0x%X').\n",
            this->_byteOrderMark, nn::util::ByteOrderMark_Normal );
        return false;
    }
}

bool BinaryFileHeader::IsEndianReverse() const NN_NOEXCEPT
{
    return this->_byteOrderMark == static_cast< uint16_t >( nn::util::ByteOrderMark_Reverse );
}

bool BinaryFileHeader::IsAlignmentValid() const NN_NOEXCEPT
{
    if ( util::ConstBytePtr( this ).IsAligned( GetAlignment() ) )
    {
        return true;
    }
    else
    {
        NN_SDK_LOG( "Alignment check failed ('%p' must be multiple of '%p').\n", this, GetAlignment() );
        return false;
    }
}

bool BinaryFileHeader::IsRelocated() const NN_NOEXCEPT
{
    util::BitPack16 flagPack;
    flagPack.Set< FlagField >( util::LoadLittleEndian( &this->_flag ) );
    return flagPack.Get< FlagRelocationBit >();
}

void BinaryFileHeader::SetRelocated( bool value ) NN_NOEXCEPT
{
    util::BitPack16 flagPack;
    flagPack.Set< FlagField >( util::LoadLittleEndian( &this->_flag ) );
    flagPack.Set< FlagRelocationBit >( value );
    util::StoreLittleEndian( &this->_flag, flagPack.Get< FlagField >() );
}

void BinaryFileHeader::SetByteOrderMark( nn::util::ByteOrderMark byteOrderMark ) NN_NOEXCEPT
{
    this->_byteOrderMark = static_cast< uint16_t >( byteOrderMark );
}

void BinaryFileHeader::SetAddressSize() NN_NOEXCEPT
{
    int addrSize = NN_BITSIZEOF( BinPtr ) ;
    util::StoreLittleEndian( &this->_targetAddressSize, static_cast< uint8_t >( addrSize ) );
}

size_t BinaryFileHeader::GetFileSize() const NN_NOEXCEPT
{
    return static_cast< size_t >( util::LoadLittleEndian( &this->_fileSize ) );
}

void BinaryFileHeader::SetFileSize( size_t size ) NN_NOEXCEPT
{
    util::StoreLittleEndian( &this->_fileSize, static_cast< uint32_t >( size ) );
}

size_t BinaryFileHeader::GetAlignment() const NN_NOEXCEPT
{
    return size_t( 1 ) << this->_alignmentShift;
}

void BinaryFileHeader::SetAlignment( size_t alignment ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( util::ispow2( alignment ) );
    this->_alignmentShift = static_cast< uint8_t >( util::cntt0( alignment ) );
}

util::string_view BinaryFileHeader::GetFileName() const NN_NOEXCEPT
{
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToFileName ) );
    if ( offset )
    {
        const char* name = util::ConstBytePtr( this, offset ).Get< char >();
        return util::string_view( name );
    }
    else
    {
        return util::string_view();
    }
}

void BinaryFileHeader::SetFileName( const util::string_view& name ) NN_NOEXCEPT
{
    if ( name.length() )
    {
        uint32_t offset = static_cast< uint32_t >( util::ConstBytePtr( this ).Distance( name.data() ) );
        util::StoreLittleEndian( &this->_offsetToFileName, offset );
    }
    else
    {
        this->_offsetToFileName = 0;
    }
}

RelocationTable* BinaryFileHeader::GetRelocationTable() NN_NOEXCEPT
{
    ptrdiff_t  offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToRelTable ) );
    if ( offset )
    {
        return util::BytePtr( this, offset ).Get< RelocationTable >();
    }
    else
    {
        return NULL;
    }
}

void BinaryFileHeader::SetRelocationTable( RelocationTable* pTable ) NN_NOEXCEPT
{
    if ( pTable )
    {
        uint32_t offset = static_cast< uint32_t >( util::ConstBytePtr( this ).Distance( pTable ) );
        util::StoreLittleEndian( &this->_offsetToRelTable, offset );
    }
    else
    {
        this->_offsetToRelTable = 0;
    }
}

BinaryBlockHeader* BinaryFileHeader::GetFirstBlock() NN_NOEXCEPT
{
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToFirstBlock ) );
    if ( offset )
    {
        return util::BytePtr( this, offset ).Get< BinaryBlockHeader >();
    }
    else
    {
        return NULL;
    }
}

const BinaryBlockHeader* BinaryFileHeader::GetFirstBlock() const NN_NOEXCEPT
{
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToFirstBlock ) );
    if ( offset )
    {
        return util::ConstBytePtr( this, offset ).Get< BinaryBlockHeader >();
    }
    else
    {
        return NULL;
    }
}

BinaryBlockHeader* BinaryFileHeader::FindFirstBlock( int32_t packedSignature ) NN_NOEXCEPT
{
    BinaryBlockHeader* pHeader = GetFirstBlock();
    if ( pHeader )
    {
        if ( pHeader->signature.GetPacked() == packedSignature )
        {
            return pHeader;
        }
        else
        {
            return pHeader->FindNextBlock(packedSignature );
        }
    }
    else
    {
        return NULL;
    }
}

const BinaryBlockHeader* BinaryFileHeader::FindFirstBlock( int32_t packedSignature ) const NN_NOEXCEPT
{
    const BinaryBlockHeader* pHeader = GetFirstBlock();
    if ( pHeader )
    {
        if ( pHeader->signature.GetPacked() == packedSignature )
        {
            return pHeader;
        }
        else
        {
            return pHeader->FindNextBlock( packedSignature );
        }
    }
    else
    {
        return NULL;
    }
}

void BinaryFileHeader::SetFirstBlock( BinaryBlockHeader* pBlock ) NN_NOEXCEPT
{
    if ( pBlock )
    {
        uint16_t offset = static_cast< uint16_t >( util::BytePtr( this ).Distance( pBlock ) );
        util::StoreLittleEndian( &this->_offsetToFirstBlock, offset );
    }
    else
    {
        this->_offsetToFirstBlock = 0;
    }
}

//--------------------------------------------------------------------------------------------------

BinaryBlockHeader* BinaryBlockHeader::GetNextBlock() NN_NOEXCEPT
{
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToNextBlock ) );
    if ( offset )
    {
        return util::BytePtr( this, offset ).Get< BinaryBlockHeader >();
    }
    else
    {
        return NULL;
    }
}

const BinaryBlockHeader* BinaryBlockHeader::GetNextBlock() const NN_NOEXCEPT
{
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_offsetToNextBlock ) );
    if ( offset )
    {
        return util::ConstBytePtr( this, offset ).Get< BinaryBlockHeader >();
    }
    else
    {
        return NULL;
    }
}

BinaryBlockHeader* BinaryBlockHeader::FindNextBlock( int packedSignature ) NN_NOEXCEPT
{
    BinaryBlockHeader* pHeader = GetNextBlock();
    while ( pHeader )
    {
        if ( pHeader->signature.GetPacked() == packedSignature )
        {
            return pHeader;
        }
        pHeader = pHeader->GetNextBlock();
    }
    return NULL;
}

const BinaryBlockHeader* BinaryBlockHeader::FindNextBlock( int packedSignature ) const NN_NOEXCEPT
{
    const BinaryBlockHeader* pHeader = GetNextBlock();
    while ( pHeader )
    {
        if ( pHeader->signature.GetPacked() == packedSignature )
        {
            return pHeader;
        }
        pHeader = pHeader->GetNextBlock();
    }
    return NULL;
}

size_t BinaryBlockHeader::GetBlockSize() const NN_NOEXCEPT
{
    return static_cast< size_t >( util::LoadLittleEndian( &this->_blockSize ) );
}

void BinaryBlockHeader::SetNextBlock( BinaryBlockHeader* pBlock ) NN_NOEXCEPT
{
    if ( pBlock )
    {
        uint32_t offset = static_cast< uint32_t >( util::BytePtr( this ).Distance( pBlock ) );
        util::StoreLittleEndian( &this->_offsetToNextBlock, offset );
    }
    else
    {
        this->_offsetToNextBlock = 0;
    }
}

void BinaryBlockHeader::SetBlockSize( size_t size ) NN_NOEXCEPT
{
    util::StoreLittleEndian( &this->_blockSize, static_cast< uint32_t >( size ) );
}

//--------------------------------------------------------------------------------------------------

NN_DEFINE_STATIC_CONSTANT( const int RelocationTable::PackedSignature );

void RelocationTable::Section::SetPtr( void* pSection ) NN_NOEXCEPT
{
    this->_ptr = pSection;
}

void* RelocationTable::Section::GetPtr() const NN_NOEXCEPT
{
    return this->_ptr;
}

void* RelocationTable::Section::GetPtrInFile( void* pFile ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pFile );
    ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_position ) );
    return util::BytePtr( pFile, offset ).Get();
}

void* RelocationTable::Section::GetBasePtr( void* pFile ) const NN_NOEXCEPT
{
    if ( void* ptr = GetPtr() )
    {
        ptrdiff_t offset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_position ) );
        return util::BytePtr( ptr, -offset ).Get();
    }
    else
    {
        // セクションのポインタが無指定の場合はファイル外に再配置されていないとみなします。
        return pFile;
    }
}

size_t RelocationTable::Section::GetSize() const NN_NOEXCEPT
{
    return static_cast< size_t >( util::LoadLittleEndian( &this->_size ) );
}

void RelocationTable::Relocate() NN_NOEXCEPT
{
    ptrdiff_t posTable = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_position ) );
    BinaryFileHeader* pFile = util::BytePtr( this, -posTable ).Get< BinaryFileHeader >();

    NN_SDK_ASSERT( !pFile->IsRelocated() );

    int length = util::LoadLittleEndian( &this->_sectionCount );
    const Entry* pEntries = reinterpret_cast< const Entry* >( &this->_sections[ length ] );
    for ( int idxSection = 0; idxSection < length; ++idxSection )
    {
        Section& sectionInfo = this->_sections[ idxSection ];
        void* pBase = sectionInfo.GetBasePtr( pFile );

        // 同一セクションを指すオフセットをまとめて変換します。
        int entryIndex = util::LoadLittleEndian( &sectionInfo._entryIndex );
        int entryCount = util::LoadLittleEndian( &sectionInfo._entryCount );
        for ( int endEntry = entryIndex + entryCount; entryIndex < endEntry; ++entryIndex )
        {
            const Entry& entry = pEntries[ entryIndex ];
            ptrdiff_t posOffset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &entry._position ) );
            int structCount = util::LoadLittleEndian( &entry._structCount );
            int offsetCount = util::LoadLittleEndian( &entry._offsetCount );
            int paddingCount = util::LoadLittleEndian( &entry._paddingCount );

            // 構造体の配列をまとめて変換します。
            BinPtr* pBinPtr = util::BytePtr( pFile, posOffset ).Get< BinPtr >();
            for ( int idxStruct = 0; idxStruct < structCount; ++idxStruct, pBinPtr += paddingCount )
            {
                for ( int idxOffset = 0; idxOffset < offsetCount; ++idxOffset, ++pBinPtr )
                {
                    pBinPtr->Relocate( pBase );
                }
            }
        }
    }

    pFile->SetRelocated( true );
}

void RelocationTable::Unrelocate() NN_NOEXCEPT
{
    ptrdiff_t posTable = static_cast< ptrdiff_t >( util::LoadLittleEndian( &this->_position ) );
    BinaryFileHeader* pFile = util::BytePtr( this, -posTable ).Get< BinaryFileHeader >();

    NN_SDK_ASSERT( pFile->IsRelocated() );

    int length = util::LoadLittleEndian( &this->_sectionCount );
    const Entry* pEntries = reinterpret_cast< const Entry* >( &this->_sections[ length ] );
    for ( int idxSection = 0; idxSection < length; ++idxSection )
    {
        Section& sectionInfo = this->_sections[ idxSection ];
        void* pBase = sectionInfo.GetBasePtr( pFile );
        sectionInfo.SetPtr( NULL ); // 再配置に備えてリセットします。

        // 同一セクションを指すオフセットをまとめて変換します。
        int entryIndex = util::LoadLittleEndian( &sectionInfo._entryIndex );
        int entryCount = util::LoadLittleEndian( &sectionInfo._entryCount );
        for ( int endEntry = entryIndex + entryCount; entryIndex < endEntry; ++entryIndex )
        {
            const Entry& entry = pEntries[ entryIndex ];
            ptrdiff_t posOffset = static_cast< ptrdiff_t >( util::LoadLittleEndian( &entry._position ) );
            int structCount = util::LoadLittleEndian( &entry._structCount );
            int offsetCount = util::LoadLittleEndian( &entry._offsetCount );
            int paddingCount = util::LoadLittleEndian( &entry._paddingCount );

            // 構造体の配列をまとめて変換します。
            BinPtr* pBinPtr = util::BytePtr( pFile, posOffset ).Get< BinPtr >();
            for ( int idxStruct = 0; idxStruct < structCount; ++idxStruct, pBinPtr += paddingCount )
            {
                for ( int idxOffset = 0; idxOffset < offsetCount; ++idxOffset, ++pBinPtr )
                {
                    pBinPtr->Unrelocate( pBase );
                }
            }
        }
    }

    pFile->SetRelocated( false );
}

RelocationTable::Section* RelocationTable::GetSection( int sectionIndex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( sectionIndex, 0, this->_sectionCount );
    return &this->_sections[ sectionIndex ];
}

const RelocationTable::Section* RelocationTable::GetSection( int sectionIndex ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE( sectionIndex, 0, this->_sectionCount );
    return &this->_sections[ sectionIndex ];
}

void RelocationTable::SetSignature() NN_NOEXCEPT
{
    this->_signature.SetPacked( RelocationTable::PackedSignature );
}

size_t RelocationTable::CalculateSize( int sectionCount, int entryCount ) NN_NOEXCEPT
{
    size_t size = sizeof( RelocationTable );
    size += sizeof( Section ) * ( sectionCount - 1 ); // RelocationTable に 1 個分含まれています。
    size += sizeof( Entry ) * entryCount;
    return size;
}

//--------------------------------------------------------------------------------------------------

NN_DEFINE_STATIC_CONSTANT( const int StringPool::PackedSignature );

int StringPool::GetStringCount() const NN_NOEXCEPT
{
    return static_cast< int >( util::LoadLittleEndian( &this->_stringCount ) );
}

void StringPool::SetStringCount( int count ) NN_NOEXCEPT
{
    util::StoreLittleEndian( &this->_stringCount, static_cast< int32_t >( count ) );
}

void StringPool::SetSignature() NN_NOEXCEPT
{
    this->_header.signature.SetPacked( StringPool::PackedSignature );
}

}} // namespace nn::util
