﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstddef>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/image/image_ExifExtractor.h>

#include "detail/image_JpegConfig.h"
#include "detail/image_ExifParser.h"

namespace nn { namespace image {

namespace
{
//! @brief 解析対象のタグの格納順序 (0th TIFF タグ用) (ID の昇順)
enum TagIndex0thTiff
{
    TagIndex0thTiff_Make = 0,
    TagIndex0thTiff_Model,
    TagIndex0thTiff_Orientation,
    TagIndex0thTiff_Software,
    TagIndex0thTiff_DateTime,
    TagIndex0thTiff_ExifIfd,
    TagIndex0thTiff_GpsIfd,
    TagIndex0thTiff_NumElements, // Number of elements.
};
//! @brief 解析対象のタグの格納順序 (0th Exif タグ用) (ID の昇順)
enum TagIndex0thExif
{
    TagIndex0thExif_MakerNote = 0,
    TagIndex0thExif_PxXDimension,
    TagIndex0thExif_PxYDimension,
    TagIndex0thExif_UniqueId,
    TagIndex0thExif_NumElements, // Number of elements.
};
//! @brief 解析対象のタグの格納順序 (1st TIFF タグ用) (ID の昇順)
enum TagIndex1stTiff
{
    TagIndex1stTiff_Compression = 0,
    TagIndex1stTiff_JpegFormat,
    TagIndex1stTiff_JpegFormatLength,
    TagIndex1stTiff_NumElements, // Number of elements.
};

//! @brief 解析用コンテキスト (ワークメモリを割り当てる)
typedef struct
{
    detail::ExifBinary exif; //!< 解析対象のバイナリ
    struct
    {
        detail::IfdHeader tiff; //!< 0th TIFF IFD のヘッダ
        detail::IfdTag tiffTags[TagIndex0thTiff_NumElements]; //!< 0th TIFF IFD が対象のタグ
        detail::IfdHeader exif; //!< 0th Exif IFD のヘッダ
        detail::IfdTag exifTags[TagIndex0thExif_NumElements]; //!< 0th Exif IFD が対象のタグ
    } ifd0;
    struct
    {
        detail::IfdHeader tiff; //!< 1st TIFF IFD のヘッダ
        detail::IfdTag tiffTags[TagIndex1stTiff_NumElements]; //!< 1st TIFF IFD が対象のタグ
    } ifd1;
} ExtractContext;

//! @brief ワークメモリの領域から内部コンテキストを取得する
NN_FORCEINLINE ExtractContext* GetExtractContext(void *workBuf) NN_NOEXCEPT
{
    return reinterpret_cast<ExtractContext*>(workBuf);
}

/**
    @brief 検索クエリ用タグの初期化
 */
ExtractContext* CleanupTags(
    ExtractContext *pContext,
    const uint8_t *exifData,
    const unsigned short kExifSize) NN_NOEXCEPT
{
    pContext->exif.data = exifData;
    pContext->exif.size = kExifSize;

    pContext->ifd0.tiffTags[TagIndex0thTiff_Make] =             detail::IfdTagMake;
    pContext->ifd0.tiffTags[TagIndex0thTiff_Model] =            detail::IfdTagModel;
    pContext->ifd0.tiffTags[TagIndex0thTiff_Orientation] =      detail::IfdTagOrientaion;
    pContext->ifd0.tiffTags[TagIndex0thTiff_Software] =         detail::IfdTagSoftware;
    pContext->ifd0.tiffTags[TagIndex0thTiff_DateTime] =         detail::IfdTagDateTime;
    pContext->ifd0.tiffTags[TagIndex0thTiff_ExifIfd] =          detail::IfdTagExifIfd;
    pContext->ifd0.tiffTags[TagIndex0thTiff_GpsIfd] =           detail::IfdTagGpsIfd;

    pContext->ifd0.exifTags[TagIndex0thExif_MakerNote] =        detail::IfdTagMakerNote;
    pContext->ifd0.exifTags[TagIndex0thExif_PxXDimension] =     detail::IfdTagPxXDimension;
    pContext->ifd0.exifTags[TagIndex0thExif_PxYDimension] =     detail::IfdTagPxYDimension;
    pContext->ifd0.exifTags[TagIndex0thExif_UniqueId] =         detail::IfdTagUniqueId;

    pContext->ifd1.tiffTags[TagIndex1stTiff_Compression] =      detail::IfdTagCompression;
    pContext->ifd1.tiffTags[TagIndex1stTiff_JpegFormat] =       detail::IfdTagJpegFormat;
    pContext->ifd1.tiffTags[TagIndex1stTiff_JpegFormatLength] = detail::IfdTagJpegFormatLen;
    return pContext;
}

/**
    @brief ASCII で構成され '\0' で終端された文字列を検査しながら取得する。
    @details
    検査内容は下記:
    - 各文字が ASCII (0x00-0x7F) に収まるかどうか
    - 文字列の終端が '\0' かどうか
    文字列がこれに反する場合、あるいは長さが 0 の場合、存在しないとして nullptr を返す。
 */
const char *ExtractAscii(
    size_t *pSize,
    const detail::IfdTag &tag,
    const detail::ExifBinary &exif) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        tag.type == detail::IfdTagType_Ascii,
        "Given tag is not typed as ASCII.\n");

    if (tag.valueCount <= 0)
    {
        return nullptr;
    }

    const uint8_t *rval = (exif.data) + tag.valueOffset;
    for (uint16_t i = 0; i < tag.valueCount - 1; i++)
    {
        if ((rval[i] & 0x80) != 0)
        {
            // 非ASCII
            NN_DETAIL_IMAGE_JPEG_LOG_WARN(
                "Tag[%04x]: Non-ASCII code found at the string[%d]: %02x\n",
                tag.id, i, rval[i]);
            return nullptr;
        }
    }
    if (rval[tag.valueCount - 1] != '\0')
    {
        // 終端文字列の違反
        NN_DETAIL_IMAGE_JPEG_LOG_WARN(
            "Tag[%04x]: String is not terminated at %d: %02x\n",
            tag.id, tag.valueCount - 1, rval[tag.valueCount - 1]);
        return nullptr;
    }

    *pSize = tag.valueCount;
    return reinterpret_cast<const char*>(rval);
}

/**
    @brief 種類が ShorOrLong のタグの値を、Short として取得する。
  */
bool ExtractShorOrLongAsShort(
    uint16_t *pValue,
    const detail::IfdTag &tag,
    const ExtractContext &context) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        tag.type == detail::IfdTagType_Short || tag.type == detail::IfdTagType_Long,
        "Unexpected tag type: %d\n", tag.type);
    if (tag.type == detail::IfdTagType_Short)
    {
        *pValue = context.exif.io.read2(context.exif.data + tag.valueOffset);
    }
    else
    {
        uint32_t value32 = context.exif.io.read4(context.exif.data + tag.valueOffset);
        if (value32 > 0xFFFF)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Too large value: %lu\n", value32);
            return false;
        }
        *pValue = static_cast<uint16_t>(value32);
    }
    return true;
}
}

size_t ExifExtractor::GetWorkBufferSize() NN_NOEXCEPT
{
    return sizeof(ExtractContext);
}

ExifExtractor::ExifExtractor(void *workBuf, const size_t kWorkBufSize) NN_NOEXCEPT :
    m_Stage(Stage_Unregistered),
    m_WorkBuf(workBuf)
{
    NN_STATIC_ASSERT(NN_ALIGNOF(ExtractContext) <= NN_ALIGNOF(std::max_align_t));
    NN_SDK_REQUIRES(
        workBuf != nullptr && kWorkBufSize >= this->GetWorkBufferSize() &&
        reinterpret_cast<uintptr_t>(workBuf) % NN_ALIGNOF(std::max_align_t) == 0,
        "Valid work buffer is required.\n");
    NN_UNUSED(kWorkBufSize);
}

ExifExtractor::~ExifExtractor() NN_NOEXCEPT {}

void ExifExtractor::SetExifData(const void *exifData, const size_t kExifSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        exifData != nullptr &&
        kExifSize > 0 && kExifSize <= detail::ExifProperty_SizeMax,
        "Valid Exif data is required.\n");

    m_ExifData = exifData;
    m_ExifSize = static_cast<uint16_t>(kExifSize);
    m_Stage = Stage_Registered;
}

JpegStatus ExifExtractor::Analyze() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Registered, "Exif data is not registered.\n");

    ExtractContext *pContext = CleanupTags(GetExtractContext(m_WorkBuf), static_cast<const uint8_t*>(m_ExifData), m_ExifSize);

    // TIFF ヘッダを読み込み、エンディアンと 0th TIFF IFD のオフセットを取得
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Parsing TIFF header\n");
    uint16_t offset0thIfd = 0;
    if (!detail::ReadTiffHeader(
        &pContext->exif.io,
        &offset0thIfd,
        pContext->exif.data,
        pContext->exif.size))
    {
        return JpegStatus_WrongFormat;
    }

    // 0th TIFF IFD のヘッダを解析
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Extracting 0th TIFF IFD @0x%04x\n", offset0thIfd);
    if (!detail::ReadIfdHeader(&pContext->ifd0.tiff, offset0thIfd, pContext->exif))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 0th TIFF IFD header.\n");
        return JpegStatus_WrongFormat;
    }

    // 0th TIFF IFD のタグを調べる
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Searching tags in 0th TIFF IFD.\n");
    if (!detail::SearchIfdTags(
        pContext->ifd0.tiffTags, TagIndex0thTiff_NumElements,
        pContext->ifd0.tiff,
        pContext->exif))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 0th TIFF IFD.\n");
        return JpegStatus_WrongFormat;
    }

    if (pContext->ifd0.tiffTags[TagIndex0thTiff_ExifIfd].valueOffset != detail::IfdTagState_NotSpecified)
    {
        // 0th Exif IFD が存在すれば...

        // IFD のオフセットを取得
        uint32_t exifOffset32 = pContext->exif.io.read4(
            pContext->exif.data + pContext->ifd0.tiffTags[TagIndex0thTiff_ExifIfd].valueOffset);
        if (!(exifOffset32 < pContext->exif.size))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid offset of 0th EXIF IFD: %08x.\n", exifOffset32);
            return JpegStatus_WrongFormat;
        }
        uint16_t exifOffset = static_cast<uint16_t>(exifOffset32);

        // 0th Exif IFD のヘッダを解析
        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Extracting 0th EXIF IFD @0x%04x\n", exifOffset);
        if (!detail::ReadIfdHeader(&pContext->ifd0.exif, exifOffset, pContext->exif))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 0th EXIF IFD header.\n");
            return JpegStatus_WrongFormat;
        }

        // 0th Exif IFD のタグが存在するか、存在すれば値のオフセットを調べる
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Searching tags in 0th EXIF IFD.\n");
        if (!detail::SearchIfdTags(
            pContext->ifd0.exifTags, TagIndex0thExif_NumElements,
            pContext->ifd0.exif,
            pContext->exif))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 0th EXIF IFD.\n");
            return JpegStatus_WrongFormat;
        }
    }
    else
    {
        // もし 0th Exif IFD が存在しなければ、すべてのタグを not found とする。
        for (uint32_t i = 0; i < TagIndex0thExif_NumElements; i++)
        {
            pContext->ifd0.exifTags[i].valueCount = 0;
            pContext->ifd0.exifTags[i].valueOffset = detail::IfdTagState_NotSpecified;
        }
    }

    if (pContext->ifd0.tiff.nextOffset != detail::IfdTagState_NotSpecified)
    {
        // 1st TIFF IFD が存在すれば...

        // 1st TIFF IFD のヘッダを解析
        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Extracting 1st TIFF IFD @0x%04x\n", pContext->ifd0.tiff.nextOffset);
        if (!detail::ReadIfdHeader(&pContext->ifd1.tiff, pContext->ifd0.tiff.nextOffset, pContext->exif))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 1st TIFF IFD header.\n");
            return JpegStatus_WrongFormat;
        }

        // 1st TIFF IFD のタグが存在するか、存在すれば値のオフセットを調べる
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Searching tags in 1st TIFF IFD.\n");
        if (!detail::SearchIfdTags(
            pContext->ifd1.tiffTags, TagIndex1stTiff_NumElements,
            pContext->ifd1.tiff,
            pContext->exif))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid format of 1st TIFF IFD.\n");
            return JpegStatus_WrongFormat;
        }
    }
    else
    {
        // もし 1st TIFF IFD が存在しなければ、すべてのタグを not found とする。
        for (uint32_t i = 0; i < TagIndex1stTiff_NumElements; i++)
        {
            pContext->ifd1.tiffTags[i].valueCount = 0;
            pContext->ifd1.tiffTags[i].valueOffset = detail::IfdTagState_NotSpecified;
        }
    }

    m_Stage = Stage_Analyzed;
    return JpegStatus_Ok;
}

/**
    @name 0th TIFF IFD のタグの値を取得する。
    @{
 */
const char* ExifExtractor::ExtractMaker(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.tiffTags[TagIndex0thTiff_Make];
    if (!(tag.valueCount > 0))
    {
        return nullptr;
    }
    return ExtractAscii(pSize, tag, pContext->exif);
}
const char* ExifExtractor::ExtractModel(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.tiffTags[TagIndex0thTiff_Model];
    if (!(tag.valueCount > 0))
    {
        return nullptr;
    }
    return ExtractAscii(pSize, tag, pContext->exif);
}
bool ExifExtractor::ExtractOrientation(ExifOrientation *pValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.tiffTags[TagIndex0thTiff_Orientation];
    if (tag.valueCount != 1)
    {
        return false;
    }
#if 0 // 真面目な実装だが、ここまでする必要はない...。
    switch (pContext->exif.io.read2(pContext->exif.data + tag.valueOffset))
    {
    case 1:
        *pValue = ExifOrientation_Normal;
        break;
    case 2:
        *pValue = ExifOrientation_FlipHorizontal;
        break;
    case 3:
        *pValue = ExifOrientation_Rotate180;
        break;
    case 4:
        *pValue = ExifOrientation_FlipVertical;
        break;
    case 5:
        *pValue = ExifOrientation_FlipTopRightToLeftBottom;
        break;
    case 6:
        *pValue = ExifOrientation_Rotate270;
        break;
    case 7:
        *pValue = ExifOrientation_FlipTopLeftToRightBottom;
        break;
    case 8:
        *pValue = ExifOrientation_Rotate90;
        break;
    default: // 見つからなかった、とする。
        return false;
    }
#else
    uint16_t raw = pContext->exif.io.read2(pContext->exif.data + tag.valueOffset);
    if (!(raw >= 1 && raw <= 8))
    {
        // ExifOrientation の値域の外なら、見つからなかった扱い
        return false;
    }
    *pValue = static_cast<ExifOrientation>(raw);
#endif
    return true;
}
const char* ExifExtractor::ExtractSoftware(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.tiffTags[TagIndex0thTiff_Software];
    if (!(tag.valueCount > 0))
    {
        return nullptr;
    }
    return ExtractAscii(pSize, tag, pContext->exif);
}
const char* ExifExtractor::ExtractDateTime(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.tiffTags[TagIndex0thTiff_DateTime];
    if (tag.valueCount != 20)
    {
        return nullptr;
    }
    return ExtractAscii(pSize, tag, pContext->exif);
}
/**
    @}
 */

/**
    @name 0th Exif IFD のタグの値を取得する。
    @{
 */
const void* ExifExtractor::ExtractMakerNote(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);;
    detail::IfdTag &tag = pContext->ifd0.exifTags[TagIndex0thExif_MakerNote];
    if (!(tag.valueCount > 0))
    {
        return nullptr;
    }
    *pSize = tag.valueCount;
    return (pContext->exif.data + tag.valueOffset);
}
bool ExifExtractor::ExtractEffectiveDimension(::nn::image::Dimension *pDimension) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);;
    detail::IfdTag &tagX = pContext->ifd0.exifTags[TagIndex0thExif_PxXDimension];
    detail::IfdTag &tagY = pContext->ifd0.exifTags[TagIndex0thExif_PxYDimension];

    uint16_t width = 0;
    uint16_t height = 0;
    if (!(tagX.valueCount == 1 && tagY.valueCount == 1 &&
        ExtractShorOrLongAsShort(&width, tagX, *pContext) &&
        ExtractShorOrLongAsShort(&height, tagY, *pContext)))
    {
        // - 有効画像幅と有効画像高さは、それぞれ 1 つないといけない。
        // 0xFFFF を超える値は JPEG の仕様上取り得ないので、存在しない扱いとする。
        return false;
    }
    pDimension->width = width;
    pDimension->height = height;

    return true;
}
const char* ExifExtractor::ExtractUniqueId(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tag = pContext->ifd0.exifTags[TagIndex0thExif_UniqueId];
    if (tag.valueCount != 33)
    {
        return nullptr;
    }
    return ExtractAscii(pSize, tag, pContext->exif);
}
/**
    @}
 */

/**
    @name 1st TIFF IFD のタグの値を取得する。
    @{
 */
const void* ExifExtractor::ExtractThumbnail(size_t *pSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Exif binary is not analyzed.\n");

    ExtractContext *pContext = GetExtractContext(m_WorkBuf);
    detail::IfdTag &tagCompression = pContext->ifd1.tiffTags[TagIndex1stTiff_Compression];
    detail::IfdTag &tagOffset = pContext->ifd1.tiffTags[TagIndex1stTiff_JpegFormat];
    detail::IfdTag &tagLength = pContext->ifd1.tiffTags[TagIndex1stTiff_JpegFormatLength];

    if (!(tagCompression.valueCount == 1 && tagOffset.valueCount == 1 && tagLength.valueCount == 1))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_WARN(
            "Invalid value count: %u %u %u\n",
            tagCompression.valueCount,
            tagOffset.valueCount,
            tagLength.valueCount);
        return nullptr;
    }

    detail::ExifBinary &exif = pContext->exif;
    uint16_t compression = exif.io.read2(exif.data + tagCompression.valueOffset);
    if (compression != 6)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_WARN("Invalid compression type: %u\n", compression);
        return nullptr;
    }

    uint32_t offset = exif.io.read4(exif.data + tagOffset.valueOffset);
    uint32_t length = exif.io.read4(exif.data + tagLength.valueOffset);
    if (!(offset <= exif.size && length <= exif.size && offset + length <= exif.size))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_WARN("Invalid size: %u+%u<=%u\n", offset, length, exif.size);
        return nullptr;
    }

    *pSize = static_cast<uint16_t>(length);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Extracted thumbnail size: %u\n", *pSize);
    return (pContext->exif.data + offset);
}
/**
    @}
 */

}}
