﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstddef>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include "image_JpegConfig.h"
#include "image_JpegParser.h"
#include "image_LibjpegHelper.h"

// libjpeg の構造体定義を使う
#define JPEG_INTERNALS          // NOLINT(name/macro)
#include "include/jinclude.h"
#include "include/jdct.h"
#include "include/jmemsys.h"

#include "include/jccolor_e.h"
#include "include/jcdctmgr_e.h"
#include "include/jchuff_e.h"
#include "include/jcmarker_e.h"
#include "include/jcmaster_e.h"
#include "include/jcprepct_e.h"
#include "include/jcsample_e.h"
#include "include/jdatadst_e.h"
#include "include/jdcoefct_e.h"
#include "include/jdcolor_e.h"
#include "include/jddctmgr_e.h"
#include "include/jdhuff_e.h"
#include "include/jdinput_e.h"
#include "include/jdmainct_e.h"
#include "include/jdmarker_e.h"
#include "include/jdmaster_e.h"
#include "include/jdmerge_e.h"
#include "include/jdpostct_e.h"
#include "include/jdsample_e.h"
#include "include/jmemmgr_e.h"
#undef JPEG_INTERNALS

namespace nn { namespace image { namespace detail {

namespace
{
// ワークバッファのアラインメント要件が std::max_align_t 以下であるかをチェック
NN_STATIC_ASSERT(
    NN_ALIGNOF(detail::jpeg::my_memory_mgr) <= NN_ALIGNOF(std::max_align_t) &&
    NN_ALIGNOF(ALIGN_TYPE) <= NN_ALIGNOF(std::max_align_t));

// メモリ量計算用ユーティリティ
NN_FORCEINLINE size_t AlignWithType(size_t size) NN_NOEXCEPT
{
    return RoundUp(size, SIZEOF(ALIGN_TYPE));
}
NN_FORCEINLINE size_t AlignWithLargePoolHdr(size_t size) NN_NOEXCEPT
{
    return AlignWithType(size) + SIZEOF(jpeg::large_pool_hdr);
}
NN_FORCEINLINE size_t AlignWithSmallPoolHdr(size_t size) NN_NOEXCEPT
{
    return AlignWithType(size) + SIZEOF(jpeg::small_pool_hdr);
}

#define NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(a_total, a_rest, a_size) \
    do \
    { \
        if ((a_rest) < AlignWithType((a_size))) \
        { \
            (a_total) += AlignWithSmallPoolHdr((a_size)) + LibjpegProperty_ImagePoolAddonSize; \
            (a_rest) = LibjpegProperty_ImagePoolAddonSize; \
        } \
        else \
        { \
            (a_rest) -= AlignWithType((a_size)); \
        } \
    } while (NN_STATIC_CONDITION(false))
#define NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(a_total, a_rest, a_size) \
    do \
    { \
        if ((a_rest) < AlignWithType((a_size))) \
        { \
            (a_total) += AlignWithSmallPoolHdr((a_size)); \
        } \
        else \
        { \
            (a_rest) -= AlignWithType((a_size)); \
        } \
    } while (NN_STATIC_CONDITION(false))

//! @brief デコード用、小さい管理領域のサイズ計算
size_t GetDecodingSmallPermanentSize(const JpegInfoPtr pinfo) NN_NOEXCEPT
{
    // jinit_marker_reader()
    size_t restSize = LibjpegProperty_PermanentPoolSize;
    size_t size = restSize + AlignWithSmallPoolHdr(SIZEOF(jpeg::my_marker_reader));

    // jinit_input_controller()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(size, restSize, SIZEOF(jpeg::my_input_controller));
    // jpeg_mem_src()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(size, restSize, SIZEOF(jpeg::jpeg_source_mgr));
    // jpeg_alloc_quant_table()
    for (uint32_t i = 0; i < pinfo->dqtCount; i++)
    {
        NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(size, restSize, SIZEOF(jpeg::JQUANT_TBL));
    }
    // jpeg_alloc_huff_table()
    for (uint32_t i = 0; i < pinfo->dhtCount; i++)
    {
        NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(size, restSize, SIZEOF(jpeg::JHUFF_TBL));
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Small pool for permanent: %lu)\n", size);
    return size;
}

//! @brief デコード用、画像データ用小さい領域のサイズ計算
size_t GetDecodingSmallImageSize(const JpegInfoPtr pinfo) NN_NOEXCEPT
{
    // get_sof()
    size_t restSize = LibjpegProperty_ImagePoolSize;
    size_t size = restSize + AlignWithSmallPoolHdr(pinfo->sof.nf * SIZEOF(jpeg::jpeg_component_info));

    // jinit_master_decompress()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_decomp_master));
    // prepare_range_limit_table()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, (5 * (MAXJSAMPLE + 1) + CENTERJSAMPLE) * SIZEOF(jpeg::JSAMPLE));

#ifdef UPSAMPLE_MERGING_SUPPORTED
    if (pinfo->isEnabledMergeUpSample == FALSE)
    {
        // jinit_color_deconverter()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_color_deconverter));
    }
#endif

    if (pinfo->outColorSpace == jpeg::JCS_RGB || pinfo->outColorSpace == jpeg::JCS_CMYK)
    {
        // build_ycc_rgb_table()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, (MAXJSAMPLE + 1) * SIZEOF(int));
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, (MAXJSAMPLE + 1) * SIZEOF(int));
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, (MAXJSAMPLE + 1) * SIZEOF(jpeg::INT32));
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, (MAXJSAMPLE + 1) * SIZEOF(jpeg::INT32));
    }

#ifdef UPSAMPLE_MERGING_SUPPORTED
    if (pinfo->isEnabledMergeUpSample == FALSE)
    {
        // jinit_upsampler()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_sample_upsampler));
    }
    else
    {
        // jinit_merged_upsampler()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_merge_upsampler));
    }
#else
    // jinit_merged_upsampler()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_merge_upsampler));
#endif

    // jinit_d_post_controller()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_post_controller));
    // jinit_inverse_dct()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_idct_controller));
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        // jinit_inverse_dct()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::multiplier_table));
    }
    // jinit_huff_decoder()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::huff_entropy_decoder));
    if (IsProgressive(pinfo))
    {
        // jinit_huff_decoder()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, pinfo->sof.nf * DCTSIZE2 * SIZEOF(int));
    }
    // jinit_d_coef_controller()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_coef_controller));
    if (IsProgressive(pinfo))
    {
        for (uint8_t i = 0; i < pinfo->sof.nf; i++)
        {
            // request_virt_barray();
            NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::jvirt_barray_control));
        }
    }
    // jinit_d_main_controller()
    NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::my_main_controller));

    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        // jinit_d_main_controller()
        JpegComponentInfo *pCompInfo = &pinfo->sof.componentInfo[i];
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(
            size,
            restSize,
            pCompInfo->vn * pCompInfo->dctVScaledSize * SIZEOF(jpeg::JSAMPROW));
    }
    if (IsProgressive(pinfo))
    {
        for (uint8_t i = 0; i < pinfo->sof.nf; i++)
        {
            // realize_virt_arrays()
            JpegComponentInfo *pCompInfo = &pinfo->sof.componentInfo[i];
            NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(
                size,
                restSize,
                jpeg::jround_up(pCompInfo->heightInBlocks, pCompInfo->vn) * SIZEOF(jpeg::JBLOCKROW));
        }
    }

    for (uint8_t i = 0; i < pinfo->firstNs; i++)
    {
        // latch_quant_tables()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::JQUANT_TBL));
    }
    for (uint32_t i = 0; i < pinfo->derivedTableNum; i++)
    {
        // jpeg_make_d_derived_tbl()
        NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, SIZEOF(jpeg::d_derived_tbl));
    }

    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        JpegComponentInfo *pCompInfo = &pinfo->sof.componentInfo[i];

        // jinit_upsampler->alloc_sarray
        uint32_t hIn = (pCompInfo->hn * pCompInfo->dctHScaledSize) / pinfo->minDctHScaledSize;
        uint32_t vIn = (pCompInfo->vn * pCompInfo->dctVScaledSize) / pinfo->minDctVScaledSize;
        if (!(pinfo->isEnabledMergeUpSample || (hIn == pinfo->sof.maxHn && vIn == pinfo->sof.maxVn)))
        {
            // jinit_d_post_controller()
            NN_DETAIL_IMAGE_COUNT_SMALL_IMAGES(size, restSize, pinfo->sof.maxVn * SIZEOF(jpeg::JSAMPROW));
        }
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Small pool for image: %lu)\n", size);
    return size;
}

//! @brief デコード用, 画像データ+管理用の大きい領域サイズ計算
size_t GetDecodingFlexLargeSize(const JpegInfoPtr pinfo) NN_NOEXCEPT
{
    size_t size = 0;
    size_t buf  = 0;

    // sampling data
    JpegComponentInfo *pCompInfo;
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        // jinit_d_main_controller->alloc_sarray
        pCompInfo = &pinfo->sof.componentInfo[i];
        buf = pCompInfo->widthInBlocks * pCompInfo->dctHScaledSize *
                pCompInfo->vn * pCompInfo->dctVScaledSize *
                sizeof(jpeg::JSAMPLE);
        size += AlignWithLargePoolHdr(buf);

        // jinit_upsampler->alloc_sarray
        uint32_t hIn = (pCompInfo->hn * pCompInfo->dctHScaledSize) / pinfo->minDctHScaledSize;
        uint32_t vIn = (pCompInfo->vn * pCompInfo->dctVScaledSize) / pinfo->minDctVScaledSize;
        if (pinfo->isEnabledMergeUpSample || (hIn == pinfo->sof.maxHn && vIn == pinfo->sof.maxVn))
        {
            continue;
        }
        buf = jpeg::jround_up(
            pinfo->outputWidth,
            pinfo->sof.maxHn) * pinfo->sof.maxVn * SIZEOF(jpeg::JSAMPLE);
        size += AlignWithLargePoolHdr(buf);
    }

    if (IsProgressive(pinfo))
    {
        for (uint8_t i = 0; i < pinfo->sof.nf; i++)
        {
            pCompInfo = &pinfo->sof.componentInfo[i];

            // alloc_barray()
            buf = jpeg::jround_up(pCompInfo->widthInBlocks, pCompInfo->hn) *
                    jpeg::jround_up(pCompInfo->heightInBlocks, pCompInfo->vn) *
                    SIZEOF(jpeg::JBLOCK);
            size += AlignWithLargePoolHdr(buf);
        }
    }
    else
    {
        if (!pinfo->hasMultipleScans)
        {
            // jinit_d_coef_controller()
            buf = D_MAX_BLOCKS_IN_MCU * SIZEOF(jpeg::JBLOCK);
            size += AlignWithLargePoolHdr(buf);
        }
    }

    // jinit_merged_upsampler()
    if (pinfo->sof.maxVn == 2 && pinfo->isEnabledMergeUpSample)
    {
        buf = pinfo->outputWidth * pinfo->outColorComponents * SIZEOF(jpeg::JSAMPLE);
        size += AlignWithLargePoolHdr(buf);
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Large pool: %lu)\n", size);
    return size;
}

//! @brief エンコード用, 画像データ+管理用の小さい領域サイズ計算
size_t GetEncodingSmallSize(JpegInfoPtr pinfo) NN_NOEXCEPT
{
    /* alloc_small(POOLID=Permanent):
        -> 2544 = 1656(1st POOL) + 296 * 3((jpeg_alloc_huff_table + 16) * 3)
            (1st POOL = 1600 + 40(40 aligned by 8) + 16)
     */
    size_t restPermanent = LibjpegProperty_PermanentPoolSize;
    // jpeg_mem_dest()
    size_t sizePermanent = restPermanent + AlignWithSmallPoolHdr(SIZEOF(jpeg::my_mem_destination_mgr));
    // jpeg_set_defaults()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, MAX_COMPONENTS * SIZEOF(jpeg::jpeg_component_info));
    // jpeg_alloc_quant_table() x2
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JQUANT_TBL));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JQUANT_TBL));
    // jpeg_alloc_huff_table() x4
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JHUFF_TBL));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JHUFF_TBL));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JHUFF_TBL));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizePermanent, restPermanent, SIZEOF(jpeg::JHUFF_TBL));

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Small pool for permanent: %lu)\n", sizePermanent);

    /* alloc_small(POOLID=Image):
        -> 16056 = 1st POOL = 16000 + 40(36 aligned by 8) + 16
     */
    size_t restImage = LibjpegProperty_ImagePoolSize;
    // jinit_c_master_control()
    size_t sizeImage = restImage + AlignWithSmallPoolHdr(SIZEOF(jpeg::my_comp_master));
    // jinit_color_converter()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_color_converter));
    // jinit_down_sample()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_downsampler));
    // jinit_c_prep_controller() (1, Nf)
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_prep_controller));
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, pinfo->sof.maxVn * SIZEOF(jpeg::JSAMPROW));
    }
    // jinit_forward_dct()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_fdct_controller));
    // jinit_huff_encoder()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::huff_entropy_encoder));
    // jinit_c_coef_controller()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_coef_controller));
    // jinit_c_main_controller() (1, Nf)
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_main_controller));
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(
            sizeImage, restImage,
            pinfo->sof.componentInfo[i].vn * pinfo->sof.componentInfo[i].dctVScaledSize * SIZEOF(jpeg::JSAMPROW));
    }
    // jinit_marker_writer()
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::my_marker_writer));
    // jinit_c_master_control
    //  - prepare_for_pass
    //    - rgb_ycc_start
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, 8 * (MAXJSAMPLE + 1) * SIZEOF(jpeg::INT32));
    //    - start_pass_fdctmgr (DCTSIZE2 * SIZEOF(DCTELEM) x2)
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, DCTSIZE2 * SIZEOF(jpeg::DCTELEM));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, DCTSIZE2 * SIZEOF(jpeg::DCTELEM));
    //    - start_pass_huff
    //      - jpeg_make_c_derived_tbl x4 (ACx2, DCx2)
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::c_derived_tbl));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::c_derived_tbl));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::c_derived_tbl));
    NN_DETAIL_IMAGE_COUNT_SMALL_PERMAS(sizeImage, restImage, SIZEOF(jpeg::c_derived_tbl));

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Small pool for image: %lu)\n", sizeImage);

    return sizePermanent + sizeImage;
}

//! @brief エンコード用, 画像データ+管理用の大きい領域サイズ計算
size_t GetEncodingFlexLargeSize(JpegInfoPtr pinfo) NN_NOEXCEPT
{
    /* Pre alloc_large */
    // jinit_c_coef_controller()
    size_t size = AlignWithLargePoolHdr(C_MAX_BLOCKS_IN_MCU * SIZEOF(jpeg::JBLOCK));

    /* alloc_sarray */
    size_t buf = 0;
    JpegComponentInfo *pCompInfo;
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        pCompInfo = &pinfo->sof.componentInfo[i];

        // jinit_c_prep_controller()
        buf = ((pCompInfo->widthInBlocks * pinfo->minDctHScaledSize * pinfo->sof.maxHn) / pCompInfo->hn)
                * pinfo->sof.maxVn * SIZEOF(jpeg::JSAMPLE);
        size += AlignWithLargePoolHdr(buf);

        // jinit_c_main_controller()
        buf = (pCompInfo->widthInBlocks * pCompInfo->dctHScaledSize)
                * pCompInfo->vn * pCompInfo->dctVScaledSize * SIZEOF(jpeg::JSAMPLE);
        size += AlignWithLargePoolHdr(buf);
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("(Large pool: %lu)\n", size);
    return size;
}
} /* ~ Unnamed namespace */

JpegStatus ExtractExifRegion(
    const void** pOutExifData,
    size_t* pOutExifSize,
    const void *jpegData,
    const size_t kJpegSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        pOutExifData != nullptr &&
        pOutExifSize != nullptr &&
        jpegData != nullptr &&
        kJpegSize > 0,
        "Bad arguments");

    // 最低限の要件
    if (kJpegSize < JpegProperty_SegmentSizeLength)
    {
        return JpegStatus_WrongFormat;
    }
    ::nn::image::JpegStatus status = CheckSoi(
        reinterpret_cast<const uint8_t*>(jpegData),
        kJpegSize);
    if (status != ::nn::image::JpegStatus_Ok)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid JPEG header\n");
        return status;
    }

    // Exif 情報があれば取得
    uint16_t exifSize;
    bool hasExif = GetExifInfo(
        pOutExifData,
        &exifSize,
        reinterpret_cast<const uint8_t*>(jpegData),
        kJpegSize);
    if (!hasExif)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Exif data not found\n");
        return JpegStatus_WrongFormat;
    }
    *pOutExifSize = exifSize;
    return ::nn::image::JpegStatus_Ok;
}

JpegStatus ExtractJpegHeader(
    JpegHeader *pHeader,
    const void *jpegData,
    const size_t kJpegSize,
    const int32_t resolutionDenom) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        pHeader != nullptr &&
        jpegData != nullptr &&
        kJpegSize > 0 &&
        CheckParamDecompResolutionDenom(resolutionDenom),
        "Bad arguments");

    // 最低限の要件
    if (kJpegSize < JpegProperty_SegmentSizeLength)
    {
        return JpegStatus_WrongFormat;
    }
    ::nn::image::JpegStatus status = CheckSoi(
        reinterpret_cast<const uint8_t*>(jpegData),
        kJpegSize);
    if (status != ::nn::image::JpegStatus_Ok)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid JPEG header\n");
        return status;
    }

    // JPEG ヘッダの解析
    JpegInfo info = {};
    status = ReadJpegHeader(
        &info,
        reinterpret_cast<const uint8_t*>(jpegData),
        kJpegSize,
        static_cast<uint32_t>(resolutionDenom));
    if (status != ::nn::image::JpegStatus_Ok)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid JPEG header\n");
        return status;
    }
    if (!(info.outputWidth <= 0xFFFF && info.outputHeight <= 0xFFFF))
    {
        // 画像が大きすぎる。
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Too large picture to be decoded: %lux%lu\n", info.outputWidth, info.outputHeight);
        return nn::image::JpegStatus_WrongFormat;
    }

    // JPEG ヘッダのパラメータを基に、 libjpeg 内部で使われるメモリ量を計算
    size_t size = sizeof(jpeg::my_memory_mgr);
    size += GetDecodingSmallPermanentSize(&info);
    size += GetDecodingSmallImageSize(&info);
    size += GetDecodingFlexLargeSize(&info);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("[Required work buffer: %lu bytes]\n", size);

    // JPEG ヘッダの要約を作成
    pHeader->dimension.width = static_cast<unsigned short>(info.outputWidth);
    pHeader->dimension.height = static_cast<unsigned short>(info.outputHeight);
    pHeader->bufSizeForDecomp = size;
    // Exif 情報があれば取得
    pHeader->hasExif = GetExifInfo(
        &pHeader->exifData,
        &pHeader->exifSize,
        reinterpret_cast<const uint8_t*>(jpegData),
        kJpegSize);

    return ::nn::image::JpegStatus_Ok;
}

::nn::image::JpegStatus GetBufferSizeForJpegCompression(
    size_t *pBufSizeForComp,
    const ::nn::image::Dimension &kDimension,
    const ::nn::image::JpegSamplingRatio kSample) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        pBufSizeForComp != NULL && kDimension.width <= 0xFFFF && kDimension.height <= 0xFFFF,
        "Bad arguments");

    // エンコード後の JPEG のパラメータを設定
    JpegInfo info = {};
    SetJpegHeader(&info, (kDimension.width & 0xFFFF), (kDimension.height & 0xFFFF), kSample);

    // パラメータを基に、 libjpeg 内部で使われるメモリ量を計算
    size_t size = sizeof(jpeg::my_memory_mgr);
    size += GetEncodingSmallSize(&info);
    size += GetEncodingFlexLargeSize(&info);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("[Required work buffer: %lu bytes]\n", size);

    *pBufSizeForComp = size;
    return ::nn::image::JpegStatus_Ok;
}

}}}
