﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Config.h>

/**
    @file
    @brief crypto ライブラリ共通で使用する汎用関数
 */

namespace nn { namespace crypto { namespace detail {

namespace
{
    /*
       @brief 32bit 値のエンディアンをビッグエンディアンに変換します。
     */
    inline Bit32 Convert32BitToBigEndian(Bit32 value) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        return ((((value) >> 24UL) & 0x000000FFUL) |
                (((value) >>  8UL) & 0x0000FF00UL) |
                (((value) <<  8UL) & 0x00FF0000UL) |
                (((value) << 24UL) & 0xFF000000UL));
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        return value;
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif
    }

    /*
       @brief 64bit 値のエンディアンをビッグエンディアンに変換します。
     */
    inline Bit64 Convert64BitToBigEndian(Bit64 value) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        return ((((value) >> 56UL) & 0x00000000000000FFULL) |
                (((value) >> 40UL) & 0x000000000000FF00ULL) |
                (((value) >> 24UL) & 0x0000000000FF0000ULL) |
                (((value) >>  8UL) & 0x00000000FF000000ULL) |
                (((value) <<  8UL) & 0x000000FF00000000ULL) |
                (((value) << 24UL) & 0x0000FF0000000000ULL) |
                (((value) << 40UL) & 0x00FF000000000000ULL) |
                (((value) << 56UL) & 0xFF00000000000000ULL));
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        return value;
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif
    }

    /*
       @brief データ列を 32bit ごとにエンディアンを変換しつつコピーします
     */
    inline void CopyDataWithSwappingEndianBy32Bit(void* pDst, const void* pSrc, size_t size) NN_NOEXCEPT
    {
        Bit8*       pDst8 = static_cast<Bit8*>(pDst);
        const Bit8* pSrc8 = static_cast<const Bit8*>(pSrc);

        for (size_t i = 0; i < (size / sizeof(Bit32)); i++)
        {
            pDst8[0] = pSrc8[3];
            pDst8[1] = pSrc8[2];
            pDst8[2] = pSrc8[1];
            pDst8[3] = pSrc8[0];
            pDst8 += 4;
            pSrc8 += 4;
        }
    }

    /*
       @brief 32bit 巡回右シフトを行います。
     */
    inline Bit32 RotateRight(Bit32 value, int shift) NN_NOEXCEPT
    {
        if (shift != 0)
        {
            return (value << (32 - shift)) | (value >> shift);
        }
        else
        {
            return value;
        }
    }

    /*
       @brief 32bit 巡回左シフトを行います。
     */
    inline Bit32 RotateLeft(Bit32 value, int shift) NN_NOEXCEPT
    {
        if (shift != 0)
        {
            return (value << shift) | (value >> (32 - shift));
        }
        else
        {
            return value;
        }
    }

}   // anonymous namespace

}}}
