﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_CbcMacImpl.h>
#include <nn/crypto/detail/crypto_Clear.h>

namespace nn { namespace crypto { namespace detail {

CbcMacImpl::~CbcMacImpl() NN_NOEXCEPT
{
    ClearMemory(this, sizeof(*this));
}

void CbcMacImpl::ProcessPartialData(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    std::memcpy(m_TemporalBlockBuffer + m_BufferedByte, pData, dataSize);
    m_BufferedByte += dataSize;
}

void CbcMacImpl::ProcessRemainingData(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    /* バッファが埋まっていたら先に処理する */
    if (m_BufferedByte == BlockSize)
    {
        ProcessBlock(m_TemporalBlockBuffer);
        m_BufferedByte = 0;
    }

    std::memcpy(m_TemporalBlockBuffer, pData, dataSize);
    m_BufferedByte = dataSize;
}

void CbcMacImpl::GetMac(void* pMac, size_t macSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((m_State == State_Initialized) || (m_State == State_Done), "Invalid state. Please restart from Initialize().");
    NN_SDK_REQUIRES(macSize >= BlockSize);
    NN_UNUSED(macSize);

    if (m_State == State_Initialized)
    {
        /* 最後のブロックの分を MAC 計算して完了 */
        if (m_BufferedByte == BlockSize)
        {
            ProcessBlock(m_TemporalBlockBuffer);
            m_BufferedByte = 0;
        }

        m_State = State_Done;
    }

    std::memcpy(pMac, m_Mac, BlockSize);
}

/* バッファされているブロックに対して入力データで XOR します */
/* CMAC モードで利用します（CBC-MAC では利用しません） */
void CbcMacImpl::MaskBufferedData(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_BufferedByte == BlockSize);
    NN_SDK_REQUIRES(dataSize == BlockSize);
    NN_UNUSED(dataSize);

    const uint8_t* pData8 = static_cast<const uint8_t*>(pData);

    for (int i = 0; i < static_cast<int>(BlockSize); i++)
    {
        m_TemporalBlockBuffer[i] ^= pData8[i];
    }
}

void CbcMacImpl::ProcessBlock(const void* pData) NN_NOEXCEPT
{
    const uint8_t* pData8 = static_cast<const uint8_t*>(pData);
    uint8_t  tmp[BlockSize];

    /* 現在の MAC (= CBC モードでの IV) と入力の XOR を取り */
    for (int i = 0; i < static_cast<int>(BlockSize); i++)
    {
        tmp[i] = pData8[i] ^ m_Mac[i];
    }

    /* 暗号化した結果が新しい IV (= MAC) になる */
    m_pCipherFunction(m_Mac, tmp, m_pCipherContext);
}

}}} // namespace nn::crypto::detail
