﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2005, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
/*
 * implementations for integer math routines needed for the
 * ECDSA and RSA  algorithm. implemented using the array of  words big
 * integer notation
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_BigNum.h>
#include "crypto_BigNumMath.h"

#define RETURN_FALSE_UNLESS(condition) \
    do {                               \
        if (!(condition))              \
        {                              \
            return false;              \
        }                              \
    } while ( NN_STATIC_CONDITION(0) )

#define RETURN_FALSE_IF_NULL(value) \
    RETURN_FALSE_UNLESS((value) != nullptr)


namespace nn { namespace crypto { namespace detail {

namespace {

/* supports up to 256 bit ECC*/
const int MaxEccDigits = 256 / BigNum::DigitBits;

BigNum::HalfDigit GetLowerHalf16Bits(BigNum::Digit x) NN_NOEXCEPT
{
    return static_cast<BigNum::HalfDigit>(x & 0xffff);
}

BigNum::HalfDigit GetHigherHalf16Bits(BigNum::Digit x) NN_NOEXCEPT
{
    return static_cast<BigNum::HalfDigit>((x >> 16) & 0xffff);
}

BigNum::Digit ToHigherHalf16Bits(BigNum::Digit x) NN_NOEXCEPT
{
    return static_cast<BigNum::Digit>(x) << 16;
}

BigNum::Digit GetMostSignificant2Bits(BigNum::Digit a) NN_NOEXCEPT
{
    return (a >> (BigNum::DigitBits - 2)) & 3;
}

} // anonymous namespace

/*
 * Clear a and a[0] = b
 */
void BigNum::AssignDigit(Digit* a, Digit b, int digits) NN_NOEXCEPT
{
    SetZero(a, digits);
    a[0] = b;
}

/*
 * computes a = b + c
 * return carry
 */
BigNum::Digit BigNum::Add(Digit* a, const Digit* b, const Digit* c, int digits) NN_NOEXCEPT
{
    return nndetailCryptoBignumAddWords(a, b, c, digits);
}

/*
 * computes a = b - c
 * return borrow
 */
BigNum::Digit BigNum::Sub(Digit* a, const Digit* b, const Digit* c, int digits) NN_NOEXCEPT
{
    return nndetailCryptoBignumSubWords(a, b, c, digits);
}

/*
 * computes a += c * b
 * return carry
 */
BigNum::Digit BigNum::AddMult(Digit* a, const Digit* b, int digits, Digit c) NN_NOEXCEPT
{
    return nndetailCryptoBignumMulAddWords(a, b, digits, c);
}

/*
 * computes a = b - c*d for
 * a[] b[] d[] and c is a digit
 * return borrow
 */
BigNum::Digit BigNum::SubMult(Digit* a, const Digit* b,
                              const Digit c, const Digit* d,
                              int digits) NN_NOEXCEPT
{
    Digit borrow, t[2];
    int i;

    if (c == 0)
    {
        return (0);
    }

    borrow = 0;
    for (i = 0; i < digits; i++)
    {
        DigitMult(t, c, d[i]);
        if ((a[i] = b[i] - borrow) > (MaxDigitValue - borrow))
        {
            borrow = 1;
        }
        else
        {
            borrow = 0;
        }
        if ((a[i] -= t[0]) > (MaxDigitValue - t[0]))
        {
            borrow++;
        }
        borrow += t[1];
    }
    return borrow;
}

/*
 * returns how many bits are significant
 */
int BigNum::GetDigitBits(Digit a) NN_NOEXCEPT
{
    int i;
    for (i = 0; i < DigitBits; i++, a >>= 1)
    {
        if (a == 0)
        {
            break;
        }
    }
    return i;
}

/*
 * a = b/ c where all are digits
 * length of b is b[2]
 * b[1] < c and HIGHER_HALF(c) > 0
 */
void BigNum::DigitDiv(Digit* a, const Digit b[2], Digit c) NN_NOEXCEPT
{
    Digit t[2], u, v;
    HalfDigit a_high, a_low, c_high, c_low;

    c_high = GetHigherHalf16Bits(c);
    c_low = GetLowerHalf16Bits(c);

    t[0] = b[0];
    t[1] = b[1];

    /* under-estimate higher half of quotient and subtract */
    if (c_high == MaxHalfDigitValue)
    {
        a_high = GetHigherHalf16Bits(t[1]);
    }
    else
    {
        a_high = static_cast<HalfDigit>((t[1] / (c_high + 1)));
    }

    u = static_cast<Digit>(a_high) * static_cast<Digit>(c_low);
    v = static_cast<Digit>(a_high) * static_cast<Digit>(c_high);

    if ((t[0] -= ToHigherHalf16Bits(static_cast<HalfDigit>(u))) > (MaxDigitValue - ToHigherHalf16Bits(static_cast<HalfDigit>(u))))
    {
        t[1]--;
    }
    t[1] -= GetHigherHalf16Bits(u);
    t[1] -= v;

    /* correct estimate */
    while ((t[1] > c_high) ||
           ((t[1] == c_high) && (t[0] >= ToHigherHalf16Bits(c_low))))
    {
        if ((t[0] -= ToHigherHalf16Bits(c_low)) > MaxDigitValue - ToHigherHalf16Bits(c_low))
        {
            t[1]--;
        }
        t[1] -= c_high;
        a_high++;
    }

    /* underestimate lower half of quotient and subtract */
    if (c_high == MaxHalfDigitValue)
    {
        a_low = GetLowerHalf16Bits(t[1]);
    }
    else
    {
        a_low = static_cast<HalfDigit>((ToHigherHalf16Bits(static_cast<HalfDigit>(t[1])) + GetHigherHalf16Bits(t[0])) / (c_high + 1));
    }

    u = static_cast<Digit>(a_low) * static_cast<Digit>(c_low);
    v = static_cast<Digit>(a_low) * static_cast<Digit>(c_high);

    if ((t[0] -= u) > (MaxDigitValue - u))
    {
        t[1]--;
    }
    if ((t[0] -= ToHigherHalf16Bits(static_cast<HalfDigit>(v))) > (MaxDigitValue - ToHigherHalf16Bits(static_cast<HalfDigit>(v))))
    {
        t[1]--;
    }
    t[1] -= GetHigherHalf16Bits(v);

    /* correct estimate */
    while ((t[1] > 0) || ((t[1] == 0) && t[0] >= c))
    {
        if ((t[0] -= c) > (MaxDigitValue - c))
        {
            t[1]--;
        }
        a_low++;
    }
    *a = ToHigherHalf16Bits(a_high) + a_low;
}

/*
 * copies b to a
 */
void BigNum::Copy(Digit* a, const Digit* b, int digits) NN_NOEXCEPT
{
    int i;
    for (i = 0; i < digits; i++)
    {
        a[i] = b[i];
    }
}

/*
 * zeros
 */
void BigNum::SetZero(Digit* a, int digits) NN_NOEXCEPT
{
    int i;
    for (i = 0; i < digits; i++)
    {
        a[i] = 0;
    }
}

/*
 * return length in digits
 */
int BigNum::GetDigits(const Digit* a, int digits) NN_NOEXCEPT
{
    int i;
    for (i = digits - 1; i >= 0; i--)
    {
        if (a[i])
        {
            break;
        }
    }

    return(i + 1);
}

/* Computes a = b * c.
   Lengths: a[2*digits], b[digits], c[digits].
*/
bool BigNum::Mult(Digit* a, const Digit* b, const Digit* c, int digits,
                  DigitAllocator* pAllocator) NN_NOEXCEPT
{
    int b_digits, c_digits, i;
    Digit* integer_math_res = pAllocator->AllocateDigits(2 * digits);

    RETURN_FALSE_IF_NULL(integer_math_res);

    SetZero(integer_math_res, 2 * digits);

    b_digits = GetDigits(b, digits);
    c_digits = GetDigits(c, digits);

    for (i = 0; i < b_digits; i++)
    {
        integer_math_res[i + c_digits] += AddMult(&integer_math_res[i], c, c_digits, b[i]);
    }

    Copy(a, integer_math_res, 2 * digits);

    pAllocator->FreeDigits(integer_math_res, 2 * digits);

    return true;
}

/*
 * a = b << c (shifts b left c bits)
 *return carry
 */
BigNum::Digit BigNum::LeftShift(Digit* a, const Digit* b, int c,
                                int digits) NN_NOEXCEPT
{
    Digit bi, carry;
    int i, t;

    if (c >= DigitBits)
    {
        return (0);
    }

    t = DigitBits - c;
    carry = 0;
    for (i = 0; i < digits; i++)
    {
        bi = b[i];
        a[i] = (bi << c) | carry;
        carry = c ? (bi >> t) : 0;
    }
    return (carry);
}

/*
 * shifts b right c times and returns in a
 */
BigNum::Digit BigNum::RightShift(Digit* a, const Digit* b, int c,
                                 int digits) NN_NOEXCEPT
{
    Digit bi, carry;
    int i;
    unsigned int t;

    if (c >= DigitBits)
    {
        return (0);
    }

    t = DigitBits - c;
    carry = 0;

    for (i = digits - 1; i >= 0; i--)
    {
        bi = b[i];
        a[i] = (bi >> c) | carry;
        carry = c ? (bi << t) : 0;
    }
    return (carry);
}

/*
 * compare and return sign of a-b
 */
int BigNum::Compare(const Digit* a, const Digit* b, int digits) NN_NOEXCEPT
{
    int i;

    for (i = digits - 1; i >= 0; i--)
    {
        if (a[i] > b[i])
        {
            return (1);
        }
        if (a[i] < b[i])
        {
            return (-1);
        }
    }
    return (0);
}

/*
 * a = c/d and b = c % d
 * a[c_digits], b[d_digits], c[c_digits], d[d_digits],
 * c_digits < 2*MAX_BIGINT_DIGITS
 */
bool BigNum::Div(Digit* a, Digit* b, const Digit* c, int c_digits,
                 const Digit* d, int d_digits, DigitAllocator* pAllocator) NN_NOEXCEPT
{
    Digit ai;
    Digit t;
    int i, dd_digits, shift;
    Digit* integer_math_cc = pAllocator->AllocateDigits(c_digits + 1);
    Digit* integer_math_dd = pAllocator->AllocateDigits(d_digits);

    RETURN_FALSE_IF_NULL(integer_math_cc);
    RETURN_FALSE_IF_NULL(integer_math_dd);

    dd_digits = GetDigits(d, d_digits);
    if (dd_digits == 0)
    {
        return false; /* zero division */
    }

    /* normalize */
    shift = DigitBits - GetDigitBits(d[dd_digits - 1]);
    SetZero(integer_math_cc, dd_digits);
    integer_math_cc[c_digits] = LeftShift(integer_math_cc, c, shift, c_digits);
    LeftShift(integer_math_dd, d, shift, dd_digits);
    t = integer_math_dd[dd_digits - 1];

    SetZero(a, c_digits);

    for (i = c_digits - dd_digits; i >= 0; i--)
    {
        /* underestimate quotient and subtract */
        if (t == MaxDigitValue)
        {
            ai = integer_math_cc[i + dd_digits];
        }
        else
        {
            DigitDiv(&ai, &integer_math_cc[i + dd_digits - 1], t + 1);
        }
        integer_math_cc[i + dd_digits] -= SubMult(&integer_math_cc[i],
                                                  &integer_math_cc[i], ai,
                                                  integer_math_dd, dd_digits);
        /*correct estimate */
        while (integer_math_cc[i + dd_digits] ||
               (Compare(&integer_math_cc[i], integer_math_dd, dd_digits) >= 0))
        {
            ai++;
            integer_math_cc[i + dd_digits] -= Sub(&integer_math_cc[i], &integer_math_cc[i],
                                                  integer_math_dd, dd_digits);
        }
        a[i] = ai;
    }
    SetZero(b, d_digits);
    RightShift(b, integer_math_cc, shift, dd_digits);

    pAllocator->FreeDigits(integer_math_dd, d_digits);
    pAllocator->FreeDigits(integer_math_cc, c_digits + 1);

    return true;
}

/*
 * compute a = b mod c
 * a[c_digits], b[b_digits], c[c_digits]
 * c > 0, b_digits < 2*MAX_BIGINT_DIGITS, c_digits < MAX_BIGINT_DIGITS
 */
bool BigNum::Mod(Digit* a, const Digit* b, int b_digits, const Digit* c,
                 int c_digits, DigitAllocator* pAllocator) NN_NOEXCEPT
{
    Digit* integer_math_mod_t = pAllocator->AllocateDigits(b_digits);
    RETURN_FALSE_IF_NULL(integer_math_mod_t);

    RETURN_FALSE_UNLESS(Div(integer_math_mod_t, a, b, b_digits, c, c_digits, pAllocator));

    pAllocator->FreeDigits(integer_math_mod_t, b_digits);

    return true;
}

/*
 * a = b*c mod d
 */
bool BigNum::ModMult(Digit* a, const Digit* b, const Digit* c,
                     const Digit* d, int digits, DigitAllocator* pAllocator) NN_NOEXCEPT
{
    Digit* integer_math_mod_mult_t = pAllocator->AllocateDigits(2 * digits);
    RETURN_FALSE_IF_NULL(integer_math_mod_mult_t);

    RETURN_FALSE_UNLESS(Mult(integer_math_mod_mult_t, b, c, digits, pAllocator));
    RETURN_FALSE_UNLESS(Mod(a, integer_math_mod_mult_t, 2 * digits, d, digits, pAllocator));

    pAllocator->FreeDigits(integer_math_mod_mult_t, 2 * digits);

    return true;
}

/* a = b^c mod d.
 * Lengths: a[d_digits, b[d_digits], c[c_digits], d[d_digits].
 * Assumes d > 0, c_digits > 0, d_digits < MAX_BIGINT_DIGITS.
 */
bool BigNum::ModExp(Digit* a, const Digit* b, const Digit* c,
                    int c_digits, const Digit* d, int d_digits,
                    DigitAllocator* pAllocator) NN_NOEXCEPT
{
    bool need[4];
    /* zero and power one come for free */
    //need[1] = need[0] = true;
    if (c_digits > 1)
    {
        need[3] = need[2] = true;
    }
    else
    {
        need[3] = need[2] = false;
        /* check input to see if power 3 is needed, then turn on 2. */
        Digit exp = c[0];

        for (size_t i = 0; i < DigitBits / 2; i++)
        {
            /* take last two bits */
            Digit set_bits = exp & 0x00000003;
            /* you need to compute that power */
            need[set_bits] = true;
            /* shift right two bits */
            exp = exp >> 2;
        }
        if (need[3])
        {
            need[2] = true; /* need to compute anyway */
        }
    }
    Digit* integer_math_b_power[3];
    integer_math_b_power[0] = pAllocator->AllocateDigits(d_digits);
    integer_math_b_power[1] = pAllocator->AllocateDigits(d_digits);
    integer_math_b_power[2] = pAllocator->AllocateDigits(d_digits);

    RETURN_FALSE_IF_NULL(integer_math_b_power[0]);
    RETURN_FALSE_IF_NULL(integer_math_b_power[1]);
    RETURN_FALSE_IF_NULL(integer_math_b_power[2]);

    /* store b, b^2 mod d, and b^3 mod d */
    Copy(integer_math_b_power[0], b, d_digits);
    if (need[2])
    {
        RETURN_FALSE_UNLESS(
            ModMult(integer_math_b_power[1], integer_math_b_power[0], b, d, d_digits, pAllocator)
        );
    }
    if (need[3])
    {
        RETURN_FALSE_UNLESS(
            ModMult(integer_math_b_power[2], integer_math_b_power[1], b, d, d_digits, pAllocator)
        );
    }

    Digit* integer_math_tt = pAllocator->AllocateDigits(d_digits);
    RETURN_FALSE_IF_NULL(integer_math_tt);
    AssignDigit(integer_math_tt, 1, d_digits);

    c_digits = GetDigits(c, c_digits);

    for (int i = c_digits - 1; i >= 0; i--)
    {
        Digit ci = c[i];
        int ci_bits = DigitBits;

        /* scan past leading zero bits of most significant digit */
        if (i == (c_digits - 1))
        {
            while (!GetMostSignificant2Bits(ci))
            {
                ci <<= 2;
                ci_bits -= 2;
            }
        }

        for (int j = 0; j < ci_bits; j += 2, ci <<= 2)
        {
            /* compute t = t^4*b^s mod d, where s = two MSBs of ci */
            RETURN_FALSE_UNLESS(ModMult(integer_math_tt, integer_math_tt, integer_math_tt, d, d_digits, pAllocator));
            RETURN_FALSE_UNLESS(ModMult(integer_math_tt, integer_math_tt, integer_math_tt, d, d_digits, pAllocator));
            int s = GetMostSignificant2Bits(ci);
            if (s)
            {
                RETURN_FALSE_UNLESS(
                    ModMult(integer_math_tt, integer_math_tt, integer_math_b_power[s - 1], d,
                            d_digits, pAllocator)
                );
            }
        }
    }
    Copy(a, integer_math_tt, d_digits);

    pAllocator->FreeDigits(integer_math_tt, d_digits);
    pAllocator->FreeDigits(integer_math_b_power[2], d_digits);
    pAllocator->FreeDigits(integer_math_b_power[1], d_digits);
    pAllocator->FreeDigits(integer_math_b_power[0], d_digits);

    return true;
}

/*
 * return nonzero iff a is zero
 */
int BigNum::IsZero(const Digit* a, int digits) NN_NOEXCEPT
{
    int i;

    for (i = 0; i < digits; i++)
    {
        if (a[i])
        {
            return (0);
        }
    }

    return (1);
}

/* compute a = 1/b mod c
 * this is only for ecc, so use the smaller MaxEccDigits
 * gcd(b,c)=1
 */
void BigNum::ModInv(Digit* a, const Digit* b,
                    const Digit* c, int digits) NN_NOEXCEPT
{
    Digit q[MaxEccDigits], t1[MaxEccDigits], t3[MaxEccDigits],
             u1[MaxEccDigits], u3[MaxEccDigits], v1[MaxEccDigits],
             v3[MaxEccDigits], w[2 * MaxEccDigits];
    int u1_sign;
    NN_SDK_ASSERT(digits <= static_cast<int>(MaxEccDigits));

    /* apply extended euclidian alg modified to avoid neg numbers */
    AssignDigit(u1, 1, digits);
    SetZero(v1, digits);
    Copy(u3, b, digits);
    Copy(v3, c, digits);
    u1_sign = 1;

    while (!IsZero(v3, digits))
    {
        Div(q, t3, u3, digits, v3, digits);
        Mult(w, q, v1, digits);
        Add(t1, u1, w, digits);
        Copy(u1, v1, digits);
        Copy(v1, t1, digits);
        Copy(u3, v3, digits);
        Copy(v3, t3, digits);
        u1_sign = -u1_sign;
    }
    /* negate if sign is negative */
    if (u1_sign < 0)
    {
        Sub(a, c, u1, digits);
    }
    else
    {
        Copy(a, u1, digits);
    }
}

/* returns a = b + c mod d, when a <= d, b <= d.
   If it is not the case, returns a number a such that
   a mod d = b + c mod d and a < 2^digits */
void BigNum::AddMod(Digit* a, const Digit* b,
                    const Digit* c, const Digit* d, int digits) NN_NOEXCEPT
{
    int carry, borrow;
    carry = Add(a, b, c, digits);
    if (carry || (Compare(a, d, digits) >= 0))
    {
        borrow = Sub(a, a, d, digits);
        while (carry != borrow)
        {
            borrow = Sub(a, a, d, digits);
        }
    }
}

/* returns a = b - c mod d, when a <= d, b <= d.
   If it is not the case, returns a number a such that
   a mod d = b - c mod d and a < 2^digits. */
void BigNum::SubMod(Digit* a, const Digit* b,
                    const Digit* c, const Digit* d, int digits) NN_NOEXCEPT
{
    NN_SDK_ASSERT( Compare( b, d, digits ) < 0 );
    int carry, borrow;
    borrow = Sub(a, b, c, digits);
    if (borrow || (Compare(a, d, digits) > 0))
    {
        carry = Add(a, a, d, digits);
        while (carry != borrow)
        {
            borrow = Add(a, a, d, digits);
        }
    }
}

/* Computes a = b * c.
 * this is only for ecc, so use the smaller MaxEccDigits
   Lengths: a[2*digits], b[digits], c[digits].
*/
void BigNum::Mult(Digit* a, const Digit* b, const Digit* c, int digits) NN_NOEXCEPT
{
    int b_digits, c_digits, i;
    Digit integer_math_res[2 * MaxEccDigits];

    SetZero(integer_math_res, 2 * digits);

    b_digits = GetDigits(b, digits);
    c_digits = GetDigits(c, digits);

    for (i = 0; i < b_digits; i++)
    {
        integer_math_res[i + c_digits] += AddMult(&integer_math_res[i], c, c_digits, b[i]);
    }

    Copy(a, integer_math_res, 2 * digits);
}

/*
 * a = c/d and b = c % d
 * this is only for ecc, so use the smaller MaxEccDigits
 * a[c_digits], b[d_digits], c[c_digits], d[d_digits],
 * c_digits < 2*MaxEccDigits
 */
void BigNum::Div(Digit* a, Digit* b, const Digit* c, int c_digits,
                 const Digit* d, int d_digits) NN_NOEXCEPT
{
    Digit ai;
    Digit t;
    int i, dd_digits, shift;
    Digit integer_math_cc[2 * MaxEccDigits + 1], integer_math_dd[MaxEccDigits];

    dd_digits = GetDigits(d, d_digits);
    if (dd_digits == 0)
    {
        return;
    }

    /* normalize */
    shift = DigitBits - GetDigitBits(d[dd_digits - 1]);
    SetZero(integer_math_cc, dd_digits);
    integer_math_cc[c_digits] = LeftShift(integer_math_cc, c, shift, c_digits);
    LeftShift(integer_math_dd, d, shift, dd_digits);
    t = integer_math_dd[dd_digits - 1];

    SetZero(a, c_digits);

    for (i = c_digits - dd_digits; i >= 0; i--)
    {
        /* underestimate quotient and subtract */
        if (t == MaxDigitValue)
        {
            ai = integer_math_cc[i + dd_digits];
        }
        else
        {
            DigitDiv(&ai, &integer_math_cc[i + dd_digits - 1], t + 1);
        }
        integer_math_cc[i + dd_digits] -= SubMult(&integer_math_cc[i],
                                                  &integer_math_cc[i], ai,
                                                  integer_math_dd, dd_digits);
        /*correct estimate */
        while (integer_math_cc[i + dd_digits] ||
               (Compare(&integer_math_cc[i], integer_math_dd, dd_digits) >= 0))
        {
            ai++;
            integer_math_cc[i + dd_digits] -= Sub(&integer_math_cc[i], &integer_math_cc[i],
                                                  integer_math_dd, dd_digits);
        }
        a[i] = ai;
    }
    SetZero(b, d_digits);
    RightShift(b, integer_math_cc, shift, dd_digits);
}

/*
 * compute a = b mod c
 * a[c_digits], b[b_digits], c[c_digits]
 * c > 0, b_digits < 2*MaxEccDigits, c_digits < MaxEccDigits
 */
void BigNum::Mod(Digit* a, const Digit* b, int b_digits, const Digit* c,
                 int c_digits) NN_NOEXCEPT
{
    Digit integer_math_mod_t[2 * MaxEccDigits];
    Div(integer_math_mod_t, a, b, b_digits, c, c_digits);
}

void OctetStringFromDigits(void* a, size_t len, const BigNum::Digit* b, size_t nDigits) NN_NOEXCEPT
{
    size_t j = len;

    for (size_t i = 0; i < nDigits && j > 0; i++)
    {
        BigNum::Digit t = b[i];
        for (size_t u = 0; j > 0 && u < sizeof(BigNum::Digit) * 8; u += 8)
        {
            static_cast<uint8_t*>(a)[--j] = static_cast<uint8_t>(t >> u);
        }
    }

    while ( j > 0 )
    {
        static_cast<uint8_t*>(a)[--j] = 0;
    }
}

void DigitsFromOctetString(BigNum::Digit* a, size_t nDigits, const void* b, size_t len) NN_NOEXCEPT
{
    size_t j = len;
    size_t i = 0;
    for (; i < nDigits && j > 0; i++)
    {
        BigNum::Digit t = 0;
        for (size_t u = 0; j > 0 && u < sizeof(BigNum::Digit) * 8; u += 8)
        {
            t |= static_cast<const uint8_t*>(b)[--j] << u;
        }
        a[i] = t;
    }
    for (; i < nDigits; i++)
    {
        a[i] = 0;
    }
}

}}} // namespace nn::crypto::detail
