﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>

namespace nn {
    namespace utilTool {

        //! @brief キーワード引数のドキュメント
        class KeywordArgumentDocument
        {
        public:
            KeywordArgumentDocument(char shortName, std::string longName, bool hasValue, std::string placeHolderName, std::string description, bool isRequired)
                : m_ShortName(std::string(&shortName, 1)), m_LongName(longName), m_HasValue(hasValue), m_PlaceHolderName(placeHolderName), m_Description(description), m_IsRequired(isRequired) {}

            static KeywordArgumentDocument MakeKeywordFlagArgument(char shortName, std::string longName, std::string placeHolderName, std::string description, bool isRequired)
            {
                return KeywordArgumentDocument(shortName, longName, false, "", description, isRequired);
            }

            static KeywordArgumentDocument MakeKeywordValueArgument(char shortName, std::string longName, std::string placeHolderName, std::string description, bool isRequired)
            {
                return KeywordArgumentDocument(shortName, longName, true, placeHolderName, description, isRequired);
            }

            bool HasShortName() const { return m_ShortName[0] != '\0'; }
            bool HasLongName() const { return m_LongName != ""; }
            std::string GetShortName() const { return m_ShortName; }
            std::string GetLongName() const { return m_LongName; }
            bool HasValue() const { return m_HasValue; }
            std::string GetPlaceHolderName() const { return m_PlaceHolderName; }
            std::string GetDescription() const { return m_Description; }
            std::string GetDisplayName() const;
            bool IsRequired() const { return m_IsRequired; }

        private:
            std::string m_ShortName;
            std::string m_LongName;
            bool m_HasValue;
            std::string m_PlaceHolderName;
            std::string m_Description;
            bool m_IsRequired;
        };

        //! @brief 位置引数のドキュメント
        class PositionalArgumentDocument
        {
        public:
            PositionalArgumentDocument(int index, std::string placeHolderName, std::string description, bool isRequired) : m_Index(index), m_PlaceHolderName(placeHolderName), m_Description(description), m_IsRequired(isRequired) {}
            int GetIndex() const { return m_Index; }
            std::string GetPlaceHolderName() const { return m_PlaceHolderName; }
            std::string GetDescription() const { return m_Description; }
            bool IsRequired() const { return m_IsRequired; }
        private:
            int m_Index;
            std::string m_PlaceHolderName;
            std::string m_Description;
            bool m_IsRequired;
        };

        //! @brief 可変長引数のドキュメント
        class VariableArgumentDocument
        {
        public:
            VariableArgumentDocument() {}
            VariableArgumentDocument(std::string placeHolderName, std::string description, bool isRequired) : m_PlaceHolderName(placeHolderName), m_Description(description), m_IsRequired(isRequired) {}
            std::string GetPlaceHolderName() const { return m_PlaceHolderName; }
            std::string GetDescription() const { return m_Description; }
            bool IsRequired() const { return m_IsRequired; }
        private:
            std::string m_PlaceHolderName;
            std::string m_Description;
            bool m_IsRequired;
        };

        //! @brief コマンドのドキュメント
        class CommandDocument
        {
        public:
            CommandDocument() : m_Name("NoName"), m_Summary("No summary."), m_HasVariableArgument(false) {}

            std::string GetName() const { return m_Name; }
            std::string GetSummary() const { return m_Summary; }
            std::string GetDescription() const { return m_Description; }
            std::vector<KeywordArgumentDocument>& GetKeywordArguments() { return m_KeywordArguments; }
            std::vector<PositionalArgumentDocument>& GetPositionalArguments() { return m_PositionalArguments; }
            VariableArgumentDocument GetVariableArgument() const { return m_VariableArgument; }
            bool HasVariableArgument() const { return m_HasVariableArgument; }
            bool HasArguments() const { return m_KeywordArguments.size() != 0 || m_PositionalArguments.size() != 0 || m_HasVariableArgument; }

            void SetName(std::string name) { m_Name = name; }
            void SetSummary(std::string summary) { m_Summary = summary; }
            void SetDescription(std::string description) { m_Description = description; }
            void SetVariableArgument(const VariableArgumentDocument &variableArgument) { m_HasVariableArgument = true; m_VariableArgument = variableArgument; }
            void AddKeywordArgument(const KeywordArgumentDocument &keywordArgument) { m_KeywordArguments.push_back(keywordArgument); }
            void AddPositionalArgument(const PositionalArgumentDocument &positionalArgument);

        private:
            std::string m_Name;
            std::string m_Summary;
            std::string m_Description;
            std::vector<KeywordArgumentDocument> m_KeywordArguments;
            std::vector<PositionalArgumentDocument> m_PositionalArguments;
            bool m_HasVariableArgument;
            VariableArgumentDocument m_VariableArgument;
        };

        //! @brief コマンドのヘルプをテキストで作成する
        std::string MakePlainTextCommandDocument(CommandDocument &commandDocument);
    }
}
