﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <vector>
#include <type_traits>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_BinTypes.h>
#include <nn/util/util_ResDic.h>
#include <nn/util/util_IntrusiveDic.h>
#include <nn/util/util_MemorySplitter.h>
#include <nn/utilTool/utilTool_BinarizerContext.h>

namespace nn { namespace utilTool {

//! @brief バイナリコンバータユーティリティ
class BinarizerUtil
{
public:
    //! @brief      コンテナとキーを元にリソース用の辞書を構築します。
    //! @tparam     IterTPtr       ポインタ型を要素とするコンテナのイテレータの型です。
    //! @tparam     KeySelector コンテナの要素を引数として、キー文字列を返す関数オブジェクトの型です。
    //! @param[in]  pCtx        バイナリコンバータのコンテクストです。
    //! @param[in]  pBlock      辞書構造体のデータを配置するメモリブロックです。
    //! @param[in]  begin       コンテナの先頭のイテレータです。
    //! @param[in]  end         コンテナの終端のイテレータです。
    //! @param[in]  keySelector コンテナの要素を引数として、キー文字列を返す関数オブジェクトです。
    //! @return     構築に成功すれば true を返します。
    //! @pre        メモリブロック内の領域は確保済みである必要があります。
    //! @pre         @ref BinarizerContext::ConvertStringPool() より後に呼び出す必要があります。
    //! @pre        キー文字列は @ref BinarizerContext::InsertString() により登録済みである必要があります。
    template< typename IterTPtr, typename KeySelector >
    static bool BuildResDic(
        BinarizerContext* pCtx,
        nn::util::MemorySplitter::MemoryBlock* pBlock,
        const IterTPtr& begin,
        const IterTPtr& end,
        KeySelector keySelector ) NN_NOEXCEPT;
};

//--------------------------------------------------------------------------------------------------

template< typename IterTPtr, typename KeySelector >
inline
bool BinarizerUtil::BuildResDic(
    BinarizerContext* pCtx,
    nn::util::MemorySplitter::MemoryBlock* pBlock,
    const IterTPtr& begin,
    const IterTPtr& end,
    KeySelector keySelector ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( std::is_pointer< typename IterTPtr::value_type >::value );

    void* pBase = pCtx->GetBasePtr();

    nn::util::ResDicData* pResDic = pBlock->Get< nn::util::ResDicData >( pBase );
    std::memcpy( pResDic->signature._str, "_DIC", 4 );

    // 各ノードの参照ビットと子ノードインデックスを移植します。
    int index = 0;
    for ( IterTPtr iter = begin; iter != end; ++iter, ++index )
    {
        nn::util::ResDicData::Entry& dicEntry = pResDic->entries[ 1 + index ];
        pCtx->LinkString( &dicEntry.pKey, keySelector( **iter ) );

        // ResDic の構築時にポインタが必要なので Relocate します。
        dicEntry.pKey.Relocate( pBase );
    }
    pResDic->count = static_cast< int32_t >( std::distance( begin, end ) );

    // ルートノードの参照ビットと子ノードインデックスを移植します。
    nn::util::ResDicData::Entry& rootEntry = pResDic->entries[ 0 ];
    pCtx->LinkString( &rootEntry.pKey, "" );
    rootEntry.pKey.Relocate( pBase );

    bool result = static_cast< nn::util::ResDic* >( pResDic )->Build();

    for ( int i = 0; i <= index; ++i)
    {
        // オフセットに戻します。
        pResDic->entries[ i ].pKey.Unrelocate( pBase );
    }

    return result;
}

}} // namespace nn::utilTool
