﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/*!
    @file
    @brief スコープを抜ける際にスコープ内の処理が失敗していた場合に実行するコードを記述するためのマクロを定義しています。
*/

#include <nn/nn_Common.h>
#include <utility>

namespace nn { namespace util {

class ScopedTransaction
{
private:
    bool m_IsCommitted = false;
public:
    ScopedTransaction() = default;
    ScopedTransaction(const ScopedTransaction&) = delete;
    ScopedTransaction(ScopedTransaction&&) = delete;
    ScopedTransaction& operator=(const ScopedTransaction&) = delete;
    ScopedTransaction& operator=(ScopedTransaction&&) = delete;
    void Commit() NN_NOEXCEPT
    {
        m_IsCommitted = true;
    }
    bool IsCommitted() const NN_NOEXCEPT
    {
        return m_IsCommitted;
    }
};

namespace detail {

template <typename F>
class ScopedTransactionHolder
{
private:

    ScopedTransaction& m_ScopedTransaction;
    F m_F;
    bool m_Valid;

public:

    ScopedTransactionHolder(const ScopedTransactionHolder&) = delete;
    ScopedTransactionHolder& operator=(const ScopedTransactionHolder&) = delete;
    ScopedTransactionHolder& operator=(ScopedTransactionHolder&&) = delete;

    explicit ScopedTransactionHolder(ScopedTransaction& scopedTransaction, F&& f) NN_NOEXCEPT
        : m_ScopedTransaction(scopedTransaction)
        , m_F(std::forward<F>(f))
        , m_Valid(true)
    {
    }

    ScopedTransactionHolder(ScopedTransactionHolder&& other) NN_NOEXCEPT
        : m_ScopedTransaction(other.m_ScopedTransaction)
        , m_F(std::move(other.m_F))
        , m_Valid(std::exchange(other.m_Valid, false))
    {
    }

    ~ScopedTransactionHolder() NN_NOEXCEPT
    {
        if (m_Valid && !m_ScopedTransaction.IsCommitted())
        {
            m_F();
        }
    }

};

struct ScopedTransactionHolderMarker
{
    ScopedTransaction& scopedTransaction;
};

template <typename F>
inline ScopedTransactionHolder<F> operator+(const ScopedTransactionHolderMarker& maker, F&& f) NN_NOEXCEPT
{
    return ScopedTransactionHolder<F>(maker.scopedTransaction, std::forward<F>(f));
}

} // detail

}}

#define NN_UTIL_RESERVE_SCOPED_ROLLBACK(trx) \
    const auto NN_MAKE_TEMPORARY_NAME(_nn_util_reserve_scoped_rollback_) = ::nn::util::detail::ScopedTransactionHolderMarker{trx} + [&] ()
