﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Common.h>

/**
* @file
* @brief    国の定義とユーティリティ関数を提供します
*/

namespace nn
{
namespace util
{

/** @brief ISO 3166-1で定義される国を表す列挙定数
    @details
        ISO 3166-1で定義される国を表す列挙定数です。
        ISO 3166-1 alpha-2に接頭辞"Country_"を付加した定数名に対し、ISO 3166-1 Numericの値を持ちます。

        この定義は最新のISO定義と異なる場合があることに注意してください。
*/
enum Country
{
    Country_Ad = 20, //!< AD, 20, Andorra
    Country_Ae = 784, //!< AE, 784, United Arab Emirates (the)
    Country_Af = 4, //!< AF, 4, Afghanistan
    Country_Ag = 28, //!< AG, 28, Antigua and Barbuda
    Country_Ai = 660, //!< AI, 660, Anguilla
    Country_Al = 8, //!< AL, 8, Albania
    Country_Am = 51, //!< AM, 51, Armenia
    Country_Ao = 24, //!< AO, 24, Angola
    Country_Aq = 10, //!< AQ, 10, Antarctica
    Country_Ar = 32, //!< AR, 32, Argentina
    Country_As = 16, //!< AS, 16, American Samoa
    Country_At = 40, //!< AT, 40, Austria
    Country_Au = 36, //!< AU, 36, Australia
    Country_Aw = 533, //!< AW, 533, Aruba
    Country_Ax = 248, //!< AX, 248, Åland Islands
    Country_Az = 31, //!< AZ, 31, Azerbaijan
    Country_Ba = 70, //!< BA, 70, Bosnia and Herzegovina
    Country_Bb = 52, //!< BB, 52, Barbados
    Country_Bd = 50, //!< BD, 50, Bangladesh
    Country_Be = 56, //!< BE, 56, Belgium
    Country_Bf = 854, //!< BF, 854, Burkina Faso
    Country_Bg = 100, //!< BG, 100, Bulgaria
    Country_Bh = 48, //!< BH, 48, Bahrain
    Country_Bi = 108, //!< BI, 108, Burundi
    Country_Bj = 204, //!< BJ, 204, Benin
    Country_Bl = 652, //!< BL, 652, Saint Barthélemy
    Country_Bm = 60, //!< BM, 60, Bermuda
    Country_Bn = 96, //!< BN, 96, Brunei Darussalam
    Country_Bo = 68, //!< BO, 68, Bolivia (Plurinational State of)
    Country_Bq = 535, //!< BQ, 535, Bonaire, Sint Eustatius and Saba
    Country_Br = 76, //!< BR, 76, Brazil
    Country_Bs = 44, //!< BS, 44, Bahamas (the)
    Country_Bt = 64, //!< BT, 64, Bhutan
    Country_Bv = 74, //!< BV, 74, Bouvet Island
    Country_Bw = 72, //!< BW, 72, Botswana
    Country_By = 112, //!< BY, 112, Belarus
    Country_Bz = 84, //!< BZ, 84, Belize
    Country_Ca = 124, //!< CA, 124, Canada
    Country_Cc = 166, //!< CC, 166, Cocos (Keeling) Islands (the)
    Country_Cd = 180, //!< CD, 180, Congo (the Democratic Republic of the)
    Country_Cf = 140, //!< CF, 140, Central African Republic (the)
    Country_Cg = 178, //!< CG, 178, Congo (the)
    Country_Ch = 756, //!< CH, 756, Switzerland
    Country_Ci = 384, //!< CI, 384, Côte d'Ivoire
    Country_Ck = 184, //!< CK, 184, Cook Islands (the)
    Country_Cl = 152, //!< CL, 152, Chile
    Country_Cm = 120, //!< CM, 120, Cameroon
    Country_Cn = 156, //!< CN, 156, China
    Country_Co = 170, //!< CO, 170, Colombia
    Country_Cr = 188, //!< CR, 188, Costa Rica
    Country_Cu = 192, //!< CU, 192, Cuba
    Country_Cv = 132, //!< CV, 132, Cabo Verde
    Country_Cw = 531, //!< CW, 531, Curaçao
    Country_Cx = 162, //!< CX, 162, Christmas Island
    Country_Cy = 196, //!< CY, 196, Cyprus
    Country_Cz = 203, //!< CZ, 203, Czechia
    Country_De = 276, //!< DE, 276, Germany
    Country_Dj = 262, //!< DJ, 262, Djibouti
    Country_Dk = 208, //!< DK, 208, Denmark
    Country_Dm = 212, //!< DM, 212, Dominica
    Country_Do = 214, //!< DO, 214, Dominican Republic (the)
    Country_Dz = 12, //!< DZ, 12, Algeria
    Country_Ec = 218, //!< EC, 218, Ecuador
    Country_Ee = 233, //!< EE, 233, Estonia
    Country_Eg = 818, //!< EG, 818, Egypt
    Country_Eh = 732, //!< EH, 732, Western Sahara*
    Country_Er = 232, //!< ER, 232, Eritrea
    Country_Es = 724, //!< ES, 724, Spain
    Country_Et = 231, //!< ET, 231, Ethiopia
    Country_Fi = 246, //!< FI, 246, Finland
    Country_Fj = 242, //!< FJ, 242, Fiji
    Country_Fk = 238, //!< FK, 238, Falkland Islands (the) [Malvinas]
    Country_Fm = 583, //!< FM, 583, Micronesia (Federated States of)
    Country_Fo = 234, //!< FO, 234, Faroe Islands (the)
    Country_Fr = 250, //!< FR, 250, France
    Country_Ga = 266, //!< GA, 266, Gabon
    Country_Gb = 826, //!< GB, 826, United Kingdom of Great Britain and Northern Ireland (the)
    Country_Gd = 308, //!< GD, 308, Grenada
    Country_Ge = 268, //!< GE, 268, Georgia
    Country_Gf = 254, //!< GF, 254, French Guiana
    Country_Gg = 831, //!< GG, 831, Guernsey
    Country_Gh = 288, //!< GH, 288, Ghana
    Country_Gi = 292, //!< GI, 292, Gibraltar
    Country_Gl = 304, //!< GL, 304, Greenland
    Country_Gm = 270, //!< GM, 270, Gambia (the)
    Country_Gn = 324, //!< GN, 324, Guinea
    Country_Gp = 312, //!< GP, 312, Guadeloupe
    Country_Gq = 226, //!< GQ, 226, Equatorial Guinea
    Country_Gr = 300, //!< GR, 300, Greece
    Country_Gs = 239, //!< GS, 239, South Georgia and the South Sandwich Islands
    Country_Gt = 320, //!< GT, 320, Guatemala
    Country_Gu = 316, //!< GU, 316, Guam
    Country_Gw = 624, //!< GW, 624, Guinea-Bissau
    Country_Gy = 328, //!< GY, 328, Guyana
    Country_Hk = 344, //!< HK, 344, Hong Kong
    Country_Hm = 334, //!< HM, 334, Heard Island and McDonald Islands
    Country_Hn = 340, //!< HN, 340, Honduras
    Country_Hr = 191, //!< HR, 191, Croatia
    Country_Ht = 332, //!< HT, 332, Haiti
    Country_Hu = 348, //!< HU, 348, Hungary
    Country_Id = 360, //!< ID, 360, Indonesia
    Country_Ie = 372, //!< IE, 372, Ireland
    Country_Il = 376, //!< IL, 376, Israel
    Country_Im = 833, //!< IM, 833, Isle of Man
    Country_In = 356, //!< IN, 356, India
    Country_Io = 86, //!< IO, 86, British Indian Ocean Territory (the)
    Country_Iq = 368, //!< IQ, 368, Iraq
    Country_Ir = 364, //!< IR, 364, Iran (Islamic Republic of)
    Country_Is = 352, //!< IS, 352, Iceland
    Country_It = 380, //!< IT, 380, Italy
    Country_Je = 832, //!< JE, 832, Jersey
    Country_Jm = 388, //!< JM, 388, Jamaica
    Country_Jo = 400, //!< JO, 400, Jordan
    Country_Jp = 392, //!< JP, 392, Japan
    Country_Ke = 404, //!< KE, 404, Kenya
    Country_Kg = 417, //!< KG, 417, Kyrgyzstan
    Country_Kh = 116, //!< KH, 116, Cambodia
    Country_Ki = 296, //!< KI, 296, Kiribati
    Country_Km = 174, //!< KM, 174, Comoros (the)
    Country_Kn = 659, //!< KN, 659, Saint Kitts and Nevis
    Country_Kp = 408, //!< KP, 408, Korea (the Democratic People's Republic of)
    Country_Kr = 410, //!< KR, 410, Korea (the Republic of)
    Country_Kw = 414, //!< KW, 414, Kuwait
    Country_Ky = 136, //!< KY, 136, Cayman Islands (the)
    Country_Kz = 398, //!< KZ, 398, Kazakhstan
    Country_La = 418, //!< LA, 418, Lao People's Democratic Republic (the)
    Country_Lb = 422, //!< LB, 422, Lebanon
    Country_Lc = 662, //!< LC, 662, Saint Lucia
    Country_Li = 438, //!< LI, 438, Liechtenstein
    Country_Lk = 144, //!< LK, 144, Sri Lanka
    Country_Lr = 430, //!< LR, 430, Liberia
    Country_Ls = 426, //!< LS, 426, Lesotho
    Country_Lt = 440, //!< LT, 440, Lithuania
    Country_Lu = 442, //!< LU, 442, Luxembourg
    Country_Lv = 428, //!< LV, 428, Latvia
    Country_Ly = 434, //!< LY, 434, Libya
    Country_Ma = 504, //!< MA, 504, Morocco
    Country_Mc = 492, //!< MC, 492, Monaco
    Country_Md = 498, //!< MD, 498, Moldova (the Republic of)
    Country_Me = 499, //!< ME, 499, Montenegro
    Country_Mf = 663, //!< MF, 663, Saint Martin (French part)
    Country_Mg = 450, //!< MG, 450, Madagascar
    Country_Mh = 584, //!< MH, 584, Marshall Islands (the)
    Country_Mk = 807, //!< MK, 807, Macedonia (the former Yugoslav Republic of)
    Country_Ml = 466, //!< ML, 466, Mali
    Country_Mm = 104, //!< MM, 104, Myanmar
    Country_Mn = 496, //!< MN, 496, Mongolia
    Country_Mo = 446, //!< MO, 446, Macao
    Country_Mp = 580, //!< MP, 580, Northern Mariana Islands (the)
    Country_Mq = 474, //!< MQ, 474, Martinique
    Country_Mr = 478, //!< MR, 478, Mauritania
    Country_Ms = 500, //!< MS, 500, Montserrat
    Country_Mt = 470, //!< MT, 470, Malta
    Country_Mu = 480, //!< MU, 480, Mauritius
    Country_Mv = 462, //!< MV, 462, Maldives
    Country_Mw = 454, //!< MW, 454, Malawi
    Country_Mx = 484, //!< MX, 484, Mexico
    Country_My = 458, //!< MY, 458, Malaysia
    Country_Mz = 508, //!< MZ, 508, Mozambique
    Country_Na = 516, //!< NA, 516, Namibia
    Country_Nc = 540, //!< NC, 540, New Caledonia
    Country_Ne = 562, //!< NE, 562, Niger (the)
    Country_Nf = 574, //!< NF, 574, Norfolk Island
    Country_Ng = 566, //!< NG, 566, Nigeria
    Country_Ni = 558, //!< NI, 558, Nicaragua
    Country_Nl = 528, //!< NL, 528, Netherlands (the)
    Country_No = 578, //!< NO, 578, Norway
    Country_Np = 524, //!< NP, 524, Nepal
    Country_Nr = 520, //!< NR, 520, Nauru
    Country_Nu = 570, //!< NU, 570, Niue
    Country_Nz = 554, //!< NZ, 554, New Zealand
    Country_Om = 512, //!< OM, 512, Oman
    Country_Pa = 591, //!< PA, 591, Panama
    Country_Pe = 604, //!< PE, 604, Peru
    Country_Pf = 258, //!< PF, 258, French Polynesia
    Country_Pg = 598, //!< PG, 598, Papua New Guinea
    Country_Ph = 608, //!< PH, 608, Philippines (the)
    Country_Pk = 586, //!< PK, 586, Pakistan
    Country_Pl = 616, //!< PL, 616, Poland
    Country_Pm = 666, //!< PM, 666, Saint Pierre and Miquelon
    Country_Pn = 612, //!< PN, 612, Pitcairn
    Country_Pr = 630, //!< PR, 630, Puerto Rico
    Country_Ps = 275, //!< PS, 275, Palestine, State of
    Country_Pt = 620, //!< PT, 620, Portugal
    Country_Pw = 585, //!< PW, 585, Palau
    Country_Py = 600, //!< PY, 600, Paraguay
    Country_Qa = 634, //!< QA, 634, Qatar
    Country_Re = 638, //!< RE, 638, Réunion
    Country_Ro = 642, //!< RO, 642, Romania
    Country_Rs = 688, //!< RS, 688, Serbia
    Country_Ru = 643, //!< RU, 643, Russian Federation (the)
    Country_Rw = 646, //!< RW, 646, Rwanda
    Country_Sa = 682, //!< SA, 682, Saudi Arabia
    Country_Sb = 90, //!< SB, 90, Solomon Islands
    Country_Sc = 690, //!< SC, 690, Seychelles
    Country_Sd = 729, //!< SD, 729, Sudan (the)
    Country_Se = 752, //!< SE, 752, Sweden
    Country_Sg = 702, //!< SG, 702, Singapore
    Country_Sh = 654, //!< SH, 654, Saint Helena, Ascension and Tristan da Cunha
    Country_Si = 705, //!< SI, 705, Slovenia
    Country_Sj = 744, //!< SJ, 744, Svalbard and Jan Mayen
    Country_Sk = 703, //!< SK, 703, Slovakia
    Country_Sl = 694, //!< SL, 694, Sierra Leone
    Country_Sm = 674, //!< SM, 674, San Marino
    Country_Sn = 686, //!< SN, 686, Senegal
    Country_So = 706, //!< SO, 706, Somalia
    Country_Sr = 740, //!< SR, 740, Suriname
    Country_Ss = 728, //!< SS, 728, South Sudan
    Country_St = 678, //!< ST, 678, Sao Tome and Principe
    Country_Sv = 222, //!< SV, 222, El Salvador
    Country_Sx = 534, //!< SX, 534, Sint Maarten (Dutch part)
    Country_Sy = 760, //!< SY, 760, Syrian Arab Republic
    Country_Sz = 748, //!< SZ, 748, Swaziland
    Country_Tc = 796, //!< TC, 796, Turks and Caicos Islands (the)
    Country_Td = 148, //!< TD, 148, Chad
    Country_Tf = 260, //!< TF, 260, French Southern Territories (the)
    Country_Tg = 768, //!< TG, 768, Togo
    Country_Th = 764, //!< TH, 764, Thailand
    Country_Tj = 762, //!< TJ, 762, Tajikistan
    Country_Tk = 772, //!< TK, 772, Tokelau
    Country_Tl = 626, //!< TL, 626, Timor-Leste
    Country_Tm = 795, //!< TM, 795, Turkmenistan
    Country_Tn = 788, //!< TN, 788, Tunisia
    Country_To = 776, //!< TO, 776, Tonga
    Country_Tr = 792, //!< TR, 792, Turkey
    Country_Tt = 780, //!< TT, 780, Trinidad and Tobago
    Country_Tv = 798, //!< TV, 798, Tuvalu
    Country_Tw = 158, //!< TW, 158, Taiwan (Province of China)
    Country_Tz = 834, //!< TZ, 834, Tanzania, United Republic of
    Country_Ua = 804, //!< UA, 804, Ukraine
    Country_Ug = 800, //!< UG, 800, Uganda
    Country_Um = 581, //!< UM, 581, United States Minor Outlying Islands (the)
    Country_Us = 840, //!< US, 840, United States of America (the)
    Country_Uy = 858, //!< UY, 858, Uruguay
    Country_Uz = 860, //!< UZ, 860, Uzbekistan
    Country_Va = 336, //!< VA, 336, Holy See (the)
    Country_Vc = 670, //!< VC, 670, Saint Vincent and the Grenadines
    Country_Ve = 862, //!< VE, 862, Venezuela (Bolivarian Republic of)
    Country_Vg = 92, //!< VG, 92, Virgin Islands (British)
    Country_Vi = 850, //!< VI, 850, Virgin Islands (U.S.)
    Country_Vn = 704, //!< VN, 704, Viet Nam
    Country_Vu = 548, //!< VU, 548, Vanuatu
    Country_Wf = 876, //!< WF, 876, Wallis and Futuna
    Country_Ws = 882, //!< WS, 882, Samoa
    Country_Ye = 887, //!< YE, 887, Yemen
    Country_Yt = 175, //!< YT, 175, Mayotte
    Country_Za = 710, //!< ZA, 710, South Africa
    Country_Zm = 894, //!< ZM, 894, Zambia
    Country_Zw = 716, //!< ZW, 716, Zimbabwe
};

/**
    @brief ISO 3166-1 alpha-2に準拠する国コード文字列から nn::util::Country の値を取得します。

    @param[out] pOutCountry code の表す国コード文字列と対応する nn::util::Country の値
    @param[in] code ISO 3166-1 alpha-2に準拠する国コード文字列
    @param[in] bytes code の指す領域の大きさ(バイト数)

    @return 処理に成功した場合に限り true

    @pre
        - pOutCountry != nullptr
        - code != nullptr
        - code[0,bytes) にひとつ以上の '\0' を含む

    @details
        ISO 3166-1 alpha-2に準拠する国コード文字列から nn::util::Country の値を取得します。
        ただし対応するISO 3166-1 alpha-2定義は本関数の実装当時のものであり、関数呼び出し時点でのISO定義と異なる可能性があります。
        従ってインターネット通信等で取得した国コードを指定する場合、本関数が false を返す可能性を考慮してプログラムを作成してください。
*/
bool GetCountryFromIso31661Alpha2(Country* pOutCountry, const char* code, size_t bytes) NN_NOEXCEPT;

} // ~namespace nn::util
}
