﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace util {

/**
*   @file
*   @brief base64 エンコード・デコードを扱うためのクラスです。
*/
class Base64
{
public:

    /**
    *   @brief  base64 エンコード・デコードの処理結果を示す列挙型です。
    */
    enum Status
    {
        Status_Success,     //!< 成功。
        Status_BufferFull,  //!< 出力バッファのサイズ不足。
        Status_BadData      //!< 入力データが不正。
    };

    /**
    *   @brief  base64 エンコード・デコードのモードを示す列挙型です。
    */
    enum Mode
    {
        Mode_Normal,    //!< 通常の base64 エンコード・デコードを行うモード。エンコード結果は76文字ごとに改行コード(CRLF)が挿入され、全体の文字数が4の倍数になるよう末尾が '=' でパディングされます。
        Mode_NormalNoLinefeed, //!< 通常の base64 エンコード・デコードを行うモード。全体の文字数が4の倍数になるよう末尾が '=' でパディングされます。エンコード結果に改行コードは挿入されません。
        Mode_UrlSafe    //!< URLセーフな base64 エンコード・デコードを行うモード。エンコード結果は改行コード、パディング文字を含みません。
    };

    /*!
    *   @brief    入力バイナリデータを base64 形式の文字列に変換します。
    *
    *   @param[out]  pDst     base64 形式文字列の出力先バッファへのポインタ
    *   @param[in]   sizeDst  pDst で示されるバッファのサイズ
    *   @param[in]   pSrc     入力データへのポインタ
    *   @param[in]   sizeSrc  入力データのサイズ
    *   @param[in]   mode     base64 エンコードのモードを示す列挙型
    *   @pre    引数がNULLの場合は内部で assert により停止します。
    *   @return 処理結果を返します。
    *   @retval nn::Base64::Status_Success      処理が正常に終了しました。
    *   @retval nn::Base64::Status_BufferFull   出力バッファのサイズ不足により処理が正しく行われませんでした。
    */
    static Status ToBase64String(char* pDst, size_t sizeDst, const void* pSrc, size_t sizeSrc, Mode mode) NN_NOEXCEPT;

    /*!
    *   @brief    base64 形式の文字列からバイナリデータに変換します。
    *
    *   @param[out]  pNum     実際に出力した文字数を格納するポインタ
    *   @param[out]  pDst     出力データ格納先バッファへのポインタ
    *   @param[in]   sizeDst  pDst で示されるバッファのサイズ
    *   @param[in]   pSrc     入力 base64 形式文字列へのポインタ
    *   @param[in]   mode     base64 デコードのモードを示す列挙型
    *   @pre    引数がNULLの場合は内部で assert により停止します。
    *   @return 処理結果を返します。
    *   @retval nn::Base64::Status_Success      処理が正常に終了しました。
    *   @retval nn::Base64::Status_BufferFull   出力バッファのサイズ不足により処理が正しく行われませんでした。
    *   @retval nn::Base64::Status_BadData      不正な入力データにより処理が正しく行われませんでした。
    *
    *   @details
    *       pSrc がヌル終端していない場合の動作は未定義です。
    */
    static Status FromBase64String(size_t* pNum, void* pDst, size_t sizeDst, const char* pSrc, Mode mode) NN_NOEXCEPT;

    /*!
    *   @brief    base64 形式の文字列からバイナリデータに変換します。
    *
    *   @param[out]  pNum     実際に出力した文字数を格納するポインタ
    *   @param[out]  pDst     出力データ格納先バッファへのポインタ
    *   @param[in]   sizeDst  pDst で示されるバッファのサイズ
    *   @param[in]   pSrc     入力 base64 形式文字列へのポインタ
    *   @param[in]   sizeSrc  変換対象となる pSrc の要素数
    *   @param[in]   mode     base64 デコードのモードを示す列挙型
    *   @pre    引数がNULLの場合は内部で assert により停止します。
    *   @return 処理結果を返します。
    *   @retval nn::Base64::Status_Success      処理が正常に終了しました。
    *   @retval nn::Base64::Status_BufferFull   出力バッファのサイズ不足により処理が正しく行われませんでした。
    *   @retval nn::Base64::Status_BadData      不正な入力データにより処理が正しく行われませんでした。
    *
    *   @details
    *       pSrc の sizeSrc - 1 の要素までを変換します。
    *
    *       pSrc 中にヌル終端が現れた時点で変換を終えるので、必ずしも sizeSrc 分の変換が行われるわけではありません。
    */
    static Status FromBase64String(size_t* pNum, void* pDst, size_t sizeDst, const char* pSrc, size_t sizeSrc, Mode mode) NN_NOEXCEPT;

};

}}

