﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using ZarfServerAccessor;

namespace ZarfUploader
{
    class Uploader
    {
        ServerSetting _setting;
        Accessor _ac;
        AccessorRest _acRest;
        ZarfServerClient _zsClient;
        Dictionary<string, IDictionary<string, object>> _familyDic;

        public Uploader(ServerSetting setting)
        {
            _setting = setting;
            if (string.IsNullOrEmpty(_setting.IssuerUri))
            {
                _ac = new Accessor();
                _acRest = new AccessorRest(_setting.ServerUrl, _setting.LoginId, _setting.Password);
            }
            else
            {
                _zsClient = new ZarfServerClient(_setting.ServerUri, _setting.IssuerUri, _setting.ClientId, Convert.FromBase64String(_setting.SharedKey));
            }
        }

        public void Upload(IEnumerable<string> zarfs)
        {
            _Login();
            foreach (var zarf in zarfs)
            {
                if (_ac != null)
                {
                    Console.WriteLine(string.Format("Uploading to MRS: {0}", zarf));
                    var zarfMeta = GetZarfMeta(zarf);
                    var zarfAbout = zarfMeta.GetValueOrDefault<IDictionary<string, object>>("about");
                    var zarfFamily = zarfAbout?.GetValueOrDefault<IDictionary<string, object>>("family");
                    var zarfDepartment = zarfFamily?.GetValueOrDefault<string>("department");
                    var zarfDepartmentKey = zarfFamily?.GetValueOrDefault<string>("key");
                    var zarfRelease = zarfAbout?.GetValueOrDefault<IDictionary<string, object>>("release");
                    var zarfDescription = zarfRelease?.GetValueOrDefault<IDictionary<string, object>>("description");
                    var zarfDescriptionJa = zarfDescription?.GetValueOrDefault<string>("ja");
                    var zarfDescriptionEn = zarfDescription?.GetValueOrDefault<string>("en");

                    var isExistFamily =
                        (   // zarfの情報が取れない場合もアップロードを試行したいため、ファミリありとする。
                            string.IsNullOrEmpty(zarfDepartment)
                            || string.IsNullOrEmpty(zarfDepartmentKey)
                            || (string.IsNullOrEmpty(zarfDescriptionJa) && string.IsNullOrEmpty(zarfDescriptionEn))
                        )
                        || _familyDic.ContainsKey(zarfDepartmentKey);
                    if (!isExistFamily)
                    {
                        string errMsg;
                        isExistFamily =
                            _acRest.TryCreateFamily(
                                zarfDepartmentKey,
                                zarfDepartmentKey,
                                zarfDescriptionJa ?? zarfDepartmentKey,
                                zarfDescriptionEn ?? zarfDepartmentKey,
                                zarfDepartment,
                                zarfDepartmentKey,
                                zarfRelease?.GetValueOrDefault<object>("addOnTo") != null,
                                out errMsg);
                        if (! isExistFamily)
                        {
                            Console.Error.WriteLine(string.Format("Failed: Create Family: {0}: {1}", zarfDepartmentKey, errMsg));
                        }
                    }

                    if (isExistFamily)
                    {
                        string errMsg;
                        if (!_ac.TryUploadZarf(zarf, out errMsg))
                        {
                            Console.Error.WriteLine(string.Format("Failed: Upload Zarf: {0}: {1}", zarf, errMsg));
                        }
                    }
                }
                else
                {
                    Console.WriteLine(string.Format("Uploading to Zarf Server: {0}", zarf));
                    try
                    {
                        _zsClient.UploadFileAsync(File.OpenRead(zarf), Path.GetFileName(zarf)).Wait();
                    }
                    catch (Exception ex)
                    {
                        Console.Error.WriteLine(string.Format("Failed: Upload Zarf: {0}: {1}", zarf, ex.ToString()));
                    }
                }
            }
        }

        public void SetCoreReleaseZarf(string coreRelease, string coreReleaseZarf)
        {
            if (_ac != null)
            {
                _Login();
                _ac.SetCoreReleaseZarf(coreRelease, coreReleaseZarf);
            }
        }

        private void _Login()
        {
            if (_ac != null)
            {
                if (!_ac.IsLogin)
                {
                    _ac.Login(_setting.ServerUrl, _setting.LoginId, _setting.Password);
                }
            }
        }

        public void StockControl(IEnumerable<string> zarfs, string familyRegex, int maxCount)
        {
            if (_ac != null)
            {
                _Login();

                var regex = new Regex(familyRegex);

                foreach (var zarf in zarfs)
                {
                    var zarfDepartmentKey =
                        GetZarfMeta(zarf)
                        .GetValueOrDefault<IDictionary<string, object>>("about")
                        ?.GetValueOrDefault<IDictionary<string, object>>("family")
                        ?.GetValueOrDefault<string>("key");

                    IDictionary<string, object> family;
                    if (!string.IsNullOrEmpty(zarfDepartmentKey) && _familyDic.TryGetValue(zarfDepartmentKey, out family))
                    {
                        var children = (IEnumerable<object>)family["children"];
                        var snapshots =
                            (from release in children.Cast<IDictionary<string, object>>()
                             let version = (string)release["version"]
                             where regex.IsMatch(version)
                             select release)
                            .Skip(maxCount);

                        foreach (var toRemove in snapshots)
                        {
                            var releaseName = ((IDictionary<string, object>)toRemove["text"]).Values.First();
                            Console.WriteLine(string.Format("Delete: {0}", releaseName));
                            int id = (int)toRemove["hiddenID"];
                            string errMsg;
                            if (!_ac.TryRemoveRelease(id, out errMsg))
                            {
                                Console.Error.WriteLine(string.Format("Failed: Delete Zarf: {0}: {1}", releaseName, errMsg));
                            }
                        }
                    }
                }
            }
        }

        public void LoadFamilyTree()
        {
            if (_ac != null)
            {
                _Login();

                _familyDic =
                    (from family in _ac.GetFamilyTree().Cast<IDictionary<string, object>>()
                    let departmentKey = (string)family["departmentKey"]
                    where !string.IsNullOrEmpty(departmentKey)
                    select family
                    ).ToDictionary(family => (string)family["departmentKey"]);
            }
        }

        private static IDictionary<string, object> GetZarfMeta(string filePath)
        {
            using (var archive = ZipFile.OpenRead(filePath))
            {
                var entry = archive.GetEntry("ndi-metadata.json");
                using (var sw = new StreamReader(entry.Open(), Encoding.UTF8))
                {
                    var selializer = new JavaScriptSerializer();
                    return (IDictionary<string, object>)selializer.DeserializeObject(sw.ReadToEnd());
                }
            }
        }
    }

    static class DictionaryExtension
    {
        /// <summary>
        /// 指定したキーが存在し、指定した型である場合はそのオブジェクトを返します。
        /// キーが無かったり、指定した方でなかった場合は型のデフォルト値が返ります。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="dic"></param>
        /// <param name="key"></param>
        /// <returns></returns>
        public static T GetValueOrDefault<T>(this IDictionary<string, object> dic, string key) where T : class
        {
            object result;
            if (dic.TryGetValue(key, out result))
            {
                return result as T;
            }
            else
            {
                return default(T);
            }
        }
    }
}
