﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Http;
using System.Web.Script.Serialization;

namespace ZarfServerAccessor
{
    public class ZarfServerClient
    {
        private Uri apiUri;
        private Uri issuerUri;
        private string clientId;
        private byte[] sharedKey;

        public ZarfServerClient(string apiUri, string issuerUri, string clientId, byte[] sharedKey)
        {
            this.apiUri = new Uri(apiUri);
            this.issuerUri = new Uri(issuerUri);
            this.clientId = clientId;
            this.sharedKey = sharedKey;
        }

        protected HttpClient BuildClient()
        {
            HttpClient client = new HttpClient();
            client.MaxResponseContentBufferSize = 1024 * 1024 * 10;
            client.BaseAddress = apiUri;

            return client;
        }

        protected async Task LoginAsync(HttpClient apiClient)
        {
            using (HttpClient authClient = new HttpClient())
            {
                byte[] prefix = Encoding.UTF8.GetBytes(string.Format("{0}:", clientId));
                byte[] credentialBytes = new byte[prefix.Length + sharedKey.Length];
                Buffer.BlockCopy(prefix, 0, credentialBytes, 0, prefix.Length);
                Buffer.BlockCopy(sharedKey, 0, credentialBytes, prefix.Length, sharedKey.Length);
                var credentials = Convert.ToBase64String(credentialBytes);

                authClient.DefaultRequestHeaders.Authorization = new System.Net.Http.Headers.AuthenticationHeaderValue("Basic", credentials);

                var content = new FormUrlEncodedContent(new List<KeyValuePair<string, string>>
                {
                    new KeyValuePair<string, string>("grant_type", "client_credentials"),
                    new KeyValuePair<string, string>("scope", "ZarfServer.Roles")
                });

                var response = await authClient.PostAsync(issuerUri, content);

                string accessToken;

                using (var sr = new StreamReader(await response.Content.ReadAsStreamAsync()))
                {
                    var serializer = new JavaScriptSerializer();
                    var resultStr = sr.ReadToEnd();
                    var json = (IDictionary<string, object>)serializer.DeserializeObject(resultStr);
                    accessToken = (string)json["access_token"];

                    if (accessToken == null)
                    {
                        throw new Exception("Unexpected Response logging in to zarf auth server: " + resultStr);
                    }
                }

                //switch out the authorization header to the bearer token for api requests
                apiClient.DefaultRequestHeaders.Authorization = new System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", accessToken);
            }
        }

        public async Task UploadFileAsync(Stream fileContents, string fileName)
        {
            using (HttpClient client = BuildClient())
            {
                await LoginAsync(client);
                //set a long timeout because this could take a while
                client.Timeout = TimeSpan.FromHours(2);

                using (var content =
                    new MultipartFormDataContent())
                {
                    content.Add(new StreamContent(fileContents, 1024 * 1024), fileName, fileName);

                    using (
                       var message =
                           await client.PostAsync("/upload", content))
                    {
                        if (!message.IsSuccessStatusCode)
                        {
                            throw new Exception(String.Format("Unexpected Response Code: {0}", message.StatusCode));
                        }
                        string contentStr = await message.Content.ReadAsStringAsync();
                        Guid taskId;
                        if(!Guid.TryParse(contentStr.Trim('"'), out taskId))
                        {
                            throw new Exception(String.Format("Unexpected Response: {0}", contentStr));
                        }
                    }
                }
            }
        }
    }
}
