﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading.Tasks;
using System.Web.Script.Serialization;

namespace ZarfServerAccessor
{
    public class AccessorRest
    {
        string _serverUrl = null;
        string _authToken = null;
        ICredentials _credentials;


        public AccessorRest(string url, string id, string password)
        {
            _serverUrl = url.EndsWith("/") ? url : url + "/";
            _authToken =
                string.Format("Basic {0}",
                    Convert.ToBase64String(
                        Encoding.ASCII.GetBytes(
                            string.Format("{0}:{1}", id, password))));

            var networkCredential = new NetworkCredential(id, password);
            var credentialCache = new CredentialCache();
            credentialCache.Add(new Uri(url), "Basic", networkCredential);
            _credentials = credentialCache;
        }

        public bool TryCreateFamily(string name, string nameEn, string description, string descriptionEn, string department, string departmentKey, bool isAddOn, out string errorMessage)
        {
            errorMessage = null;
            var data =
                new Dictionary<string, object>()
                {
                    { "names",  new Dictionary<string, object>()
                                {
                                    { "ja-JP", name },
                                    { "en-US", nameEn }
                                }
                    },
                    { "department", department },
                    { "departmentKey", departmentKey },
                    { "descriptions",   new Dictionary<string, object>()
                                        {
                                            { "ja-JP", description },
                                            { "en-US", descriptionEn },
                                        }
                    },
                    { "isAddOn", isAddOn }
            };

            var body = CreateSendString(data);
            try
            {
                InvokeNdiServerMethod("POST", "Families", body);
            }
            catch (WebException e)
            {
                // ファミリがあるときに作ろうとしたり、パラメータが足りてなかったりすると、
                // 「リモート サーバーがエラーを返しました: (400) 要求が不適切です」が返る。
                // FileUpload.php などは、JSON でSuccess, Error
                //      WebExceptionStatus.ProtocolError && HttpStatusCode.BadRequest
                errorMessage = e.Message;
                return false;
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="method">メソッドを指定します。</param>
        /// <param name="api"></param>
        /// <param name="body"></param>
        /// <returns></returns>
        private object InvokeNdiServerMethod(string method, string api, string body)
        {
            const int retryMax = 2;

            HttpWebResponse httpResponse = null;
            for (var retryCnt = 0; httpResponse == null; ++retryCnt)
            {
                var uri = _serverUrl + "api/" + api;

                // HTTP でリクエストを送信する準備
                var httpRequest = CreateHttpRequest(method, uri);

                // コンテンツをセット
                if (null != body)
                {
                    var content = Encoding.UTF8.GetBytes(body);
                    httpRequest.ContentLength = content.Length;
                    using (var requestStream = httpRequest.GetRequestStream())
                    {
                        requestStream.Write(content, 0, content.Length);
                    }
                }

                try
                {
                    httpResponse = (HttpWebResponse)httpRequest.GetResponse();
                }
                catch (WebException ex) when (
                    ex.Status == WebExceptionStatus.ProtocolError
                    && ((HttpWebResponse)ex.Response).StatusCode == HttpStatusCode.Unauthorized
                    && retryCnt < retryMax - 1)
                {
                    var msg = string.Format("Retry HttpWebRequest: {0} {1}", method, api);
                    Debug.WriteLine(msg);
                }
            }

            // HTTP リクエストを送信・結果を取得
            using (httpResponse)
            {
                // 結果を取得
                using (var responseStream = httpResponse.GetResponseStream())
                {
                    var responseStr = (new StreamReader(responseStream)).ReadToEnd();
                    // JSON を解析して返す
                    var selializer = new JavaScriptSerializer();
                    return selializer.DeserializeObject(responseStr);
                }
            }
        }

        private HttpWebRequest CreateHttpRequest(string method, string uri, int timeoutSec = 100)
        {
            var httpRequest = (HttpWebRequest)WebRequest.Create(uri);
            httpRequest.Method = method;
            httpRequest.ContentType = "application/json; charset=utf-8";
            httpRequest.Headers["Authorization"] = _authToken;
            httpRequest.UserAgent = Accessor.UserAgent;
            httpRequest.Timeout = timeoutSec * 1000;

            httpRequest.UseDefaultCredentials = false;
            httpRequest.Credentials = _credentials;
            return httpRequest;
        }

        private string CreateSendString(IDictionary<string, object> args)
        {
            var serializer = new JavaScriptSerializer();
            return serializer.Serialize(args);
        }
    }
}
