﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ZarfCreator.ZarfDefinitionData
{
    public class ZarfDefinition
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="contentsLocale">ContentsLocaleの初期値を指定します。</param>
        public ZarfDefinition(string contentsLocale)
        {
            this.ContentsLocale = contentsLocale;
        }

        /// <summary>
        /// Zarf の名前です。Family に相当します。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// Zarf の公開タイプです。
        /// </summary>
        public string Type { get; set; }

        /// <summary>
        /// Zarf の説明です。
        /// </summary>
        public TextInfo Description { get; set; }

        /// <summary>
        /// Zarf のバージョンが書かれたファイルの相対パスです。
        /// </summary>
        public string VersionFile { get; set; }

        /// <summary>
        /// Relsteap をバージョン文字列に含めるかどうかを判別します。
        /// </summary>
        public bool EnableRelstep { get; set; }

        /// <summary>
        /// Zarf のバージョンです。
        /// </summary>
        public string Version { get; set; }

        /// <summary>
        /// Zarf の Family です。
        /// </summary>
        public string Family { get; set; }

        /// <summary>
        /// Zarf の department です。
        /// </summary>
        public string Department { get; set; }

        /// <summary>
        /// アドオン先の Zarf です。
        /// </summary>
        public RelatedZarfInfo AddonTargetZarf { get; set; }

        /// <summary>
        /// Zarf のインストール・アンインストール用のスクリプト等です。
        /// </summary>
        public IEnumerable<string> SetupFiles { get; set; }

        /// <summary>
        /// Zarf のインストール・アンインストール処理が書かれたクラスです。
        /// </summary>
        public InstructionInfo Instruction { get; set; }

        /// <summary>
        /// この Zarf が依存する他の Zarf の相対パスリストです。
        /// </summary>
        public IEnumerable<RelatedZarfInfo> Dependencies { get; set; }

        /// <summary>
        /// この Zarf と互換性のある Zarf の相対パスリストです。
        /// </summary>
        public IEnumerable<RelatedZarfInfo> Compatibilities { get; set; }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<RelatedZarfInfo> RuntimeDependencies { get; set; }

        /// <summary>
        /// この Zarf が対応するContentsLocaleです。
        /// </summary>
        public string ContentsLocale { get; set; }

        /// <summary>
        /// この Zarf が対応するプラットフォームのリストです。
        /// </summary>
        public StringCollection Platforms { get; set; }

        /// <summary>
        /// ロールバック用のリストが必要な場合は true がセットされます。
        /// 不要な場合は false がセットされます。
        /// デフォルトは true です。
        /// </summary>
        public bool NeedRollback { get; set; }

        public IEnumerable<string> RollbackFiles { get; set; }

        public RollbackInfo Rollback { get; set; }

        public TextInfo Eula { get; set; }

        /// <summary>
        /// 必要フィールドが設定されているかどうかを検証します。必須でないフィールドにはデフォルト値が入ります。
        /// </summary>
        /// <param name="errorMessage">失敗時のエラーメッセージ</param>
        /// <returns>検証結果</returns>
        public bool Validate(out string errorMessage)
        {
            var msg = new StringBuilder();
            var result = true;

            // Name は必須
            if (this.Name == null)
            {
                msg.AppendLine("Name is required.");
                result = false;
            }

            // Type は必須
            if (this.Type == null)
            {
                msg.AppendLine("Type is required.");
                result = false;
            }

            // Family は必須
            if (this.Family == null)
            {
                msg.AppendLine("Family is required.");
                result = false;
            }

            // Department は必須
            if (this.Department == null)
            {
                msg.AppendLine("Department is required.");
                result = false;
            }

            // Description は必須
            if (this.Description == null)
            {
                msg.AppendLine("Description is required.");
                result = false;
            }

            // VersionFile は必須
            if (this.VersionFile == null)
            {
                msg.AppendLine("VersionFile is required.");
                result = false;
            }

            // Instructions は必須
            if (this.Instruction == null)
            {
                msg.AppendLine("Instructions is required.");
                result = false;
            }

            // デフォルト値の設定
            if (this.SetupFiles == null)
            {
                this.SetupFiles = new List<string>();
            }

            if (this.Dependencies == null)
            {
                this.Dependencies = new List<RelatedZarfInfo>();
            }

            if (this.Compatibilities == null)
            {
                this.Compatibilities = new List<RelatedZarfInfo>();
            }

            if (this.RuntimeDependencies == null)
            {
                this.RuntimeDependencies = new List<RelatedZarfInfo>();
            }

            // デフォルトでは NintendoSDK に対応させる
            if (this.Platforms == null)
            {
                this.Platforms = new StringCollection();
                this.Platforms.Add("NintendoSDK");
            }

            if (this.RollbackFiles == null)
            {
                this.RollbackFiles = new List<string>();
            }

            errorMessage = msg.ToString();
            return result;
        }
    }
}
