﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace ZarfCreator.VersionData
{
    public static class VersionReplace
    {
        const string primitive = @"<=?|>=?|[=^~]";
        const string partStr = @"[-0-9A-Za-z]+";
        static readonly string qualifier = $@"(?:-{partStr})?(?:\+{partStr})?";
        static readonly string version = $@"([Rr]|\d+)(?:\.([Rr]|\d+)(?:\.([Rr]|\d+){qualifier})?)?";
        static readonly string wildVer = $@"([Xx*Rr]|\d+)(?:\.([Xx*Rr]|\d+)(?:\.([Xx*Rr]|\d+){qualifier})?)?";
        static readonly string versionRange = $@"(?:(?<primitive>{primitive})?{version}(?(primitive)|(?:\s+-\s+{version})?)|{wildVer})";
        static readonly string versionRanges = $@"\G{versionRange}(?:\s*\|\|\s*|\s+(?!-))";
        static readonly Regex versRegex = new Regex(versionRanges);

        static readonly Regex relVerRegex = new Regex(@"^(\d+)(?:\.(\d+)(?:\.(\d+))?)?");

        /// <summary>
        /// 'R.R.*' のような、バージョン文字列中に現れる 'R', 'r' を参照元のバージョンの値に置換します。
        /// </summary>
        /// <param name="versionStrs">バージョン文字列。</param>
        /// <param name="referenceFactory">参照するバージョン文字列を取得するデリゲート。</param>
        /// <returns></returns>
        public static IEnumerable<string> Replace(IEnumerable<string> versionStrs, Func<string> referenceFactory)
        {
            // 'R' の参照先となるバージョン値配列。
            // 実際に 'R' が使われるまで referenceFactory を呼び出さないように遅延している。
            var refVersion = new Lazy<string[]>(() => GetVersionNumber(referenceFactory()));

            // versionSource 毎に 'R' の置換を行う
            // 上記の refVersion をキャプチャしている
            string RepaceImpl(string versionStr)
            {
                // 前後の' 'を除いて、パースしやすいように'||'を付加(後で取り除く)
                var wkVerStr = versionStr.Trim(new char[] { ' ' }) + "||";

                var matches = versRegex.Matches(wkVerStr);

                // 1つもmatchしていない、または文字列の最後までマッチしなかったか
                if (matches.Count == 0 || matches[matches.Count - 1].Index + matches[matches.Count - 1].Length < wkVerStr.Length)
                {
                    throw new FormatException("Invalid version format.");
                }

                // バージョンを構成する数値の処理
                string AddVerNum(Group group, int refVerIdx, ref int lstartIdx)
                {
                    var isReplace = group.Value.ToUpperInvariant() == "R";
                    var result =
                        isReplace && refVersion.Value[refVerIdx] == null ?
                            "" :    // 参照先が空の場合は空のままにしておく
                            wkVerStr.Substring(lstartIdx, group.Index - lstartIdx) +
                                (isReplace ? refVersion.Value[refVerIdx] : group.Value);

                    lstartIdx = group.Index + group.Length;
                    return result;
                }

                var sb = new StringBuilder();
                var startIdx = 0;
                foreach (Match match in matches)
                {
                    // バージョンの置換を "4.3.2", "- 4.99.99", "*.*.*" の 3つのグループ分行う
                    for (var i = 1; i < 10; i += 3)
                    {
                        if (match.Groups[i + 0].Success)    // メジャー
                        {
                            sb.Append(AddVerNum(match.Groups[i + 0], 0, ref startIdx));
                            if (match.Groups[i + 1].Success)    // マイナー
                            {
                                sb.Append(AddVerNum(match.Groups[i + 1], 1, ref startIdx));
                                if (match.Groups[i + 2].Success)    // マイクロ
                                {
                                    sb.Append(AddVerNum(match.Groups[i + 2], 2, ref startIdx));
                                }
                            }
                        }
                    }

                    sb.Append(wkVerStr.Substring(startIdx, match.Index + match.Length - startIdx));
                    startIdx = match.Index + match.Length;
                }

                // 末尾の'||'を除いた分を文字列化
                return sb.ToString(0, sb.Length - 2);
            }

            return versionStrs.Select(RepaceImpl);
        }

        /// <summary>
        /// バージョン文字列を分解して、メジャー、マイナー、マイクロの文字列配列を取得します。
        /// </summary>
        /// <param name="versionStr"></param>
        /// <returns></returns>
        static string[] GetVersionNumber(string versionStr)
        {
            var match = relVerRegex.Match(versionStr);
            if (!match.Success)
            {
                throw new FormatException("Invalid version format.");
            }

            var result = new string [] { null, null, null };
            result[0] = match.Groups[1].Value;
            if (match.Groups[2].Success)
            {
                result[1] = match.Groups[2].Value;
                if (match.Groups[3].Success)
                {
                    result[2] = match.Groups[3].Value;
                }
            }

            return result;
        }
    }
}
