﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.Parser
{
    internal class RelatedZarfParser
    {
        #region 互換・依存部分のパース
        /// <summary>
        /// 互換または依存をパースします。
        /// </summary>
        /// <param name="source">互換または依存のリスト</param>
        /// <returns>パース結果</returns>
        internal ICollection<RelatedZarfInfo> Parse(List<object> source)
        {
            try
            {
                var relatedZarf = from zarfInfo in source.ConvertToDictionaryList()
                                  select this.Parse(zarfInfo);

                return relatedZarf.ToList();
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Invalid compatibility or dependency.");
                throw;
            }
        }

        /// <summary>
        /// 互換または依存をパースします。
        /// </summary>
        /// <param name="source">データソース</param>
        /// <returns>パース結果</returns>
        internal RelatedZarfInfo Parse(Dictionary<string, object> source)
        {
            var relatedZarf = new RelatedZarfInfo();
            try
            {
                foreach (var key in source.Keys)
                {
                    try
                    {
                        switch (key)
                        {
                            case RelatedZarfElement.Target:
                                relatedZarf.Target = (string)source[key];
                                break;
                            case RelatedZarfElement.Versions:
                                relatedZarf.VersionSource = ((List<object>)source[key]).ConvertToStringList();
                                break;
                            default:
                                throw new FormatException("Unknown key was specified.");
                        }
                    }
                    catch (Exception)
                    {
                        Console.Error.WriteLine("ERROR: Error has occured at '{0}'. ", key);
                        throw;
                    }
                }
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to parse compatibility or dependency information.");
                throw;
            }

            string error;
            if (!relatedZarf.Validate(out error))
            {
                Console.Error.WriteLine(error);

                throw new FormatException("Required items have not been completed.");
            }

            return relatedZarf;
        }
        #endregion

        #region 依存・互換先の読み込み
        /// <summary>
        /// 互換または依存先の Zarf 定義の情報を読み込みます。
        /// </summary>
        /// <param name="source">互換または依存先のリスト</param>
        /// <param name="rootPath">SDK のルートパス</param>
        /// <returns>パース結果</returns>
        internal IEnumerable<RelatedZarfInfo> Read(IEnumerable<RelatedZarfInfo> source, string rootPath, string contentsLocale)
        {
            var zarfs = from zarf in source
                        select this.Read(zarf, rootPath, contentsLocale);

            return zarfs.ToList();
        }

        /// <summary>
        /// 互換または依存先の Zarf 定義の情報を読み込みます。
        /// </summary>
        /// <param name="source">データソース</param>
        /// <param name="rootPath">SDK のルートパス</param>
        /// <returns>パース結果</returns>
        internal RelatedZarfInfo Read(RelatedZarfInfo zarf, string rootPath, string contentsLocale)
        {
            try
            {
                ZarfDefinition relatedZarf;

                var zarfParser = new ZarfDefinitionParser(rootPath);
                var yamlPath = Path.Combine(rootPath, zarf.Target);
                if (Path.GetExtension(zarf.Target) == ".yml")
                {
                    var yamlData = ZarfYamlReader.Read(yamlPath);
                    relatedZarf = zarfParser.Parse(yamlData["Zarf"] as Dictionary<string, object>);

                }
                else
                {
                    relatedZarf = zarfParser.Parse(zarf.Target);
                }

                zarf.Family = new FamilyInfo()
                {
                    Key = relatedZarf.Family,
                    Department = relatedZarf.Department,
                };

                zarf.Versions = new List<TargetVersionSpecifier>();

                var versionReader = new VersionData.VersionDataReader();

                var versionStrs =
                    zarf.VersionSource != null ?
                        // バージョンを直接指定している場合
                        from versionSource in zarf.VersionSource
                        select versionReader.Read(versionSource, rootPath, false):

                        // 指定した Zarf 定義に書かれたバージョンを使う場合
                        new string[] { "R.R.*" }; // マイクロバージョンアップは互換があるとみなす

                var repVersionStrs =
                    VersionData.VersionReplace.Replace(
                        versionStrs,
                        () => versionReader.Read(relatedZarf.VersionFile, rootPath, false));

                var versions =
                    from version in repVersionStrs
                    select new TargetVersionSpecifier()
                    {
                        Version = version,
                        ContentsLocale = contentsLocale
                    };

                zarf.Versions = versions.ToList();

                return zarf;
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Failed to read '{0}'.", zarf.Target);
                throw;
            }
        }
        #endregion

        private static class RelatedZarfElement
        {
            public const string Target = "Target";
            public const string Versions = "Versions";
        }
    }
}
