﻿using Nintendo.Zarf.v1;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.ZarfDefinitionData;

namespace ZarfCreator.Parser
{
    internal class InstructionParser
    {
        private readonly InstructionCommandParser commandParser;

        public InstructionParser()
        {
            this.commandParser = new InstructionCommandParser();
        }

        /// <summary>
        /// セットアップに関するデータのパースを行います
        /// </summary>
        /// <param name="source">データソース</param>
        /// <returns>パース結果</returns>
        internal InstructionInfo Parse(Dictionary<string, object> source)
        {
            var instructions = new InstructionInfo();

            foreach (var key in source.Keys)
            {
                try
                {
                    switch (key)
                    {
                        case InstructionElement.Install:
                            try
                            {
                                instructions.Install = this.commandParser.Parse(((List<object>)source[key]));
                            }
                            catch (Exception)
                            {
                                Console.Error.WriteLine("ERROR: Failed to parse install commands.");
                                throw;
                            }
                            break;
                        case InstructionElement.UnInstall:
                            try
                            {
                                instructions.Uninstall = this.commandParser.Parse(((List<object>)source[key]));
                            }
                            catch (Exception)
                            {
                                Console.Error.WriteLine("ERROR: Failed to parse uninstall commands.");
                                throw;
                            }
                            break;
                        case InstructionElement.Prerequisites:
                            try
                            {
                                instructions.Prerequisites = this.commandParser.Parse(((List<object>)source[key]));
                            }
                            catch (Exception)
                            {
                                Console.Error.WriteLine("ERROR: Failed to parse prerequisites commands.");
                                throw;
                            }
                            break;
                        case InstructionElement.PostInstall:
                            try
                            {
                                instructions.PostInstall = this.commandParser.Parse(((List<object>)source[key]), true);
                            }
                            catch (Exception)
                            {
                                Console.Error.WriteLine("ERROR: Failed to parse post install commands.");
                                throw;
                            }
                            break;
                        case InstructionElement.RollBack:
                            instructions.RollBackType = GetRollbackType((string)source[key]);
                            break;
                        case InstructionElement.InstallDirectory:
                            instructions.InstallDirectory = (string)source[key];
                            break;
                        case InstructionElement.RequireBinDir:
                            instructions.RequireBinDir = Convert.ToBoolean((string)source[key]);
                            break;
                        default:
                            throw new FormatException("Unknown key was specified.");
                    }
                }
                catch (Exception)
                {
                    Console.Error.WriteLine("ERROR: Error has occured at '{0}'. ", key);
                    throw;
                }
            }

            string error;
            if (!instructions.Validate(out error))
            {
                Console.Error.WriteLine(error);

                throw new FormatException("Required items have not been completed.");
            }
            return instructions;
        }

        private Install.RollbackAction? GetRollbackType(string type)
        {
            switch (type)
            {
                case InstructionElement.RollbackType.TryDelete:
                    return Install.RollbackAction.TryDelete;
                case InstructionElement.RollbackType.TryUninstall:
                    return Install.RollbackAction.TryUninstall;

                default:
                    Console.Error.WriteLine("ERROR: Cannot specify '{0}' to the 'Rollback' field.", type);
                    throw new FormatException("Invalid parameter in the 'Rollback' field.");
            }
        }

        private static class InstructionElement
        {
            public const string Install = "Install";
            public const string UnInstall = "Uninstall";
            public const string Prerequisites = "Prerequisites";
            public const string PostInstall = "PostInstall";
            public const string RollBack = "RollBack";
            public const string InstallDirectory = "InstallDirectory";
            public const string RequireBinDir = "LogicRequiresContentsFromBinFolder";

            public static class RollbackType
            {
                public const string TryDelete = "tryDelete";
                public const string TryUninstall = "tryUninstall";
            }
        }
    }
}
