﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Nintendo.ToolFoundation.CommandLine;
using VsSolutionBuilderNinjaExecutor;

namespace VsSolutionBuilder
{
    public class Program
    {
        /// <summary>
        /// メイン関数
        /// </summary>
        /// <param name="args">プログラム引数</param>
        public static void Main(string[] args)
        {
            IEnumerable<string> slns;
            var option = ParseArgs(args);

            PrintBuildSetting(option);

            if (option.FilePath != null)
            {
                slns = option.FilePath;
            }
            else
            {
                slns = from path in Directory.EnumerateFiles(option.DirectoryPath, "*.sln",
                            SearchOption.AllDirectories)
                       where option.ExcludeName != null
                            ? !option.ExcludeName.Any(
                                excludeName => Path.GetFileName(path).Contains(excludeName))
                            : true                  // exclude name で絞り込み
                       where option.Spec != null
                            ? option.Spec.Any(spec => path.Contains(spec))
                            : true                  // spec で絞り込み
                       where option.VsVersion != null
                            ? option.VsVersion.Any(version => path.Contains("vs" + version))
                            : true                  // VS version で絞り込み
                       select path;
            }

            try
            {
                // Ninjaを用いて、複数のプロジェクトをグループごとに並列にビルドします
                var buildRuleMaker = new BuildRuleMaker(
                    option.DirectoryPath,
                    slns.ToList(),
                    option.Configuration,
                    option.Platform,
                    new List<string>(option.Target),
                    option.IsCiMode,
                    option.MaxParallelNum);

                buildRuleMaker.Run();
            }
            catch (Exception e)
            {
                BuildRuleMaker.PrintException(e);

                Environment.Exit(1);
            }
        }

        /// <summary>
        /// コマンドラインオプションをコンソールに出力する。
        /// </summary>
        /// <param name="option">コマンドラインオプション</param>
        private static void PrintBuildSetting(CommandLineOption option)
        {
            Console.WriteLine("Build settings");
            Console.WriteLine("----------------------------------------------------");
            Console.WriteLine("File          : {0}", string.Join(",", option.FilePath != null
                ? option.FilePath : new List<string>()));
            Console.WriteLine("Path          : {0}", option.DirectoryPath);
            Console.WriteLine("Spec          : {0}", string.Join(",", option.Spec != null
                ? option.Spec : new List<string>()));
            Console.WriteLine("VS Version    : {0}", string.Join(",", option.VsVersion != null
                ? option.VsVersion : new List<string>()));
            Console.WriteLine("Exclude Name  : {0}", string.Join(",", option.ExcludeName != null
                ? option.ExcludeName : new List<string>()));
            Console.WriteLine("Target        : {0}", string.Join(",", option.Target != null
                ? option.Target : new List<string>()));
            Console.WriteLine("Configuration : {0}", string.Join(",", option.Configuration != null
                ? option.Configuration : new List<string>()));
            Console.WriteLine("Platform      : {0}", string.Join(",", option.Platform != null
                ? option.Platform : new List<string>()));
            Console.WriteLine("CI mode       : {0}", option.IsCiMode);
            Console.WriteLine("Parallel num  : {0}", option.MaxParallelNum);
            Console.WriteLine("----------------------------------------------------");
        }

        /// <summary>
        /// プログラム引数をチェックする。
        /// </summary>
        /// <param name="args">プログラム引数</param>
        /// <returns></returns>
        private static CommandLineOption ParseArgs(string[] args)
        {
            var options = new CommandLineOption();
            var actionCommand = ActionCommand.CreateRootCommand();
            options.RegiserWith(actionCommand);
            switch (CommandLine.ParseArgs(args, actionCommand))
            {
                case ParseResultType.Success:
                    break;
                case ParseResultType.Failure:
                case ParseResultType.Help:
                case ParseResultType.Version:
                default:
                    Environment.Exit(1);
                    break;
            }

            if (!((options.FilePath != null) ^ (options.DirectoryPath != null)))
            {
                throw new ArgumentException("'path' and 'file' are either-or options");
            }

            if ((options.DirectoryPath != null) && !Directory.Exists(options.DirectoryPath))
            {
                throw new DirectoryNotFoundException(
                    string.Format($"Directory '{options.DirectoryPath}' not found"));
            }

            if ((options.FilePath != null))
            {
                var notFoundPath = options.FilePath.Where(path => !File.Exists(path));

                if (notFoundPath.Count() > 0)
                {
                    var notFoundPathList = notFoundPath.Aggregate(string.Empty,
                        (error, element) => error += $"File '{element}' not found\n");
                    throw new FileNotFoundException(notFoundPathList);
                }
            }

            if (options.VsVersion == null)
            {
                options.VsVersion = new List<string>();
                options.VsVersion.Add(BuildRuleMaker.MsbuildToolsetVersion2015);
                options.VsVersion.Add(BuildRuleMaker.MsbuildToolsetVersion2017);
            }
            return options;
        }
    }
}
