﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace UpgradeSigloProject
{
    class Program
    {
        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                Console.Error.WriteLine("Argument Error: UpgradeSigloProject.exe <projectFile>");
                return;
            }

            var projectFile = args[0];
            if (!File.Exists(projectFile))
            {
                Console.Error.WriteLine("Error: Project '{0}' is not found.", projectFile);
                return;
            }

            var extension = Path.GetExtension(projectFile);
            var tmpFile = string.Concat(projectFile, ".tmp");
            bool isConverted = false;

            switch (extension)
            {
                case ".csproj":
                    isConverted = ConvertCsProject(projectFile, tmpFile);
                    if (isConverted)
                    {
                        DeleteOldCsLocalTargets(Path.GetDirectoryName(projectFile));
                        CopyCsLocalProps(Path.GetDirectoryName(projectFile));
                    }
                    break;
                default:
                    Console.Out.WriteLine("'{0}' is not converted.", projectFile);
                    break;
            }

            if (isConverted)
            {
                try
                {
                    File.Copy(tmpFile, projectFile, true);
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("Error: Failed to overwrite '{0}'.", projectFile);
                    Console.Error.WriteLine(e.Message);
                }
                File.Delete(tmpFile);
            }
        }

        // 変換に必要な文字列定義
        private const string MSCSHARPTARGETS_IMPORT = "<Import Project=\"$(MSBuildToolsPath)\\Microsoft.CSharp.targets\" />\r\n";
        private const string TAB = "  ";

        private const string TOOL_OLD_CSLOCALTARGETS = "SigloToolCsProjectSettings.local.targets";
        private const string TEST_OLD_CSLOCALTARGETS = "SigloToolTestCsProjectSettings.local.targets";
        private const string TOOL_CSLOCALPROPS = "SigloToolCsProjectSettings.local.props";

        private static string FormatImport(string file)
        {
            return $"<Import Project=\"{file}\" Condition=\"exists('{file}')\" />\r\n";
        }

        private static string FormatImportOnly(string file)
        {
            return $"<Import Project=\"{file}\" />\r\n";
        }

        private static string FormatNoneInclude(string file)
        {
            return $"<None Include=\"{file}\" />\r\n";
        }

        private static bool ConvertCsProject(string project, string destination)
        {
            using (var sr = new StreamReader(project, true))
            {
                var text = sr.ReadToEnd();

                // 新しい .local.props を使用していれば変換済みとして無視
                if (text.Contains(TOOL_CSLOCALPROPS))
                {
                    Console.Out.WriteLine("'{0}' is already converted.", project);
                    return false;
                }

                using (var sw = new StreamWriter(destination, false, new UTF8Encoding(true)))
                {
                    // 既存の .local.targets のインポートを削除
                    text = text.Replace(TAB + FormatImport(TOOL_OLD_CSLOCALTARGETS), "");
                    text = text.Replace(TAB + FormatImportOnly(TOOL_OLD_CSLOCALTARGETS), "");
                    text = text.Replace(TAB + FormatImport(TEST_OLD_CSLOCALTARGETS), "");
                    text = text.Replace(TAB + FormatImportOnly(TEST_OLD_CSLOCALTARGETS), "");

                    // Project の先頭へのインポートの挿入
                    var projectTagRegex = new Regex("\n<Project .+\n");
                    text = projectTagRegex.Replace(text, match => match.Value
                        + TAB + FormatImport(TOOL_CSLOCALPROPS)
                        + TAB + FormatImport("$(SigloCommonSettingsPropertySheet)"),
                        1);

                    // PropertyGroup と ItemGroup の間へのインポートの挿入
                    text = text.Replace("</PropertyGroup>\r\n" + TAB + "<ItemGroup>\r\n",
                        "</PropertyGroup>\r\n" + TAB + FormatImport("$(SigloProjectSettingsPropertySheet)") + TAB + "<ItemGroup>\r\n");

                    // Item の置換
                    text = text.Replace(FormatNoneInclude(TOOL_OLD_CSLOCALTARGETS), FormatNoneInclude(TOOL_CSLOCALPROPS));
                    text = text.Replace(FormatNoneInclude(TEST_OLD_CSLOCALTARGETS), FormatNoneInclude(TOOL_CSLOCALPROPS));

                    // Microsoft.CSharp.targets のインポート前後の置換
                    text = text.Replace(MSCSHARPTARGETS_IMPORT, MSCSHARPTARGETS_IMPORT + TAB + FormatImport("$(SigloTargetsPropertySheet)"));

                    sw.Write(text);
                }
            }

            Console.Out.WriteLine("Convert '{0}'.", project);
            return true;
        }

        private static void DeleteOldCsLocalTargets(string directory)
        {
            File.Delete(Path.Combine(directory, TOOL_OLD_CSLOCALTARGETS));
            File.Delete(Path.Combine(directory, TEST_OLD_CSLOCALTARGETS));
        }

        private static void CopyCsLocalProps(string directory)
        {
            var asm = Assembly.GetExecutingAssembly();
            using (var res = asm.GetManifestResourceStream("UpgradeSigloProject.Resources.SigloToolCsProjectSettings.local.props"))
            using (var dst = File.Create(Path.Combine(directory, TOOL_CSLOCALPROPS)))
            {
                res.CopyTo(dst);
            }
        }
    }
}
