﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.Linq;
    using System.Text.RegularExpressions;
    using System.Xml;
    using Microsoft.Build.Execution;
    using Executer.InsertionRule;

    /// <summary>
    /// MSTest に関する機能を提供します。
    /// </summary>
    internal static class MsTest
    {
        /// <summary>
        /// VSTest.Console.exe の絶対パスを返します。
        /// </summary>
        internal static string VsTestConsolePath
        {
            get
            {
                return VisualStudio.VsTestConsolePath;
            }
        }

        /// <summary>
        /// サポート対象のプロジェクトインスタンスか否かを表す値を返します。
        /// </summary>
        /// <returns>サポート対象のプロジェクトインスタンスか否かを表す値です。</returns>
        internal static bool Supports(ProjectInstance project)
        {
             string[] references = project.Items
                .Where(x => x.ItemType == "Reference")
                .Select(x => x.ToString()).ToArray();

            if (references.Contains(
                    "Microsoft.VisualStudio.QualityTools.UnitTestFramework"))
            {
                return true;
            }

            if (references.Any(
                    x => x.StartsWith(
                        "Microsoft.VisualStudio.TestPlatform.TestFramework")))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// MSTest に渡す引数をコマンドライン引数に整形します。
        /// </summary>
        /// <param name="targetPath">テスト対象の絶対パスです。</param>
        /// <param name="parameter">MSTest の制御パラメータです。</param>
        /// <param name="runSettingsPath">.runsettings ファイルの絶対パスです。</param>
        /// <returns></returns>
        internal static string GetOption(
            string targetPath,
            string parameter,
            string runSettingsPath)
        {
            var options = new List<string>();

            options.Add("\"" + targetPath + "\"");

            if (!string.IsNullOrWhiteSpace(parameter))
            {
                options.Add(parameter);
            }

            options.Add("/logger:trx");

            options.Add("/settings:\"" + runSettingsPath + "\"");

            return string.Join(" ", options);
        }

        /// <summary>
        /// レポートファイルのワイルドカード表現を返します。
        /// </summary>
        /// <param name="resultDirectoryPath">レポートファイルの保存先となるディレクトリの絶対パスです。</param>
        /// <returns>レポートファイルのワイルドカード表現です。</returns>
        internal static string GetReportPath(string resultDirectoryPath)
        {
            return resultDirectoryPath + @"\*" + MSTestXmlRule.Extention;
        }

        /// <summary>
        /// .runsettings ファイルの内容を生成します。
        /// </summary>
        /// <param name="runSettings">.runsettings ファイルの内容です。</param>
        /// <param name="runSettingsPath">.runsettings ファイルの絶対パスです。</param>
        /// <param name="resultDirectoryPath">MSTest の実行結果の保存先となるディレクトリの絶対パスです。</param>
        /// <param name="testContextPath">テスト結果ファイルの絶対パスです。</param>
        /// <param name="platform">テスト対象のプラットフォームです。</param>
        /// <param name="option">テスト対象に渡す引数です。</param>
        internal static void GenerateRunSettings(
            out string runSettings,
            out string runSettingsPath,
            out string resultDirectoryPath,
            string testContextPath,
            string platform,
            string option)
        {
            var fileName = Utility.RemoveExtension(
                testContextPath, ExtensionDefinition.TestResult);

            var xml = new XmlDocument();

            xml.AppendChild(xml.CreateXmlDeclaration("1.0", "utf-8", null));

            var rootNode = xml.CreateElement("RunSettings");

            var configurationNode = xml.CreateElement("RunConfiguration");

            var directoryNode = xml.CreateElement("ResultsDirectory");

            resultDirectoryPath = fileName + "_VSTest";

            directoryNode.AppendChild(xml.CreateTextNode(resultDirectoryPath));

            configurationNode.AppendChild(directoryNode);

            if (new[] { PlatformDefinition.X86,
                        PlatformDefinition.X64 }.Contains(platform))
            {
                var platformNode = xml.CreateElement("TargetPlatform");

                platformNode.AppendChild(xml.CreateTextNode(platform));

                configurationNode.AppendChild(platformNode);
            }

            rootNode.AppendChild(configurationNode);

            IDictionary<string, string> parameters = ParseOption(option);

            if (parameters.Count > 0)
            {
                var paramsNode = xml.CreateElement("TestRunParameters");

                foreach (var pair in parameters)
                {
                    var paramNode = xml.CreateElement("Parameter");

                    paramNode.SetAttribute("name", pair.Key);

                    paramNode.SetAttribute("value", pair.Value);

                    paramsNode.AppendChild(paramNode);
                }

                rootNode.AppendChild(paramsNode);
            }

            var msTestNode = xml.CreateElement("MSTest");

            var deploymentNode = xml.CreateElement("DeploymentEnabled");

            deploymentNode.AppendChild(xml.CreateTextNode("False"));

            msTestNode.AppendChild(deploymentNode);

            rootNode.AppendChild(msTestNode);

            xml.AppendChild(rootNode);

            runSettings = xml.OuterXml;

            runSettingsPath = fileName + ".runsettings";
        }

        private static IDictionary<string, string> ParseOption(string option)
        {
            var parameters = new Dictionary<string, string>();

            if (string.IsNullOrEmpty(option))
            {
                return parameters;
            }

            var anyCharPattern = new Regex(@"\S");

            var keyPattern = new Regex(@"^/(\w+):");

            var valuePattern = new Regex(@"^\S*");

            while (true)
            {
                Match match = anyCharPattern.Match(option);

                if (!match.Success)
                {
                    break;
                }

                if (parameters.Count > 0 && match.Index == 0)
                {
                    throw new TestRunnerException(string.Format(
                        "Space is expected but found: {0}", option));
                }

                option = option.Substring(match.Index);

                match = keyPattern.Match(option);

                if (!match.Success)
                {
                    throw new TestRunnerException(string.Format(
                        "Key is expected but found: {0}", option));
                }

                var key = match.Groups[1].Value;

                option = option.Substring(match.Length);

                if (option.Length == 0)
                {
                    parameters[key] = string.Empty;

                    break;
                }

                if (option[0] != '"')
                {
                    match = valuePattern.Match(option);

                    parameters[key] = match.Value;

                    option = option.Substring(match.Length);

                    continue;
                }

                int index = 1;

                while (true)
                {
                    index = option.IndexOf('"', index);

                    if (index == -1)
                    {
                        throw new TestRunnerException(string.Format(
                            "Invalid string: {0}", option));
                    }

                    if (index + 1 < option.Length && option[index + 1] == '"')
                    {
                        index += 2;

                        continue;
                    }

                    break;
                }

                var value = option.Substring(1, index - 1);

                parameters[key] = value.Replace("\"\"", "\"");

                option = option.Substring(index + 1);
            }

            return parameters;
        }
    }
}
