﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.Rule
{
    using System.Collections.Generic;
    using Executer;

    /// <summary>
    /// PowerShell スクリプト用のインストラクション生成ルールです。
    /// </summary>
    internal sealed class PowerShellScriptRule : InstructionGenerator.IRule
    {
        /// <summary>
        /// サポート対象のテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
        public bool Supports(TestContext context)
        {
            return context.TargetPath.EndsWith(".ps1");
        }

        /// <summary>
        /// ルール適用時にターゲットファイルを必要とするかどうかを示す値を取得します。
        /// </summary>
        /// <returns>ルール適用時にターゲットファイルを必要とするかどうかを示す値です。</returns>
        public bool NeedsTargetFile()
        {
            return false;
        }

        /// <summary>
        /// テストコンテキストにインストラクション生成ルールを適用します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>生成されたテストインストラクションです。</returns>
        public InstructionGenerator.Instruction Generate(TestContext context)
        {
            var inst = new InstructionGenerator.Instruction();

            inst.Command = "powershell.exe";

            var option = new List<string>();

            if (!string.IsNullOrEmpty(context.Parameter))
            {
                option.Add(context.Parameter);
            }

            option.Add("-NoProfile");

            option.Add("-ExecutionPolicy RemoteSigned");

            var command = new List<string>();

            command.Add(string.Format("& '{0}'", context.TargetPath));

            if (!string.IsNullOrEmpty(context.Option))
            {
                command.Add(context.Option);
            }

            option.Add(string.Format("-Command \"{0}\"", string.Format(
                "try {{ {0} }} catch {{ {1} }}",
                string.Format(
                    "{0}; exit ${{LastExitCode}}", string.Join(" ", command)),
                string.Format(
                    "Write-Host ({0}); exit 1",
                    "'[PowerShell] Error : ' + ${_}.Exception.Message"))));

            inst.Option = string.Join(" ", option);

            inst.Platform = string.IsNullOrEmpty(context.Platform)
                ? BuildSystem.DefaultPlatform : context.Platform;

            inst.BuildType = string.IsNullOrEmpty(context.BuildType)
                ? BuildSystem.DefaultBuildType : context.BuildType;

            inst.TargetPath = context.TargetPath;

            inst.TargetProjectPath = context.TargetProjectPath;

            inst.ReportPath = context.ReportPath;

            return inst;
        }
    }
}
