﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.GenerationRule
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using Microsoft.Build.Evaluation;
    using Microsoft.Build.Execution;
    using Executer;
    using static InstructionGenerator;

    /// <summary>
    /// csproj ファイル用のインストラクション生成ルールです。
    /// </summary>
    internal sealed class CsProjectFileRule : IRule
    {
        /// <summary>
        /// サポート対象のテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
        public bool Supports(TestContext context)
        {
            return context.TargetPath.EndsWith(".csproj");
        }

        /// <summary>
        /// ルール適用時にターゲットファイルを必要とするかどうかを示す値を取得します。
        /// </summary>
        /// <returns>ルール適用時にターゲットファイルを必要とするかどうかを示す値です。</returns>
        public bool NeedsTargetFile()
        {
            return true;
        }

        /// <summary>
        /// テストコンテキストにインストラクション生成ルールを適用します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>生成されたテストインストラクションです。</returns>
        public Instruction Generate(TestContext context)
        {
            TestFrameworkName framework = TestFrameworkName.None;

            Instruction instruction =
                string.IsNullOrEmpty(context.TargetEpiPath)
                    ? this.GenerateFromCsProjectFile(out framework, context)
                    : this.GenerateFromEpiFile(out framework, context);

            instruction.Command = instruction.TargetPath;

            instruction.Option = context.Option;

            instruction.TargetProjectPath = context.TargetPath;

            instruction.ReportPath = context.ReportPath;

            switch (framework)
            {
                case TestFrameworkName.None:
                    break;

                case TestFrameworkName.MSTest:
                    UpdateInstructionForVsTest(instruction, context);
                    break;

                case TestFrameworkName.xUnit:
                    UpdateInstructionForXunitNet(instruction, context);
                    break;
            }

            return instruction;
        }

        private static string GetConfiguration(string buildType)
        {
            if (string.IsNullOrEmpty(buildType))
            {
                buildType = BuildSystem.DefaultBuildType;
            }

            switch (buildType)
            {
                case BuildTypeDefinition.Debug:
                    return BuildTypeDefinition.Debug;

                case BuildTypeDefinition.Develop:
                case BuildTypeDefinition.Release:
                    return BuildTypeDefinition.Release;

                default:
                    throw new TestRunnerException(
                        $"BuildType '{buildType}' is not supported");
            }
        }

        private static void ParseProject(
            out TestFrameworkName framework,
            Instruction instruction,
            TestContext context,
            Tuple<string, string> toolsVersion)
        {
            var platform = context.Platform;

            var buildType = context.BuildType;

            var properties = new Dictionary<string, string>()
            {
                { "Configuration", GetConfiguration(buildType) }
            };

            if (new[] { PlatformDefinition.X86,
                        PlatformDefinition.X64 }.Contains(platform))
            {
                properties["Platform"] = platform;
            }

            var project = new ProjectInstance(
                context.TargetPath,
                properties,
                toolsVersion.Item1,
                toolsVersion.Item2,
                ProjectCollection.GlobalProjectCollection);

            instruction.Platform = project
                .Properties.First(x => x.Name == "Platform").EvaluatedValue;

            instruction.BuildType = properties["Configuration"];

            instruction.TargetPath = Path.GetFullPath(project
                .Properties.First(x => x.Name == "TargetPath").EvaluatedValue);

            if (MsTest.Supports(project))
            {
                framework = TestFrameworkName.MSTest;
            }
            else if (XunitNet.Supports(project))
            {
                framework = TestFrameworkName.xUnit;
            }
            else
            {
                framework = TestFrameworkName.None;
            }
        }

        private static void UpdateInstructionForVsTest(
            Instruction instruction, TestContext context)
        {
            var vsTestPath = string.Empty;

            try
            {
                vsTestPath = MsTest.VsTestConsolePath;
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            var runSettings = string.Empty;

            var runSettingsPath = string.Empty;

            var resultDirectoryPath = string.Empty;

            try
            {
                MsTest.GenerateRunSettings(
                    out runSettings,
                    out runSettingsPath,
                    out resultDirectoryPath,
                    context.Path,
                    context.Platform,
                    context.Option);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.ERROR;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            instruction.Command = vsTestPath;

            instruction.Option = MsTest.GetOption(
                instruction.TargetPath, context.Parameter, runSettingsPath);

            instruction.RunSettings = runSettings;

            if (string.IsNullOrEmpty(instruction.ReportPath))
            {
                instruction.ReportPath =
                    MsTest.GetReportPath(resultDirectoryPath);
            }

            instruction.RunSettingsPath = runSettingsPath;
        }

        private static void UpdateInstructionForXunitNet(
            Instruction instruction, TestContext context)
        {
            var xunitPath = string.Empty;

            try
            {
                xunitPath = XunitNet.XunitConsolePath;
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context, ex);
            }

            if (string.IsNullOrEmpty(instruction.ReportPath))
            {
                instruction.ReportPath = XunitNet.GetReportPath(context.Path);
            }

            instruction.Command = xunitPath;

            instruction.Option = XunitNet.GetOption(
                instruction.TargetPath, instruction.ReportPath,
                context.Parameter);
        }

        private Instruction GenerateFromCsProjectFile(
            out TestFrameworkName framework, TestContext context)
        {
            Tuple<string, string> toolsVersion = null;

            try
            {
                toolsVersion = new Tuple<string, string>(
                    VisualStudio.ToolsVersion,
                    VisualStudio.SubToolsetVersion);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.NO_TOOL;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context);
            }

            var instruction = new Instruction();

            try
            {
                ParseProject(
                    out framework, instruction, context, toolsVersion);
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.WRONG_FILE;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context, ex);
            }

            return instruction;
        }

        private Instruction GenerateFromEpiFile(
            out TestFrameworkName framework, TestContext context)
        {
            var instruction = new Instruction();

            try
            {
                var epiManager = new EpiManager();

                var epi = epiManager.GetEpi(context.TargetEpiPath);

                instruction.TargetPath = epi.ExecutableFiles[0].Path;

                instruction.Platform = epi.Platform;

                instruction.BuildType = epi.BuildType;

                if (epi.TestFrameworks.Length == 0)
                {
                    framework = TestFrameworkName.None;
                }
                else
                {
                    switch (epi.TestFrameworks[0].Name)
                    {
                        case TestFrameworkName.MSTest:
                            framework = TestFrameworkName.MSTest;
                            break;

                        case TestFrameworkName.xUnit:
                            framework = TestFrameworkName.xUnit;
                            break;

                        default:
                            framework = TestFrameworkName.None;
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                context.ResultCode = ResultCode.WRONG_FILE;

                context.ErrorMessage = ex.Message;

                throw new TestContextException(ex.Message, context, ex);
            }

            return instruction;
        }
    }
}
