﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.Executer
{
    using System;
    using System.Diagnostics;
    using System.Runtime.InteropServices;
    using Microsoft.Win32.SafeHandles;

    /// <summary>
    /// プロセスをジョブオブジェクトに関連付けて終了を保証します。
    /// </summary>
    internal sealed class ProcessJob : IDisposable
    {
        private JobHandle jobHandle = null;

        private bool disposed = false;

        /// <summary>
        /// ProcessJob クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="process">プロセスです。</param>
        internal ProcessJob(Process process)
        {
            IntPtr handle = NativeMethods.CreateJobObject(IntPtr.Zero, null);

            if (handle == IntPtr.Zero)
            {
                throw new InvalidOperationException(string.Format(
                    "failed to create a job object. (0x{0})",
                    Marshal.GetLastWin32Error().ToString("X8")));
            }

            this.jobHandle = new JobHandle(handle);

            if (!NativeMethods.AssignProcessToJobObject(
                    this.jobHandle, process.Handle))
            {
                throw new InvalidOperationException(string.Format(
                    "failed to add a process to a job object. (0x{0})",
                    Marshal.GetLastWin32Error().ToString("X8")));
            }
        }

        ~ProcessJob()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// 関連付けられたプロセスがオブジェクトを破棄する際に強制的に終了されるようになります。
        /// </summary>
        internal void SetKillOnJobCloseFlag()
        {
            var info = new NativeMethods.JOBOBJECT_BASIC_LIMIT_INFORMATION
            {
                LimitFlags = 0x2000,
            };

            var extendedInfo =
                new NativeMethods.JOBOBJECT_EXTENDED_LIMIT_INFORMATION
            {
                BasicLimitInformation = info,
            };

            if (!NativeMethods.SetInformationJobObject(
                    this.jobHandle,
                    NativeMethods.JOBOBJECTINFOCLASS.ExtendedLimitInformation,
                    ref extendedInfo,
                    (uint)Marshal.SizeOf<
                        NativeMethods.JOBOBJECT_EXTENDED_LIMIT_INFORMATION>()))
            {
                throw new InvalidOperationException(string.Format(
                    "failed to configure a job object. (0x{0})",
                    Marshal.GetLastWin32Error().ToString("X8")));
            }
        }

        private void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                this.disposed = true;

                this.jobHandle.Dispose();
                this.jobHandle = null;
            }
        }

        private static class NativeMethods
        {
            internal enum JOBOBJECTINFOCLASS
            {
                ExtendedLimitInformation = 9,
            }

            [DllImport("kernel32.dll", SetLastError = true,
                CharSet = CharSet.Unicode)]
            internal static extern IntPtr CreateJobObject(
                IntPtr attributes, string name);

            [DllImport("kernel32.dll", SetLastError = true)]
            internal static extern bool SetInformationJobObject(
                JobHandle jobHandle,
                JOBOBJECTINFOCLASS JobObjectInfoClass,
                [In] ref JOBOBJECT_EXTENDED_LIMIT_INFORMATION jobObjectInfo,
                uint jobObjectInfoLength);

            [DllImport("kernel32.dll", SetLastError = true)]
            internal static extern bool AssignProcessToJobObject(
                JobHandle jobHandle, IntPtr processHandle);

            [DllImport("kernel32.dll", SetLastError = true)]
            internal static extern bool CloseHandle(IntPtr handle);

            [StructLayout(LayoutKind.Sequential)]
            internal struct IO_COUNTERS
            {
                internal ulong ReadOperationCount;
                internal ulong WriteOperationCount;
                internal ulong OtherOperationCount;
                internal ulong ReadTransferCount;
                internal ulong WriteTransferCount;
                internal ulong OtherTransferCount;
            }

            [StructLayout(LayoutKind.Sequential)]
            internal struct JOBOBJECT_BASIC_LIMIT_INFORMATION
            {
                internal long PerProcessUserTimeLimit;
                internal long PerJobUserTimeLimit;
                internal uint LimitFlags;
                internal UIntPtr MinimumWorkingSetSize;
                internal UIntPtr MaximumWorkingSetSize;
                internal uint ActiveProcessLimit;
                internal UIntPtr Affinity;
                internal uint PriorityClass;
                internal uint SchedulingClass;
            }

            [StructLayout(LayoutKind.Sequential)]
            internal struct JOBOBJECT_EXTENDED_LIMIT_INFORMATION
            {
                internal JOBOBJECT_BASIC_LIMIT_INFORMATION BasicLimitInformation;
                internal IO_COUNTERS IoInfo;
                internal UIntPtr ProcessMemoryLimit;
                internal UIntPtr JobMemoryLimit;
                internal UIntPtr PeakProcessMemoryUsed;
                internal UIntPtr PeakJobMemoryUsed;
            }
        }

        private sealed class JobHandle : SafeHandleZeroOrMinusOneIsInvalid
        {
            internal JobHandle(IntPtr handle) : base(true)
            {
                this.SetHandle(handle);
            }

            protected override bool ReleaseHandle()
            {
                return NativeMethods.CloseHandle(this.handle);
            }
        }
    }
}
