﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SystemUpdaterNx.Initialize
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;
    using TargetShell.PluginInterface;

    /// <summary>
    /// InitializeSdev を行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    [ExportMetadata("SupportedHardwareType", "Sdev")]
    public class Sdev :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// InitializeSdev.exe の実行ファイル名
        /// </summary>
        private string initializeSdevExe = "InitializeSdev.exe";
        /// <summary>
        /// カレントに存在しないときに使用するDLLのパス
        /// </summary>
        private const string RunnerToolsPath = @"Tools\CommandLineTools";

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Sdev()
        {
            this.ParallelModeFlag = true;
        }

        /// <summary>
        /// デバイスの情報取得
        /// </summary>
        /// <returns>デバイスの情報リスト</returns>
        public override List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoSdev();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            this.initializeSdevExe = Path.Combine(
                    parameter.NintendoSdkRoot, RunnerToolsPath, this.initializeSdevExe);
            try
            {
                // コマンドの構築
                var runOnTargetCommand = CreateRunOnTargetCommand(parameter,
                            CommonConstants.OutputLogName);
                // Initialize を行う
                var returnCode = TargetShellLibrary.InvokeExe(initializeSdevExe,
                            runOnTargetCommand, parameter.SerialDirectroy,
                            parameter.TargetEntry.SerialNumber,
                            CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// Initialize コマンド構築を行う
        /// </summary>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <param name="logFileName">ログファイル名</param>
        /// <returns>True:成功 False:失敗</returns>
        private string CreateRunOnTargetCommand(CommandParameter parameter,
                    string logFileName)
        {
            var runOnTargetCommandBuilder = new StringBuilder();
            try
            {
                var firmwares = AvailableFirmwareVersions(parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber, logFileName);
                if (firmwares == null || firmwares.Count == 0)
                {
                    throw new TargetShellLibraryException("Not found FirmwareFile");
                }
                switch (parameter.TargetFirmware)
                {
                    case FirmwareType.Prod:
                        var Prod = firmwares.FindAll(entry => entry.Contains("Prod") &&
                                !entry.Contains("ProdWithLog"));
                        if (Prod != null && Prod.Count > 0)
                        {
                            runOnTargetCommandBuilder.AppendFormat(" --target-version \"{0}\"", Prod[0]);
                        }
                        break;
                    case FirmwareType.ProdWithLog:
                        var ProdWithLog = firmwares.FindAll(entry => entry.Contains(
                                "ProdWithLog"));
                        if (ProdWithLog != null && ProdWithLog.Count > 0)
                        {
                            runOnTargetCommandBuilder.AppendFormat(
                                    " --target-version \"{0}\"", ProdWithLog[0]);
                        }
                        break;
                    case FirmwareType.Develop:
                    default:
                        break;
                }
                if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
                {
                    runOnTargetCommandBuilder.AppendFormat(" --target-ip {0}", parameter.TargetEntry.IpAddress);
                }
                if (!string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
                {
                    runOnTargetCommandBuilder.AppendFormat(" --target-serial {0}", parameter.TargetEntry.SerialNumber);
                }
                if (parameter.Verbose)
                {
                    runOnTargetCommandBuilder.AppendFormat(" --verbose");
                }
                // 追加オプションを指定する
                runOnTargetCommandBuilder.AppendFormat(" --keep-targetmanager-alive");
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
            return runOnTargetCommandBuilder.ToString();
        }

        /// <summary>
        /// 利用可能なファームウェアを列挙する
        /// </summary>
        /// <param name="serialDirectroy">ディレクトリ名</param>
        /// <param name="serialNumber">シリアルナンバー</param>
        /// <param name="LogFileName">ログファイル名</param>
        /// <returns>利用可能なファームウェア一覧を返す</returns>
        private List<string> AvailableFirmwareVersions(string serialDirectroy,
                    string serialNumber, string logFileName)
        {
            var firmwareVersions = string.Empty;
            var command = " --display-available-version";
            var success = TargetShellLibrary.InvokeExe(this.initializeSdevExe, command,
                    serialDirectroy, serialNumber,
                    logFileName, out firmwareVersions);
            if (!success)
            {
                return null;
            }
            var firmwareVersionsList = new List<string>();
            using (var stringReader = new StringReader(firmwareVersions))
            {
                var firmwareVersion = string.Empty;
                while ((firmwareVersion = stringReader.ReadLine()) != null)
                {
                    if (firmwareVersion.Contains("[INFO]"))
                    {
                        firmwareVersion = firmwareVersion.Replace("[INFO]  ", string.Empty);
                        firmwareVersionsList.Add(firmwareVersion);
                    }
                }
            }
            return firmwareVersionsList;
        }
    }
}
