﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace PrivateNx.WriteBootConfig
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Security.AccessControl;
    using System.Text;
    using System.Threading;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;
    using TargetShell.PluginInterface;

    /// <summary>
    /// BootConfig書き換えを行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    public class Sdev :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Sdev()
        {
            this.ParallelModeFlag = true;
        }

        /// <summary>
        /// サーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public override List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoSdev();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                // BootConfig書き換え用ファイルの取得
                var bootConfigFile = string.Empty;
                if (parameter.SignedType.Equals(ConstantsSdev.CommandString.SignedString))
                {
                    // 署名ありの場合はダミーファイルを書き込む
                    bootConfigFile = ConstantsSdev.DummyBootConfigFile;
                }
                else
                {
                    bootConfigFile = TargetShellLibrary.ReplaceFirst(ConstantsSdev.BootConfigFile,
                            @"\" + ConstantsSdev.Caret,
                            parameter.TargetEntry.SerialNumber);
                }

                // BootConfig書き換え用ファイルがあるかチェック
                bootConfigFile = Path.Combine(parameter.NintendoSdkRoot,
                        bootConfigFile);
                if (!TargetShellLibrary.IsExistFile(bootConfigFile))
                {
                    throw new TargetShellLibraryException(
                            $"Write BootConfig file not found\nBootConfigFile = {bootConfigFile}");
                }

                // キータイプのコンバート
                if (parameter.KeyType.Equals("Auto"))
                {
                    ConvertKeyType(ref parameter);
                }

                // HostBridgeの更新
                UpdateHostBridge(parameter);

                // デバイスを起動
                StartTarget(parameter);

                // キータイプ別にemmcWriterファイルを取得
                var emmcWriterImageName = string.Empty;
                switch (parameter.KeyType)
                {
                    case ConstantsSdev.CommandString.KeyTypeK1:
                        emmcWriterImageName = ConstantsSdev.EmmcWriterK1;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK2:
                        emmcWriterImageName = ConstantsSdev.EmmcWriterK2;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK3:
                        emmcWriterImageName = ConstantsSdev.EmmcWriterK3;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK5:
                        emmcWriterImageName = ConstantsSdev.EmmcWriterK5;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeM2:
                        emmcWriterImageName = ConstantsSdev.EmmcWriterM2;
                        break;
                    default:
                        throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                                "WriteBootConfig");
                }

                emmcWriterImageName = Path.Combine(parameter.NintendoSdkRoot,
                        emmcWriterImageName);
                var paramsBc1 = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsSdev.ParamsBc1);
                var paramsBc2 = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsSdev.ParamsBc2);
                var paramsBc3 = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsSdev.ParamsBc3);
                var paramsBc4 = Path.Combine(parameter.NintendoSdkRoot,
                        ConstantsSdev.ParamsBc4);

                // BootConfig書き換え
                InvokeRecoveryBoot(parameter, emmcWriterImageName,
                        paramsBc1, bootConfigFile);
                if (parameter.AllBootConfig)
                {
                    InvokeRecoveryBoot(parameter, emmcWriterImageName,
                            paramsBc2, bootConfigFile);
                    InvokeRecoveryBoot(parameter, emmcWriterImageName,
                            paramsBc3, bootConfigFile);
                    InvokeRecoveryBoot(parameter, emmcWriterImageName,
                            paramsBc4, bootConfigFile);
                }

                // デバイスの再起動
                RestartTarget(parameter);

                Interlocked.Increment(ref successDeviceCount);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// 取得したハードウエアバージョンからキータイプを設定する
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool ConvertKeyType(ref CommandParameter parameter)
        {
            var isSuccess = false;
            var hardwareVersion = GetHostBridgeHardwareVersion(parameter);

            // HardwareConfigからKeyTypeに変更する
            var data = new ConstantsSdev.ConvertHardwareConfigToKeyType();
            for (var i = 0; i < data.ConvertHardwareConfigToKeyTypeTable.GetLength(0); i++)
            {
                if (hardwareVersion.ToString().Equals(
                        data.ConvertHardwareConfigToKeyTypeTable[i, 0]))
                {
                    parameter.KeyType = data.ConvertHardwareConfigToKeyTypeTable[i, 1];
                    isSuccess = true;
                    break;
                }
            }

            return isSuccess;
        }

        /// <summary>
        /// HostBridgeの更新
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        private void UpdateHostBridge(CommandParameter parameter)
        {
            // HostBridgeHardwareVersionを取得
            var hostBridgeWardwareVersion = GetHostBridgeHardwareVersion(parameter);

            // HostBridgeのファームウェアバージョンを取得する
            var hostBridgeFirmwareVersion = GetHostBridgeFirmwareVersion(parameter);
            if (hostBridgeFirmwareVersion.Contains(ConstantsSdev.HostBridge.LatestFirmware))
            {
                // ファームウェアバージョンが最新の時はRecoveryFirmwareだけ行って抜ける
                if (hostBridgeWardwareVersion < ConstantsSdev.HostBridge.LatestHardware)
                {
                    if (!UpdateHostBridgeRecoveryFirmware(parameter))
                    {
                        throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                                "UpdateHostBridge");
                    }
                }
                return;
            }

            // HostBridgeの更新
            if (!UpdateHostBridgeByConsole(parameter))
            {
                throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                        "UpdateHostBridge");
            }

            // RecoveryFirmware
            if (hostBridgeWardwareVersion < ConstantsSdev.HostBridge.LatestHardware)
            {
                if (!UpdateHostBridgeRecoveryFirmware(parameter))
                {
                    throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                            "UpdateHostBridge");
                }
            }
        }

        /// <summary>
        /// HostBridgeのハードウェアバージョンを取得
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>HostBridgeのハードウェアバージョン</returns>
        private int GetHostBridgeHardwareVersion(CommandParameter parameter)
        {
            var outputString = string.Empty;
            var hostBridgeHardwareVersion = 0;
            var hostBridgeHardwareVersionString = string.Empty;
            var command = $" get-hostbridge-version --target-ip {parameter.TargetEntry.IpAddress}";
            if (InvokeExe(parameter, ConstantsSdev.ControlTargetPrivate,
                    command, CommonConstants.OutputLogName, out outputString))
            {
                // ハードウェアバージョンが含まれているか
                if (outputString.Contains(ConstantsSdev.HostBridge.HardwareText))
                {
                    var hardwareVersionPos = outputString.IndexOf(
                            ConstantsSdev.HostBridge.HardwareText);
                    hostBridgeHardwareVersionString = outputString.Substring(
                            hardwareVersionPos + ConstantsSdev.HostBridge.HardwareText.Length, 1);
                    if (!int.TryParse(hostBridgeHardwareVersionString,
                            out hostBridgeHardwareVersion))
                    {
                        throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                                "GetHostBridgeHardwareVersion");
                    }
                }
            }
            else
            {
                throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                        "GetHostBridgeHardwareVersion");
            }

            return hostBridgeHardwareVersion;
        }

        /// <summary>
        /// HostBridgeのファームウェアバージョンを取得
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>HostBridgeのファームウェアバージョンを含むGPIOControllerCUI.exe実行のログ</returns>
        private string GetHostBridgeFirmwareVersion(CommandParameter parameter)
        {
            var outputString = string.Empty;
            var command = $" --get_hb_build_version --ip {parameter.TargetEntry.IpAddress}";
            if (!InvokeExe(parameter, ConstantsSdev.GPIOControllerCUI,
                    command, CommonConstants.OutputLogName, out outputString))
            {
                throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                        "GetHostBridgeFirmwareVersion");
            }

            return outputString;
        }

        /// <summary>
        /// HostBridgeの更新
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpdateHostBridgeByConsole(CommandParameter parameter)
        {
            var isSuccess = false;

            var updateImagePath = Path.Combine(parameter.NintendoSdkRoot,
                    ConstantsSdev.HostBridge.Package);

            // ファイル無い場合は抜ける
            if (!TargetShellLibrary.IsExistFile(updateImagePath))
            {
                return isSuccess;
            }

            var command = new StringBuilder();
            command.AppendFormat(" update --target {0}", parameter.TargetEntry.IpAddress);
            command.AppendFormat(" --image {0}", updateImagePath);
            if (parameter.Verbose)
            {
                command.AppendFormat(" --verbose");
            }

            // ターゲット側FWとホストブリッジFWの組み合わせにによって
            // 初回のアップデートが失敗することがあるのでリトライする
            for (var i = 0; i < ConstantsSdev.HostBridge.MaxRetry; i++)
            {
                if (InvokeExe(parameter, ConstantsSdev.UpdateHostBridge,
                        command.ToString(), CommonConstants.OutputLogName))
                {
                    Thread.Sleep(20000);
                    isSuccess = true;
                    break;
                }
            }
            return isSuccess;
        }

        /// <summary>
        /// HostBridgeの更新
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpdateHostBridgeRecoveryFirmware(CommandParameter parameter)
        {
            var isSuccess = false;
            var outputString = string.Empty;
            var command = new StringBuilder();
            command.AppendFormat(" required --target {0}", parameter.TargetEntry.IpAddress);
            if (parameter.Verbose)
            {
                command.AppendFormat(" --verbose");
            }
            InvokeExe(parameter, ConstantsSdev.UpdateHostBridge,
                    command.ToString(), CommonConstants.OutputLogName, out outputString);

            // RecoveryFirmwareを実行する必要があるか
            if (outputString.Contains(ConstantsSdev.HostBridge.CheckStringsRecoveryFirmware))
            {
                command.Clear();
                command.AppendFormat(" update --target {0}", parameter.TargetEntry.IpAddress);
                command.AppendFormat(" --recovery --image {0}", Path.Combine(
                        parameter.NintendoSdkRoot, ConstantsSdev.HostBridge.RecoveryImage));

                if (parameter.Verbose)
                {
                    command.AppendFormat(" --verbose");
                }
                if (InvokeExe(parameter, ConstantsSdev.UpdateHostBridge,
                        command.ToString(), CommonConstants.OutputLogName))
                {
                    isSuccess = true;
                }
            }
            else
            {
                // RecoveryFirmwareを実行する必要がない場合は成功で抜ける
                isSuccess = true;
            }
            return isSuccess;
        }

        /// <summary>
        /// BootConfigの書き換え
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="emmcWriterFile">EmmcWriter***.imgファイル</param>
        /// <param name="paramsBcFile">params-bc-*.binファイル</param>
        /// <param name="bootConfigFile">書き換え用のBootConfigファイル</param>
        private void InvokeRecoveryBoot(CommandParameter parameter,
                string emmcWriterFile, string paramsBcFile, string bootConfigFile)
        {
            var bootConfigImage = string.Empty;

            try
            {
                // BootConfigの書き換え用のイメージを作成
                bootConfigImage = CreateBootConfigImage(parameter,
                        emmcWriterFile, paramsBcFile, bootConfigFile);
                if (string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
                {
                    throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                            "InvokeRecoveryBoot");
                }
                if (string.IsNullOrEmpty(bootConfigImage))
                {
                    throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                            "InvokeRecoveryBoot");
                }

                // RecoveryBootのコマンド生成
                var command = new StringBuilder();
                command.AppendFormat(" --target {0}", parameter.TargetEntry.IpAddress);
                command.AppendFormat(" --image {0}", bootConfigImage);
                if (parameter.Verbose)
                {
                    command.AppendFormat(" --verbose");
                }

                // BootConfig書き換え実行
                if (!InvokeExe(parameter, ConstantsSdev.RecoveryBoot, command.ToString(),
                        CommonConstants.OutputLogName))
                {
                    throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev." +
                            "InvokeRecoveryBoot");
                }

                // 連続でBootConfig書き換えを行う為とBootConfig書き換え後にリセットを行う為に5秒待つ
                // (Write-BootConfig.ps1と同様)
                Thread.Sleep(5000);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }

            // BootConfig書き換え用のイメージファイルを削除する
            if (!string.IsNullOrEmpty(bootConfigImage))
            {
                File.Delete(bootConfigImage);
            }
        }

        /// <summary>
        /// BootConfigの書き換え用のイメージを作成
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="emmcWriterFile">EmmcWriter***.imgファイル</param>
        /// <param name="paramsBcFile">params-bc-*.binファイル</param>
        /// <param name="bootConfigFile">書き換え用のBootConfigファイル</param>
        /// <returns>BootConfig書き換え用イメージ</returns>
        private string CreateBootConfigImage(CommandParameter parameter,
                string emmcWriterFile, string paramsBcFile, string bootConfigFile)
        {
            // 連結したブートコンフィグのファイルネーム
            var createdBootConfigPath = Path.GetTempFileName();
            var createdBootConfigFileStream = new FileStream(
                    createdBootConfigPath, FileMode.Append, FileAccess.Write);
            try
            {
                // アクセス権を付与
                TargetShellLibrary.SetAccessControlCurrentUser(createdBootConfigPath,
                        FileSystemRights.WriteData);

                // EmmcWriter***.imgファイルを読み込み、連結ファイルに書き込む
                ReadAndWriteFile(emmcWriterFile, ref createdBootConfigFileStream);
                // params-bc-*.binファイルを読み込み、連結ファイルに書き込む
                ReadAndWriteFile(paramsBcFile, ref createdBootConfigFileStream);
                // 書き換え用のBootConfigファイルを読み込み、連結ファイルに書き込む
                ReadAndWriteFile(bootConfigFile, ref createdBootConfigFileStream);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
            finally
            {
                if (createdBootConfigFileStream != null)
                {
                    createdBootConfigFileStream.Dispose();
                }
            }
            return createdBootConfigPath;
        }

        /// <summary>
        /// 読み込み用ファイルの内容を、書き込み用ファイルの最後に追加して書き込む
        /// </summary>
        /// <param name="readFile">読み込み用ファイル</param>
        /// <param name="writeFile">書き込みファイル</param>
        /// <param name="writeFile">起動を実行するデバイスのIPアドレスを含むパラメータ</param>
        private void ReadAndWriteFile(string readFile, ref FileStream writeFile)
        {
            using (var writerFileStream = new FileStream(
                    readFile, FileMode.Open, FileAccess.Read))
            {
                try
                {
                    var fileSize = writerFileStream.Length;
                    var fileBuffer = new byte[fileSize];
                    var readBuffer = writerFileStream.Read(fileBuffer, 0, (int)fileSize);
                    if (readBuffer < (int)fileSize)
                    {
                        throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev.ReadAndWriteFile");
                    }
                    writeFile.Write(fileBuffer, 0, (int)fileSize);
                }
                catch (TargetShellLibraryException exception)
                {
                    TargetShellLibrary.PrintException(exception);
                }
            }
        }

        /// <summary>
        /// 特定IPのデバイスに対して起動を実行
        /// </summary>
        /// <param name="parameter">起動を実行するデバイスのIPアドレスを含むパラメータ</param>
        private void StartTarget(CommandParameter parameter)
        {
            var outputDetectTarget = string.Empty;
            var isSuccess = false;

            // デバイスが起動するまで1秒間隔で待機して起動チェックをする
            // （10秒程で起動するはず）
            for (var i = 0; i < ConstantsSdev.MaxWaitStartUpDevice; i++)
            {
                if (!InvokeExe(parameter, ConstantsSdev.ControlTargetPrivatePath,
                        "detect-target", CommonConstants.OutputLogName,
                        out outputDetectTarget))
                {
                    throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                            "WaitStartUpDevice");
                }
                if (outputDetectTarget.Contains(parameter.TargetEntry.SerialNumber))
                {
                    isSuccess = true;
                    break;
                }
                Thread.Sleep(1000);
            }

            // タイムアウトしている場合はエラーとする
            if (!isSuccess)
            {
                throw new TargetShellLibraryException("InitializeBuildImageEdev." +
                        "WaitStartUpDevice");
            }
            var command = $" --poweron --ip {parameter.TargetEntry.IpAddress}";
            if (!InvokeExe(parameter, ConstantsSdev.GPIOControllerCUI,
                    command, CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev.RestartTarget");
            }
        }

        /// <summary>
        /// 特定IPのデバイスに対して再起動を実行
        /// </summary>
        /// <param name="parameter">再起動を実行するデバイスのIPアドレスを含むパラメータ</param>
        private void RestartTarget(CommandParameter parameter)
        {
            var command = $" --reset --ip {parameter.TargetEntry.IpAddress}";
            if (!InvokeExe(parameter, ConstantsSdev.GPIOControllerCUI,
                    command, CommonConstants.OutputLogName))
            {
                throw new TargetShellLibraryException("WriteBootConfigPlugin.Sdev.RestartTarget");
            }
        }

        /// <summary>
        /// 実行ファイルを呼び出す
        /// </summary>
        /// <param name="parameter">実行ファイルを呼び出すパラメータ</param>
        /// <param name="exeName">SDKルート以降のパスを含む実行ファイル名</param>
        /// <param name="command">指定するコマンド</param>
        /// <param name="logName">ログファイル名</param>
        /// <param name="outputString">実行した出力結果の文字列</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool InvokeExe(CommandParameter parameter,
            string exeName, string command, string logName, out string outputString)
        {
            var returnCode = TargetShellLibrary.InvokeExe(
                    Path.Combine(parameter.NintendoSdkRoot, exeName), command,
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber, logName,
                    out outputString);
            return returnCode;
        }

        /// <summary>
        /// 実行ファイルを呼び出す
        /// </summary>
        /// <param name="parameter">実行ファイルを呼び出すパラメータ</param>
        /// <param name="exeName">SDKルート以降のパスを含む実行ファイル名</param>
        /// <param name="command">指定するコマンド</param>
        /// <param name="logName">ログファイル名</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool InvokeExe(CommandParameter parameter,
            string exeName, string command, string logName)
        {
            var returnCode = TargetShellLibrary.InvokeExe(
                    Path.Combine(parameter.NintendoSdkRoot, exeName), command,
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber, logName);
            return returnCode;
        }
    }
}
