﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace PrivateNx.InitializeBuildImage
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Security.AccessControl;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using TargetShell.Library;
    using TargetShell.CommandPluginBase;
    using TargetShell.DevicePluginBase;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 初期化イメージ(initimg)を使ってSDEVの初期化を行うクラス
    /// </summary>
    [Export(typeof(IDevicePluginInterface<CommandParameter>))]
    [ExportMetadata("SupportedHardwareType", "Sdev")]
    public class InitializeBuildImageSdev :
            BaseDevicePlugin<CommandParameter>, IDevicePluginInterface<CommandParameter>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InitializeBuildImageSdev()
        {
            this.ParallelModeFlag = true;
        }

        /// <summary>
        /// サーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public override List<DevicePluginInfo> GetDeviceInfo()
        {
            var searchDeviceInfoList = TargetShellLibrary.GetDeviceInfoSdev();
            return searchDeviceInfoList;
        }

        /// <summary>
        /// 初期化イメージ(initimg)を使ってSDEVの初期化を行う
        /// </summary>
        /// <param name="deviceList">デバイスリストのパラメータ</param>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>成功したデバイスの数</returns>
        public override int RunDevicePlugin(
                List<TargetEntry> deviceList, CommandParameter parameter)
        {
            // 成功したデバイスの数
            var successDeviceCount = 0;
            // Sdevのみを取得する
            var devicesSdev = GetTargetDevice(deviceList, GetDeviceInfo());
            // コマンドを並列処理で実行する
            var logSubDir = new string[devicesSdev.Count];
            Parallel.For(0, devicesSdev.Count, id =>
            {
                var localParameter = new CommandParameter
                {
                    TargetEntry = devicesSdev[id],
                    Verbose = parameter.Verbose,
                    NintendoSdkRoot = parameter.NintendoSdkRoot,
                    BuildImageParam = parameter.BuildImageParam,
                    LogOutputDir = parameter.LogOutputDir
                };
                // ログ出力先を設定している場合のみログディレクトリを作成する
                localParameter.SerialDirectroy = TargetShellLibrary.CreateSerialNumberLogDirectory(
                        parameter.LogOutputDir, localParameter.TargetEntry.SerialNumber);
                SpecificProcessDevicePlugin(localParameter, ref successDeviceCount);
            });
            return successDeviceCount;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        protected override void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                ConvertPlatformName(ref parameter);
                if (parameter.BuildImageParam.KeyType.Equals("Auto"))
                {
                    ConvertKeyType(ref parameter);
                }
                // 署名済みOSならダミーのブートコンフィグを書き込む
                if (parameter.BuildImageParam.SignedType ==
                            ConstantsSdev.CommandString.SignedString ||
                            parameter.BuildImageParam.DeveloperType ==
                            ConstantsSdev.CommandString.PublicString)
                {
                    if (!UpDateBootConfig(parameter))
                    {
                        return;
                    }
                }
                // ホストブリッジの更新
                if (!UpDateHostBridgeInitialize(parameter))
                {
                    return;
                }
                // デベロッパータイプがPublicの場合は署名をSignedに変更する
                if (parameter.BuildImageParam.DeveloperType ==
                            ConstantsSdev.CommandString.PublicString)
                {
                    parameter.BuildImageParam.SignedType =
                            ConstantsSdev.CommandString.SignedString;
                }
                // リカバリーライター更新
                if (!PowerOn(parameter))
                {
                    return;
                }
                Thread.Sleep(15000);
                if (!UpDateRecoveryWriter(parameter))
                {
                    return;
                }
                Thread.Sleep(15000);
                if (!Reset(parameter))
                {
                    return;
                }
                Thread.Sleep(10000);
                // K1 K5ならシリアル番号を確認して書き込まれていなければ書き込み
                if (parameter.BuildImageParam.KeyType.Equals(
                        ConstantsSdev.CommandString.KeyTypeK1) ||
                        parameter.BuildImageParam.KeyType.Equals(
                        ConstantsSdev.CommandString.KeyTypeK5))
                {
                    if (string.IsNullOrEmpty(parameter.TargetEntry.SerialNumber))
                    {
                        WriteSerialNumber(parameter);
                    }
                }
                // システムイメージを書き込む
                if (!WriteInitializeImage(parameter))
                {
                    return;
                }
                // HelloWorldをスキップしないなら
                if (!parameter.BuildImageParam.SkipHelloWorld)
                {
                    // HellowWorldの実行
                    if (!RunHelloWorld(parameter))
                    {
                        return;
                    }
                }
                Interlocked.Increment(ref successDeviceCount);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// プラットフォームネームをディレクトリ名に変更する
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        private void ConvertPlatformName(ref CommandParameter parameter)
        {
            if (parameter.BuildImageParam.PlatformName.Equals(
                    ConstantsSdev.CommandString.Nxfp2a64))
            {
                parameter.BuildImageParam.PlatformName =
                        ConstantsSdev.CommandString.Nxfp2a64Path;
            }
            else if (parameter.BuildImageParam.PlatformName.Equals(
                    ConstantsSdev.CommandString.Nxfp2a32))
            {
                parameter.BuildImageParam.PlatformName =
                    ConstantsSdev.CommandString.Nxfp2a32Path;
            }
        }

        /// <summary>
        /// ブートコンフィグの生成
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <param name="bootConfigErementUniqueFilePath">開発機ごとのブートコンフィグ</param>
        /// <param name="keyType">キータイプ</param>
        /// <param name="paramsCnt">ブートコンフィグファイル書き込みカウント</param>
        /// <returns>生成したブートコンフィグのパス</returns>
        private string CreateBootConfigFile(CommandParameter parameter,
            string bootConfigErementUniqueFilePath, string keyType, string paramsBc)
        {
            // 連結したブートコンフィグのファイルネーム
            var createdBootConfigPath = Path.GetTempFileName();
            using (var createdBootConfigFileStream = new FileStream(
                    createdBootConfigPath, FileMode.Append, FileAccess.Write))
            {
                // アクセス権を付与
                TargetShellLibrary.SetAccessControlCurrentUser(createdBootConfigPath,
                        FileSystemRights.WriteData);

                // ブートコンフィグ生成用のを読み込み、連結ファイルに書き込む
                var emmcWriterImageName = string.Empty;
                switch (keyType)
                {
                    case ConstantsSdev.CommandString.KeyTypeK1:
                        emmcWriterImageName = ConstantsSdev.RecoveryBoot.EmmcWriterImageK1;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK2:
                        emmcWriterImageName = ConstantsSdev.RecoveryBoot.EmmcWriterImageK2;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK3:
                        emmcWriterImageName = ConstantsSdev.RecoveryBoot.EmmcWriterImageK3;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeK5:
                        emmcWriterImageName = ConstantsSdev.RecoveryBoot.EmmcWriterImageK5;
                        break;
                    case ConstantsSdev.CommandString.KeyTypeM2:
                        emmcWriterImageName = ConstantsSdev.RecoveryBoot.EmmcWriterImageM2;
                        break;
                    default:
                        break;
                }
                var bootConfigErementEmmcFilePath = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.ImagePath,
                    emmcWriterImageName);
                using (var emmcWriterFileStream = new FileStream(
                        bootConfigErementEmmcFilePath, FileMode.Open, FileAccess.Read))
                {
                    var emmcWriterFileSize = emmcWriterFileStream.Length;
                    var emmcFileBuf = new byte[emmcWriterFileSize];
                    emmcWriterFileStream.Read(emmcFileBuf, 0, (int)emmcWriterFileSize);
                    createdBootConfigFileStream.Write(emmcFileBuf, 0, (int)emmcWriterFileSize);
                }
                // 固有のブートコンフィグを読み込み、連結ファイルに書き込む
                using (var uniqueFileStream = new FileStream(
                        bootConfigErementUniqueFilePath, FileMode.Open, FileAccess.Read))
                {
                    var uniqueFileSize = uniqueFileStream.Length;
                    var uniqueFilebuf = new byte[uniqueFileSize];
                    uniqueFileStream.Read(uniqueFilebuf, 0, (int)uniqueFileSize);
                    createdBootConfigFileStream.Write(uniqueFilebuf, 0, (int)uniqueFileSize);
                }
                // ブートコンフィグ生成用のを読み込み、連結ファイルに書き込む
                var bootConfigErementBcFilePath = Path.Combine(
                        parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.ImagePath,
                        paramsBc);
                using (var bcFileStream = new FileStream(
                        bootConfigErementBcFilePath, FileMode.Open, FileAccess.Read))
                {
                    var bcFileSize = bcFileStream.Length;
                    var bcFileBuf = new byte[bcFileSize];
                    bcFileStream.Read(bcFileBuf, 0, (int)bcFileSize);
                    createdBootConfigFileStream.Write(bcFileBuf, 0, (int)bcFileSize);
                }
            }
            return createdBootConfigPath;
        }

        /// <summary>
        /// 取得したハードウエアバージョンからキータイプを設定する
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool ConvertKeyType(ref CommandParameter parameter)
        {
            var returnCode = false;
            var logStrings = string.Empty;
            // controlTargetPrivate.exeのパス生成
            var controlTargetPrivateExe = ConstantsSdev.ControlTargetPrivate.ExeName;
            controlTargetPrivateExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.ControlTargetPrivate.Path,
                    controlTargetPrivateExe);
            // KeyType取得のコマンド生成
            var controlTargetPrivateCommandBuilder = new StringBuilder();
            controlTargetPrivateCommandBuilder.AppendFormat(
                    " get-hostbridge-version");
            controlTargetPrivateCommandBuilder.AppendFormat(
                    " --target-ip {0}", parameter.TargetEntry.IpAddress);
            // controlTargetPrivate.exeの実行
            returnCode = TargetShellLibrary.InvokeExe(controlTargetPrivateExe,
                    controlTargetPrivateCommandBuilder.ToString(),
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber,
                    CommonConstants.OutputLogName, out logStrings);
            if (!returnCode)
            {
                return returnCode;
            }
            // HardwareConfigからKeyTypeに変更する
            parameter.BuildImageParam.KeyType = ConvertHardwareConfigToKeyType(logStrings);
            if (string.IsNullOrEmpty(parameter.BuildImageParam.KeyType))
            {
                return false;
            }

            return returnCode;
        }

        /// <summary>
        /// ホストブリッジのリカバリが必要かどうか確認する
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:リカバリが必要 False:リカバリが不要</returns>
        private bool IsRequiredRecoveryHostBridge(CommandParameter parameter)
        {
            var returnCode = false;
            var Required = false;
            var logStrings = string.Empty;
            // ホストブリッジリカバリを実施するか確認
            var updateHostBridgeExe = ConstantsSdev.UpdateHostBridge.ExeName;
            updateHostBridgeExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.UpdateHostBridge.Path,
                    updateHostBridgeExe);
            // ホストブリッジリカバリを確認するコマンド生成
            var hostBridgeRecoveryCommandBuilder = new StringBuilder();
            hostBridgeRecoveryCommandBuilder.AppendFormat(" required");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                hostBridgeRecoveryCommandBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            // ホストブリッジリカバリを確認する
            returnCode = TargetShellLibrary.InvokeExe(updateHostBridgeExe,
                    hostBridgeRecoveryCommandBuilder.ToString(),
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber,
                    CommonConstants.OutputLogName, out logStrings);
            Required = logStrings.Contains(ConstantsSdev.UpdateHostBridge.CheckString);
            return Required;
        }

        /// <summary>
        /// ホストブリッジのバージョンを取得する
        /// </summary>
        /// <param name="parameter">ブートコンフィグ書き込みを行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool GetHostBridgeVersion(CommandParameter parameter,
                out string hostBridgeVersion)
        {
            var returnCode = false;
            var logStrings = string.Empty;
            hostBridgeVersion = string.Empty;
            // GPIOControllerCUI.exeのパス生成
            var gpioControllerCUIExe = ConstantsSdev.GPIOControllerCUI.ExeName;
            gpioControllerCUIExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.GPIOControllerCUI.Path,
                    gpioControllerCUIExe);
            // HostBridgeのイメージパス生成
            var hostBridgeImagePath = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.GPIOControllerCUI.HostBridgeImage);
            // UpdateHostBridgeのコマンド生成
            var getHostBridgeVersionCommandBuilder = new StringBuilder();
            getHostBridgeVersionCommandBuilder.AppendFormat(" --get_hb_build_version");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                getHostBridgeVersionCommandBuilder.AppendFormat(
                        " --ip {0}", parameter.TargetEntry.IpAddress);
            }
            // ホストブリッジのバージョンを取得
            returnCode = TargetShellLibrary.InvokeExe(gpioControllerCUIExe,
                    getHostBridgeVersionCommandBuilder.ToString(),
                    parameter.SerialDirectroy, parameter.TargetEntry.SerialNumber,
                    CommonConstants.OutputLogName, out logStrings);
            hostBridgeVersion = logStrings;
            return returnCode;
        }

        /// <summary>
        /// NXの電源ONする
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool PowerOn(CommandParameter parameter)
        {
            var returnCode = false;
            returnCode = ControlGPIO(parameter, parameter.TargetEntry.IpAddress, "--poweron");
            return returnCode;
        }

        /// <summary>
        /// NXをリセットする
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool Reset(CommandParameter parameter)
        {
            var returnCode = false;
            returnCode = ControlGPIO(parameter, parameter.TargetEntry.IpAddress, "--reset");
            return returnCode;
        }

        /// <summary>
        /// GPIOの制御
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <param name="targetIp">ターゲットのIPアドレス</param>
        /// <param name="command">コマンド</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool ControlGPIO(CommandParameter parameter,
                string targetIp, string command)
        {
            var returnCode = false;
            // GPIOControllerCUI.exeのパス生成
            var gpioControllerCUIExe = ConstantsSdev.GPIOControllerCUI.ExeName;
            gpioControllerCUIExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.GPIOControllerCUI.Path,
                    gpioControllerCUIExe);
            // GPIOControllerCUIのコマンド生成
            var gpioControllerCUICommandBuilder = new StringBuilder();
            if (!string.IsNullOrEmpty(command))
            {
                gpioControllerCUICommandBuilder.AppendFormat(command);
            }
            if (!string.IsNullOrEmpty(targetIp))
            {
                gpioControllerCUICommandBuilder.AppendFormat(" --ip {0}", targetIp);
            }
            // GPIOControllerCUI.exeの実行
            returnCode = TargetShellLibrary.InvokeExe(gpioControllerCUIExe,
                    gpioControllerCUICommandBuilder.ToString(), parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
            return returnCode;
        }

        /// <summary>
        /// 初期化イメージの書き込み
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool WriteInitializeImage(CommandParameter parameter)
        {
            var returnCode = false;
            var connectionType = ConvertHostBridgeName(parameter);
            // 実行ファイル
            var systemUpdaterHostFsNsp = ConstantsSdev.RunOnTargetPrivate.SystemUpdaterHostFsNsp;
            systemUpdaterHostFsNsp = Path.Combine(
                    parameter.NintendoSdkRoot,
                    ConstantsSdev.RunOnTargetPrivate.SystemUpdaterHostFsNspPathBase,
                    parameter.BuildImageParam.PlatformName,
                    ConstantsSdev.RunOnTargetPrivate.SystemUpdaterHostFsNspPath,
                    parameter.BuildImageParam.BuildType,
                    systemUpdaterHostFsNsp);
            // イメージファイル
            var internalInitimg = ConstantsSdev.RunOnTargetPrivate.InternalInitimg;
            // シリアル番号にXG*(Copper)が含まれていたら32Gのイメージファイルを使用する
            var ImageSize = ConstantsSdev.RunOnTargetPrivate.LargeImageSize;
            if (parameter.TargetEntry.SerialNumber.Contains(
                    ConstantsSdev.CommandString.CopperDevice))
            {
                ImageSize = ConstantsSdev.RunOnTargetPrivate.SmallImageSize;
            }
            var internalInitimgPath =
                    ConstantsSdev.RunOnTargetPrivate.DirectoryPrefix +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    parameter.BuildImageParam.KeyType +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    connectionType +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    parameter.BuildImageParam.SignedType +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    parameter.BuildImageParam.BootType +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    ImageSize +
                    ConstantsSdev.RunOnTargetPrivate.DirectoryDelimiter +
                    parameter.BuildImageParam.DeveloperType;
            internalInitimg = Path.Combine(
                    parameter.NintendoSdkRoot,
                    ConstantsSdev.RunOnTargetPrivate.InternalInitimgPathBase,
                    parameter.BuildImageParam.PlatformName,
                    ConstantsSdev.RunOnTargetPrivate.InternalInitimgPath,
                    internalInitimgPath, parameter.BuildImageParam.BuildType,
                    internalInitimgPath + internalInitimg);
            // RunOnTargetPrivateのコマンド生成
            var runOnTargetPrivateCommandBuilder = new StringBuilder();
            runOnTargetPrivateCommandBuilder.AppendFormat(" run");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                runOnTargetPrivateCommandBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --reset");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --wait-after-reset {0}", "10");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --verbose");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --monitor-serial");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --hostbridge {0}", parameter.TargetEntry.IpAddress);
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --failure-timeout {0}", "600");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --pattern-success-exit \"Succeeded initializing the system.\"");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " {0}", systemUpdaterHostFsNsp);
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " -- --input {0}", internalInitimg);
            returnCode = InvokeRunOnTargetPrivate(parameter,
                    runOnTargetPrivateCommandBuilder.ToString());
            return returnCode;
        }

        /// <summary>
        /// HelloWorldの実行
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool RunHelloWorld(CommandParameter parameter)
        {
            var returnCode = false;
            // HelloWorld.nspファイル
            var helloWorldNsp = ConstantsSdev.RunOnTargetPrivate.HelloWorldNsp;
            helloWorldNsp = Path.Combine(
                    parameter.NintendoSdkRoot,
                    ConstantsSdev.RunOnTargetPrivate.HelloWorldNspPathBase,
                    parameter.BuildImageParam.PlatformName,
                    ConstantsSdev.RunOnTargetPrivate.HelloWorldNspPath,
                    parameter.BuildImageParam.BuildType,
                    ConstantsSdev.RunOnTargetPrivate.HelloWorldNsp);
            // RunOnTargetPrivateのコマンド生成
            var runOnTargetPrivateCommandBuilder = new StringBuilder();
            runOnTargetPrivateCommandBuilder.AppendFormat(" run");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                runOnTargetPrivateCommandBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --reset");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --wait-after-reset {0}", "10");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --verbose");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --monitor-serial");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --hostbridge {0}", parameter.TargetEntry.IpAddress);
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --failure-timeout {0}", "30");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " --pattern-success-exit \"Hello, world.\"");
            runOnTargetPrivateCommandBuilder.AppendFormat(
                    " {0} -- 5", helloWorldNsp);
            returnCode = InvokeRunOnTargetPrivate(parameter,
                    runOnTargetPrivateCommandBuilder.ToString());
            return returnCode;
        }

        /// <summary>
        /// RunOnTargetPrivateの実行
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool InvokeRunOnTargetPrivate(CommandParameter parameter,
                string command)
        {
            var returnCode = false;
            // RunOnTargetPrivate.exeのパス生成
            var runOnTargetPrivateExe = ConstantsSdev.RunOnTargetPrivate.ExeName;
            runOnTargetPrivateExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RunOnTargetPrivate.Path,
                    runOnTargetPrivateExe);
            // GPIOControllerCUI.exeの実行
            returnCode = TargetShellLibrary.InvokeExe(runOnTargetPrivateExe,
                    command, parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
            return returnCode;
        }

        /// <summary>
        /// シリアル番号の書き込み
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool WriteSerialNumber(CommandParameter parameter)
        {
            var returnCode = false;
            // WriteSerialNumberToHostBridge.exeのパス生成
            var writeSerialNumberToHostBridgeExe = ConstantsSdev.WriteSerialNumberToHostBridge.ExeName;
            writeSerialNumberToHostBridgeExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.WriteSerialNumberToHostBridge.Path,
                    writeSerialNumberToHostBridgeExe);
            // WriteSerialNumberToHostBridgeExe.exeのコマンド生成
            var writeSerialNumberToHostBridgeCommandBuilder = new StringBuilder();
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                writeSerialNumberToHostBridgeCommandBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            writeSerialNumberToHostBridgeCommandBuilder.AppendFormat(
                    " --write-name --verbose");
            // シリアル番号書き込み
            returnCode = TargetShellLibrary.InvokeExe(writeSerialNumberToHostBridgeExe,
                    writeSerialNumberToHostBridgeCommandBuilder.ToString(),
                    parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber,
                    CommonConstants.OutputLogName);
            return returnCode;
        }

        /// <summary>
        /// リカバリーライターを書き込む
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpDateRecoveryWriter(CommandParameter parameter)
        {
            var returnCode = false;
            var connectionType = ConvertHostBridgeName(parameter);
            // RecoveryBootExe.exeのパス生成
            var recoveryBootExe = ConstantsSdev.RecoveryBoot.ExeName;
            recoveryBootExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.Path, recoveryBootExe);
            // リカバリライターのパス生成
            var recoveryWriterFile = ConstantsSdev.RecoveryBoot.RecoveryWriterFile;
            var recoveryWirterPath = ConstantsSdev.RecoveryBoot.RecoveryWirterPath;
            recoveryWirterPath = recoveryWirterPath +
                    ConstantsSdev.RecoveryBoot.DirectoryDelimiter +
                    parameter.BuildImageParam.KeyType +
                    ConstantsSdev.RecoveryBoot.DirectoryDelimiter +
                    connectionType +
                    ConstantsSdev.RecoveryBoot.DirectoryDelimiter +
                    parameter.BuildImageParam.SignedType;
            recoveryWriterFile = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.OutputImagePath,
                    parameter.BuildImageParam.PlatformName,
                    ConstantsSdev.RecoveryBoot.SystemImagesPath, recoveryWirterPath,
                    parameter.BuildImageParam.BuildType,
                    recoveryWirterPath + recoveryWriterFile);
            // RecoveryBootのコマンド生成
            var recoveryBootCommandBuilder = new StringBuilder();
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                recoveryBootCommandBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            if (!string.IsNullOrEmpty(recoveryWriterFile))
            {
                recoveryBootCommandBuilder.AppendFormat(
                        " --image {0}", recoveryWriterFile);
            }
            // ブートコンフィグ書き込み実行
            returnCode = TargetShellLibrary.InvokeExe(recoveryBootExe,
                    recoveryBootCommandBuilder.ToString(), parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
            return returnCode;
        }

        /// <summary>
        /// ブートコンフィグの更新 を行う
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpDateBootConfig(CommandParameter parameter)
        {
            // RecoveryBootExe.exeのパス生成
            var recoveryBootExe = ConstantsSdev.RecoveryBoot.ExeName;
            recoveryBootExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.Path, recoveryBootExe);
            // ブートコンフィグ作成
            var dummyBootConfigFile = ConstantsSdev.RecoveryBoot.NxDummyBootConfigFile;
            dummyBootConfigFile = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.RecoveryBoot.NxBootConfigsPath,
                    dummyBootConfigFile);
            // 4つのBootConfigファイルを書き込む
            foreach (var paramsBc in ConstantsSdev.RecoveryBoot.ParamsBc)
            {
                var createBootConfigFilePath = CreateBootConfigFile(
                        parameter, dummyBootConfigFile, parameter.BuildImageParam.KeyType, paramsBc);
                // RecoveryBootのコマンド生成
                var recoveryBootCommandBuilder = new StringBuilder();
                if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
                {
                    recoveryBootCommandBuilder.AppendFormat(
                            " --target {0}", parameter.TargetEntry.IpAddress);
                }
                if (!string.IsNullOrEmpty(createBootConfigFilePath))
                {
                    recoveryBootCommandBuilder.AppendFormat(
                            " --image {0}", createBootConfigFilePath);
                }
                // ブートコンフィグ書き込み実行
                if (!TargetShellLibrary.InvokeExe(recoveryBootExe,
                        recoveryBootCommandBuilder.ToString(), parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName))
                {
                    return false;
                }
                // 連続でBootConfig書き換えを行う為とBootConfig書き換え後にリセットを行う為に5秒待つ
                // (Write-BootConfig.ps1と同様)
                Thread.Sleep(5000);
            }
            return true;
        }

        /// <summary>
        /// ホストブリッジの更新 を行う
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpDateHostBridgeInitialize(CommandParameter parameter)
        {
            var returnCode = false;
            var hostBridgeVersion = string.Empty;
            // ホストブリッジの更新ツールのパス生成
            var hostBridgeControllerExe = ConstantsSdev.HostBridgeController.ExeName;
            // ホストブリッジのイメージ
            var hostBridgeImagePath = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.HostBridgeController.ImagePackage);
            var versions = new ConstantsSdev.HostBridgeControllerVersions();
            // ホストブリッジを更新するかチェック
            GetHostBridgeVersion(parameter, out hostBridgeVersion);
            for (var i = 0; i < versions.List.GetLength(0); i++)
            {
                if (hostBridgeVersion.Contains(versions.List[i, 0]) &&
                        (string.IsNullOrEmpty(versions.List[i, 1])))
                {
                    // 最新で更新不要
                    return true;
                }
            }
            hostBridgeControllerExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.HostBridgeController.Path,
                    hostBridgeControllerExe);
            // UpdateHostBridgeのコマンド生成
            var updateHostBridgeBuilder = new StringBuilder();
            updateHostBridgeBuilder.AppendFormat(" --UpdateBridgeFirmware");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                updateHostBridgeBuilder.AppendFormat(
                        " --ip {0}", parameter.TargetEntry.IpAddress);
            }
            if (!string.IsNullOrEmpty(hostBridgeImagePath))
            {
                updateHostBridgeBuilder.AppendFormat(
                        " --fwpath {0}", hostBridgeImagePath);
            }
            // ホストブリッジ書き込み実行
            returnCode = TargetShellLibrary.InvokeExe(hostBridgeControllerExe,
                    updateHostBridgeBuilder.ToString(), parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
            if (!returnCode)
            {
                return returnCode;
            }
            // 必須スリープ
            Thread.Sleep(45000);
            if (parameter.BuildImageParam.KeyType.Equals(
                    ConstantsSdev.CommandString.KeyTypeK1) ||
                    parameter.BuildImageParam.KeyType.Equals(
                    ConstantsSdev.CommandString.KeyTypeK2) ||
                    parameter.BuildImageParam.KeyType.Equals(
                    ConstantsSdev.CommandString.KeyTypeK3))
            {
                if (IsRequiredRecoveryHostBridge(parameter))
                {
                    // ホストブリッジのリカバリを実施
                    returnCode = UpDateHostBridge(parameter);
                    if (!returnCode)
                    {
                        return returnCode;
                    }
                }
            }
            return returnCode;
        }

        /// <summary>
        /// ホストブリッジの更新 を行う
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>True:成功 False:失敗</returns>
        private bool UpDateHostBridge(CommandParameter parameter)
        {
            var returnCode = false;
            // UpdateHostBridge.exeのパス生成
            var updateHostBridgeExe = ConstantsSdev.UpdateHostBridge.ExeName;
            updateHostBridgeExe = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.UpdateHostBridge.Path,
                    updateHostBridgeExe);
            // HostBridgeのイメージパス生成
            var hostBridgeImagePath = Path.Combine(
                    parameter.NintendoSdkRoot, ConstantsSdev.UpdateHostBridge.ImagePath);
            // UpdateHostBridgeのコマンド生成
            var updateHostBridgeBuilder = new StringBuilder();
            updateHostBridgeBuilder.AppendFormat(" update");
            if (!string.IsNullOrEmpty(parameter.TargetEntry.IpAddress))
            {
                updateHostBridgeBuilder.AppendFormat(
                        " --target {0}", parameter.TargetEntry.IpAddress);
            }
            if (!string.IsNullOrEmpty(hostBridgeImagePath))
            {
                updateHostBridgeBuilder.AppendFormat(
                        " --image {0}", hostBridgeImagePath);
            }
            updateHostBridgeBuilder.AppendFormat(" --waitboot");
            if (parameter.Verbose)
            {
                updateHostBridgeBuilder.AppendFormat(" --verbose");
            }
            // ホストブリッジ書き込み実行
            returnCode = TargetShellLibrary.InvokeExe(updateHostBridgeExe,
                    updateHostBridgeBuilder.ToString(), parameter.SerialDirectroy,
                    parameter.TargetEntry.SerialNumber, CommonConstants.OutputLogName);
            return returnCode;
        }

        /// <summary>
        /// HardwareConfigからKeyTypeに変更する
        /// </summary>
        /// <param name="invokeExeName">実行ファイル名</param>
        /// <returns>変換したKeyType</returns>
        private string ConvertHardwareConfigToKeyType(string hardwareConfig)
        {
            var data = new ConstantsSdev.ConvertHardwareConfigToKeyTypeClass();
            for (var i = 0; i < data.ConvertHardwareConfigToKeyTypeTable.GetLength(0); i++)
            {
                if (hardwareConfig.Contains(data.ConvertHardwareConfigToKeyTypeTable[i, 0]))
                {
                    return data.ConvertHardwareConfigToKeyTypeTable[i, 1];
                }
            }
            return string.Empty;
        }

        /// <summary>
        /// HostBridgeの名称の変換を行う
        /// </summary>
        /// <param name="parameter">デバイスの初期化を行うパラメータ</param>
        /// <returns>変換結果</returns>
        private string ConvertHostBridgeName(CommandParameter parameter)
        {
            if (parameter.BuildImageParam.ConnectionType.Equals(
                    ConstantsSdev.CommandString.HostBridge))
            {
                return ConstantsSdev.CommandString.Hb;
            }
            else
            {
                return parameter.BuildImageParam.ConnectionType;
            }
        }
    }
}
