﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.PluginInterface
{
    using System;
    using System.Collections.Generic;
    using TargetShell.Properties;

    /// <summary>
    /// オプション引数
    /// </summary>
    public class OptionArgs
    {
        /// <summary>
        /// 共通サブコマンドオプション
        /// </summary>
        public CommonOptions CommonArgs { get; set; }
        /// <summary>
        /// 共通デバイスプラグインオプション
        /// </summary>
        public CommonDevicePluginOptions DevicePluginArgs { get; set; }
    }

    /// <summary>
    /// デバイスプラグインで使用する共通のオプション
    /// </summary>
    public class CommonDevicePluginOptions : ICloneable
    {
        /// <summary>
        /// 冗長モードかどうか
        /// </summary>
        public bool Verbose { get; set; }
        /// <summary>
        /// ログ出力ディレクトリパス
        /// </summary>
        public string LogOutputDir { get; set; }
        /// <summary>
        /// シリアル番号のディレクトリパス
        /// </summary>
        public string SerialDirectroy { get; set; }
        /// <summary>
        /// デバイスエントリ
        /// </summary>
        public TargetEntry TargetEntry { get; set; }
        /// <summary>
        /// ログ表示文字列
        /// </summary>
        public string[] LogHideString { get; set; }
        /// <summary>
        /// ログ表示のタイトル
        /// </summary>
        public string OutputTitleString { get; set; }
        /// <summary>
        /// ログの修飾文字
        /// </summary>
        public string OutputOneLinePrefix { get; set; }
        /// <summary>
        /// 実行カウント
        /// </summary>
        public int RunCount { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommonDevicePluginOptions()
        {
            this.TargetEntry = new TargetEntry();
        }

        /// <summary>
        /// クラスのクローン生成
        /// </summary>
        public CommonDevicePluginOptions Clone()
        {
            var instance = (CommonDevicePluginOptions)this.MemberwiseClone();
            return instance;
        }

        /// <summary>
        /// ICloneable.Cloneの明示的な実装
        /// </summary>
        object ICloneable.Clone()
        {
            return Clone();
        }
    }

    /// <summary>
    /// コマンドオプションの共通定義
    /// </summary>
    public class Constants
    {
        /// <summary>
        /// オプション connection-path の文字列
        /// <summary>
        public const string OptionConnectionPath = "connection-path";
        /// <summary>
        /// オプション detect-device の文字列
        /// <summary>
        public const string OptionDetectDevice = "detect-device";
        /// <summary>
        /// オプション name の文字列
        /// <summary>
        public const string OptionDetectName = "name";
        /// <summary>
        /// オプション hw-config の文字列
        /// <summary>
        public const string OptionHwConfig = "hw-config";
        /// <summary>
        /// オプション hw-type の文字列
        /// <summary>
        public const string OptionHwType = "hw-type";
        /// <summary>
        /// オプション ip-address の文字列
        /// <summary>
        public const string OptionIpAddress = "ip-address";
        /// <summary>
        /// オプション output の文字列
        /// <summary>
        public const string OptionOutput = "output";
        /// <summary>
        /// オプション serial-number の文字列
        /// <summary>
        public const string OptionSerialNumber = "serial-number";
        /// <summary>
        /// オプション verbose の省略文字列
        /// <summary>
        public const char OptionShortVerbose = 'v';
        /// <summary>
        /// オプション verbose の文字列
        /// <summary>
        public const string OptionVerbose = "verbose";
    }

    /// <summary>
    /// サブコマンドの共通オプションです。
    /// </summary>
    public class CommonOptions
    {
        /// <summary>
        /// 冗長モードかどうか
        /// </summary>
        public bool Verbose { get; set; }
        /// <summary>
        /// 接続されているデバイスをすべて対象とするかどうか
        /// </summary>
        public bool DetectDevice { get; set; }
        /// <summary>
        /// デバイス名
        /// </summary>
        public string Name { get; set; }
        /// <summary>
        /// シリアル番号
        /// </summary>
        public string SerialNumber { get; set; }
        /// <summary>
        /// IP アドレス
        /// </summary>
        public List<string> Ipaddress { get; set; }
        /// <summary>
        /// 開発機種別
        /// </summary>
        public string HardwareType { get; set; }
        /// <summary>
        /// ハードウェア構成
        /// </summary>
        public string HardwareConfig { get; set; }
        /// <summary>
        /// 接続方式
        /// </summary>
        public ConnectionPath ConnectionPath { get; set; }
        /// <summary>
        /// ログ出力ディレクトリ
        /// </summary>
        public string OutputDirectory { get; set; }
        /// <summary>
        /// 共通オプション設定
        /// </summary>
        public OptionSetting Setting { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommonOptions()
        {
            this.Setting = new OptionSetting();
            this.Ipaddress = new List<string>();
        }

        /// <summary>
        /// 共通オプション設定
        /// </summary>
        public class OptionSetting
        {
            /// <summary>
            /// オプション verbose のショート文字列
            /// </summary>
            public char OptionShortVerbose { get; set; }
            /// <summary>
            /// オプション hw-type の説明文
            /// </summary>
            public string DescriptionHwType { get; set; }
            /// <summary>
            /// オプション connection-path の説明文
            /// </summary>
            public string DescriptionConnectionPath { get; set; }

            public OptionSetting()
            {
                /// オプション hw-type の初期値
                /// WriteBootConfigで書き換えが必要なため残しておく
                DescriptionHwType = Resources.DescriptionHwType;
                /// オプション connection-path の初期値
                /// WriteBootConfigで書き換えが必要なため残しておく
                DescriptionConnectionPath = Resources.DescriptionConnectionPath;
            }
        }

        /// <summary>
        /// コマンドラインパーサに自身を登録します。
        /// </summary>
        /// <param name="context">コマンドを構築するコンテキストです。</param>
        public void RegisterWith(dynamic command)
        {
            command
                .AddFlagOption(Constants.OptionShortVerbose, Constants.OptionVerbose,
                        new Action(() => this.Verbose = true))
                .GetBuilder()
                .SetDescription(Resources.DescriptionVerbose);
            command
                .AddFlagOption(
                    Constants.OptionDetectDevice,
                    new Action(() => this.DetectDevice = true))
                .GetBuilder()
                .SetDescription(Resources.DescriptionDetectDevice);
            command
                .AddValueOption(
                    Constants.OptionDetectName,
                    new Action<string>(x => this.Name = x))
                .GetBuilder()
                .SetDescription(Resources.DescriptionName);
            command
                .AddValueOption(
                    Constants.OptionSerialNumber,
                    new Action<string>(x => this.SerialNumber = x))
                .GetBuilder()
                .SetDescription(Resources.DescriptionNumber);
            command
                .AddValueOption(
                    Constants.OptionIpAddress,
                    new Action<string>(x => this.Ipaddress.Add(x)))
                .GetBuilder()
                .SetCanHaveValues(true)
                .SetDescription(Resources.DescriptionIpAddress);
            command
                .AddValueOption(
                    Constants.OptionHwType,
                    new Action<string>(x => this.HardwareType = x))
                .GetBuilder()
                .SetDescription(Setting.DescriptionHwType);
            command
                .AddValueOption(
                    Constants.OptionHwConfig,
                    new Action<string>(x => this.HardwareConfig = x))
                .GetBuilder()
                .SetDescription(Resources.DescriptionHwConfig);
            command
                .AddValueOption(
                    Constants.OptionConnectionPath,
                    new Action<string>(AcceptConnectionpathOption))
                .GetBuilder()
                .SetDescription(Setting.DescriptionConnectionPath);
            command
                .AddValueOption(
                    Constants.OptionOutput,
                    new Action<string>(x => this.OutputDirectory = x))
                .GetBuilder()
                .SetDescription(Resources.DescriptionOutput);
        }

        /// <summary>
        /// Connectionpathを解析して取得します
        /// </summary>
        /// <param name="connectionPath">解析されて取得した文字列</param>
        public void AcceptConnectionpathOption(string connectionPath)
        {
            // ConnectionPathに変換して取得する
            if (Enum.IsDefined(typeof(ConnectionPath), connectionPath))
            {
                ConnectionPath = (ConnectionPath)Enum.Parse(
                        typeof(ConnectionPath), connectionPath);
            }
            else
            {
                ConnectionPath = ConnectionPath.NotSpecified;
            }
        }
    }
}
