﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell.Library
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Security.AccessControl;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading;
    using TargetShell.PluginInterface;

    /// <summary>
    /// 共通で使用する定義一覧
    /// </summary>
    public static class CommonConstants
    {
        /// <summary>
        /// NintendoSdkRootを検索する文字列
        /// </summary>
        public const string NintendoSdkRootMark = @"NintendoSdkRootMark";
        /// <summary>
        /// CommandLineToolsのパス
        /// </summary>
        public const string RunnerToolsPath = @"Tools\CommandLineTools";
        /// <summary>
        /// 打刻フォーマット
        /// </summary>
        public const string TimeStampFormat = @"yyyy/MM/dd HH:mm:ss.fff";
        /// <summary>
        /// 出力ログのファイル名
        /// </summary>
        public const string OutputLogName = @"TargetSehllOutput.log";
    }

    /// <summary>
    /// 共通で使用するデバイスの定義
    /// </summary>
    public static class CommonDeviceConstants
    {
        /// <summary>
        /// Edevのサーチ条件
        /// </summary>
        public const string SearchDeviceKeywordEdev = "EDEV";
        /// <summary>
        /// サポートハードウエアタイプ
        /// </summary>
        public const string SupportHardwareTypeEdev = "Edev";
        /// <summary>
        /// Sdevのサーチ条件
        /// </summary>
        public const string SearchDeviceKeywordSdev = "SDEV";
        /// <summary>
        /// サポートハードウエアタイプ
        /// </summary>
        public const string SupportHardwareTypeSdev = "Sdev";
    }

    /// <summary>
    /// コマンド解析結果
    /// Nintendo.ToolFoundation.CommandLine.ParseResultTypeのParseResultTypeと同じ
    /// </summary>
    public enum CommandLineParserResult
    {
        /// <summary>
        /// 処理に成功しました。
        /// </summary>
        Success = 0,
        /// <summary>
        /// エラーがあり、処理に失敗しました。
        /// </summary>
        Failure = 1,
        /// <summary>
        /// ヘルプオプションが指定されました。
        /// </summary>
        Help = 2,
        /// <summary>
        /// バージョンオプションが指定されました。
        /// </summary>
        Version = 3
    }

    public class TargetShellLibraryException : Exception
    {
        /// <summary>
        /// TargetShellLibraryのエクセプション処理
        /// </summary>
        public TargetShellLibraryException(string message) : base(message)
        {
        }
    }

    /// <summary>
    /// コマンド解析クラス
    /// </summary>
    public class CommandLineParser
    {
        /// <summary>
        /// ToolFoundationCommandLineのDLL情報
        /// </summary>
        public Assembly ToolFoundationCommandLine { get; set; }

        /// <summary>
        /// CommandLineの型
        /// </summary>
        public Type CommandLineType { get; set; }

        /// <summary>
        /// Actionの型
        /// </summary>
        public Type ActionType { get; set; }

        /// <summary>
        /// ParseSettingsの型
        /// </summary>
        public Type ParseSettingsType { get; set; }

        /// <summary>
        /// ParseResultTypeの型
        /// </summary>
        public Type ParseResultType { get; set; }

        /// <summary>
        /// CreateRootCommandのメソッド情報
        /// </summary>
        public MethodInfo CreateRootCommandMethodInfo { get; set; }

        /// <summary>
        /// ParseArgsのメソッド情報
        /// </summary>
        public MethodInfo ParseArgsMethodInfo { get; set; }

        /// <summary>
        /// アクションコマンド
        /// </summary>
        public dynamic ActionCommand { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommandLineParser()
        {
            try
            {
                this.ToolFoundationCommandLine = Assembly.LoadFrom(
                    Path.Combine(
                        Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                        @"Libraries\TargetShell\Nintendo.ToolFoundation.CommandLine.dll"));
                this.CommandLineType = this.ToolFoundationCommandLine.GetType(
                    "Nintendo.ToolFoundation.CommandLine.CommandLine");
                this.ActionType = this.ToolFoundationCommandLine.GetType(
                    "Nintendo.ToolFoundation.CommandLine.ActionCommand");
                this.ParseSettingsType = this.ToolFoundationCommandLine.GetType(
                    "Nintendo.ToolFoundation.CommandLine.ParseSettings");
                this.ParseResultType = this.ToolFoundationCommandLine.GetType(
                    "Nintendo.ToolFoundation.CommandLine.ParseResultType");
                this.CreateRootCommandMethodInfo = this.ActionType.GetMethod(
                    "CreateRootCommand",
                    new Type[] { });
                this.ParseArgsMethodInfo = this.CommandLineType.GetMethod(
                    "ParseArgs",
                    new Type[]
                    {
                        typeof(IEnumerable<string>),
                        this.ActionType,
                        this.ParseSettingsType
                    });
                this.ActionCommand = this.CreateRootCommandMethodInfo.Invoke(null, null);
            }
            catch (Exception exception)
            {
                throw exception;
            }
        }

        /// <summary>
        /// コマンド解析実行
        /// </summary>
        /// <param name="command">コマンド文字列配列</param>
        /// <returns>CommandLineParserResult 結果</returns>
        public CommandLineParserResult ParseArgs(string[] command)
        {
            var result = (CommandLineParserResult)Convert.ChangeType(
                    this.ParseArgsMethodInfo.Invoke(
                            null,
                            new object[] { command, this.ActionCommand, null }),
                    this.ParseResultType);
            return result;
        }
    }

    public class TargetShellLibrary
    {
        /// <summary>
        /// ロックオブジェクト
        /// </summary>
        private static object lockObject = new object();

        /// <summary>
        /// 型変換する
        /// </summary>
        /// <param name="castParameter">キャスト元</param>
        /// <returns>キャスト後</returns>
        public static T CastParameter<T>(object castParameter)
        {
            T localCastParameter = default(T);
            try
            {
                if (castParameter is T)
                {
                    localCastParameter = (T)castParameter;
                }
                else
                {
                    throw new TargetShellLibraryException("Type switch error.");
                }
            }
            catch (TargetShellLibraryException message)
            {
                throw message;
            }
            return localCastParameter;
        }

        /// <summary>
        /// 打刻文字列を取得する
        /// </summary>
        public static string GetTimeStamp()
        {
            return DateTime.Now.ToString(CommonConstants.TimeStampFormat);
        }

        /// <summary>
        /// NintendoSdkRootのパスを取得する
        /// </summary>
        /// <returns>NintendoSdkRootのパス</returns>
        public static string GetNintendoSdkRootPath()
        {
            var sdkRootPath = string.Empty;
            try
            {
                sdkRootPath = GetNintendoSdkRootPathString();
            }
            catch (TargetShellLibraryException exception)
            {
                PrintException(exception);
                // SdkRootが取れない場合は終了させる
                Environment.Exit(1);
            }
            return sdkRootPath;
        }

        /// <summary>
        /// NintendoSdkRootのパスを取得する
        /// </summary>
        /// <returns>NintendoSdkRootのパス</returns>
        private static string GetNintendoSdkRootPathString()
        {
            var nintendoSdkRootDirectory = new DirectoryInfo(Path.GetDirectoryName(
                    Process.GetCurrentProcess().MainModule.FileName));
            while (!File.Exists(Path.Combine(nintendoSdkRootDirectory.FullName,
                    CommonConstants.NintendoSdkRootMark)))
            {
                if (nintendoSdkRootDirectory.Parent == null)
                {
                    throw new TargetShellLibraryException("Not found NintendoSdkRootMark file.");
                }
                nintendoSdkRootDirectory = nintendoSdkRootDirectory.Parent;
            }
            return nintendoSdkRootDirectory.FullName;
        }

        /// <summary>
        /// 検索したデバイス一覧を表示する
        /// </summary>
        /// <param name="deviceList">検索したデバイス一覧</param>
        public static void DumpList(List<TargetEntry> deviceList)
        {
            Console.WriteLine("\n--Search Result--");
            var i = 0;
            foreach (var entry in deviceList)
            {
                Console.WriteLine("Device:{0}", ++i);
                Console.WriteLine("  IP: {0}", entry.IpAddress);
                Console.WriteLine("  SerialNumber: {0}", entry.SerialNumber);
                Console.WriteLine("  DeviceName: {0}", entry.DeviceName);
                Console.WriteLine("  HardwareType: {0}", entry.HardwareType);
                Console.WriteLine("  HardwareConfig: {0}", entry.HardwareConfig);
                Console.WriteLine("  ConnectionPath: {0}\n", entry.ConnectionPath);
            }
            Console.WriteLine(" Total {0} Devices found\n", i);
        }

        /// <summary>
        /// Target Manager を終了させる
        /// </summary>
        /// <returns>成功、失敗を返す</returns>
        public static bool StopTargetManager()
        {
            bool status = StopProcesses("NintendoTargetManager");
            return status;
        }

        /// <summary>
        /// 指定したプロセスを終了させる
        /// </summary>
        /// <param name="processName">終了させるプロセス名</param>
        /// <returns>成功、失敗を返す</returns>
        public static bool StopProcesses(string processName)
        {
            Process[] ps = Process.GetProcessesByName(processName);
            var loopCount = 0;
            while (ps.Length > 0 && loopCount++ < 10)
            {
                foreach (Process p in ps)
                {
                    // 終了しているプロセスを取得することがあるので
                    // 終了プロセスの場合は、処理を行わない
                    if (!p.HasExited)
                    {
                        // プロセスを閉じる
                        var closeRet = p.CloseMainWindow();
                        if (closeRet)
                        {
                            p.WaitForExit(10000);
                            if (p.HasExited)
                            {
                                ps = Process.GetProcessesByName(processName);
                                continue;
                            }
                        }

                        // プロセスを終了する
                        ProcessJob processJob = null;
                        try
                        {
                            processJob = new ProcessJob(p);
                            processJob.SetKillOnJobCloseFlag();
                        }
                        finally
                        {
                            processJob.Dispose();
                            processJob = null;
                        }

                        p.WaitForExit(10000);
                    }
                    else
                    {
                        Console.WriteLine($"Exited process={p.Id}");
                        Thread.Sleep(1000);
                    }
                }
                ps = Process.GetProcessesByName(processName);
            }

            ps = Process.GetProcessesByName(processName);
            if (ps.Length > 0)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// ログを出力しながら処理を実行する
        /// </summary>
        /// <param name="invokeExeName">実行ファイル名</param>
        /// <param name="optionString">実行ファイルに渡す追加パラメータ</param>
        /// <param name="serialDirectroy">ディレクトリ名</param>
        /// <param name="serialNumber">シリアルナンバー</param>
        /// <param name="logFileName">ログのファイル名</param>
        /// <returns>True:成功 False:失敗</returns>
        public static bool InvokeExe(string invokeExeName, string optionString,
                string serialDirectroy, string serialNumber, string logFileName)
        {
            var tempStrings = string.Empty;
            return InvokeExe(invokeExeName, optionString, serialDirectroy,
                    serialNumber, logFileName, out tempStrings);
        }

        /// <summary>
        /// ログを出力しながら処理を実行する
        /// </summary>
        /// <param name="invokeExeName">実行ファイル名</param>
        /// <param name="optionString">実行ファイルに渡す追加パラメータ</param>
        /// <param name="serialDirectroy">ディレクトリ名</param>
        /// <param name="serialNumber">シリアルナンバー</param>
        /// <param name="logFileName">ログのファイル名</param>
        /// <param name="logStrings">ログに書き込む文字列</param>
        /// <param name="streamWriterData">作成済みストリーム</param>
        /// <param name="logHide">ログ表示</param>
        /// <returns>True:成功 False:失敗</returns>
        public static bool InvokeExe(string invokeExeName, string optionString,
                string serialDirectroy, string serialNumber, string logFileName,
                out string logStrings, StreamWriter streamWriterData = null, bool logHide = true)
        {
            var returnCode = false;
            var localLogStrings = new StringBuilder();
            using (var process = new Process())
            {
                StreamWriter streamWriter = null;
                if ((streamWriterData == null) && (!string.IsNullOrEmpty(serialDirectroy)))
                {
                    streamWriter = new StreamWriter(Path.Combine(serialDirectroy,
                            logFileName), true);
                }
                try
                {
                    if (streamWriter != null)
                    {
                        // アクセス権を付与
                        TargetShellLibrary.SetAccessControlCurrentUser(Path.Combine(
                                serialDirectroy, logFileName), FileSystemRights.WriteData);
                    }
                    if (streamWriterData != null)
                    {
                        streamWriter = streamWriterData;
                    }

                    // ログ出力の準備
                    var argument = optionString;
                    var path = Assembly.GetExecutingAssembly().Location;
                    var directory = Path.GetDirectoryName(path);
                    var fileNameLog = GetTimeStamp() +
                            " FileName: " + invokeExeName;
                    var argumentLog = GetTimeStamp() +
                            " Argument: " + argument;
                    Console.WriteLine(serialNumber + " " + fileNameLog);
                    Console.WriteLine(serialNumber + " " + argumentLog);
                    if (streamWriter != null)
                    {
                        streamWriter.WriteLine(fileNameLog);
                        streamWriter.WriteLine(argument);
                        streamWriter.Flush();
                    }
                    var handler = new DataReceivedEventHandler(
                        (object obj, DataReceivedEventArgs args) =>
                        {
                            if (args.Data == null)
                            {
                                return;
                            }
                            var stringNow = GetTimeStamp();
                            var data = stringNow + " " + args.Data;
                            if (logHide)
                            {
                                Console.WriteLine(serialNumber + " " + data);
                            }
                            if (streamWriter != null)
                            {
                                streamWriter.WriteLine(data);
                                streamWriter.Flush();
                            }
                            localLogStrings.AppendLine(args.Data);
                        });
                    // 処理実行
                    process.OutputDataReceived += handler;
                    process.ErrorDataReceived += handler;
                    process.StartInfo.FileName = Path.Combine(directory, invokeExeName);
                    process.StartInfo.Arguments = argument;
                    process.StartInfo.RedirectStandardOutput = true;
                    process.StartInfo.RedirectStandardError = true;
                    process.StartInfo.UseShellExecute = false;
                    process.Start();
                    process.BeginOutputReadLine();
                    process.BeginErrorReadLine();
                    process.WaitForExit();
                    Console.WriteLine("{0} {1} exit code = {2}",
                        serialNumber,
                        GetTimeStamp(),
                        process.ExitCode);
                    returnCode = (process.ExitCode == 0) ? true : false;
                }
                catch (TargetShellLibraryException exception)
                {
                    Console.WriteLine(serialNumber + " " +
                            GetTimeStamp() + "[ERROR] " +
                            exception.Message);
                    Console.WriteLine(serialNumber + " " +
                            GetTimeStamp() +
                            "StackTrace: " + exception.StackTrace);
                    if (streamWriter != null)
                    {
                        streamWriter.WriteLine(GetTimeStamp() +
                            "[ERROR] " + exception.Message);
                        streamWriter.WriteLine(GetTimeStamp() +
                                "StackTrace: " + exception.StackTrace);
                        streamWriter.Flush();
                    }
                    localLogStrings.AppendLine(GetTimeStamp() +
                            "[ERROR] " + exception.Message);
                    localLogStrings.AppendLine(GetTimeStamp() +
                                "StackTrace: " + exception.StackTrace);
                }
                finally
                {
                    if ((streamWriter != null) && (streamWriterData == null))
                    {
                        streamWriter.Dispose();
                    }
                }
            }
            logStrings = localLogStrings.ToString();
            return returnCode;
        }

        /// <summary>
        /// 例外が発生したときに例外を画面に表示する
        /// </summary>
        /// <param name="exception">発生した例外</param>
        public static void PrintException(Exception exception)
        {
            lock (lockObject)
            {
                Console.WriteLine("[ERROR] {0}", exception.Message);
                Console.WriteLine("StackTrace: {0}", exception.StackTrace);
            }
        }

        /// <summary>
        /// ファイルをコピーする
        /// </summary>
        /// <param name="sourceFileName">コピーするファイル。</param>
        /// <param name="destFileName">コピー先ファイルの名前。
        /// このパラメーターには、ディレクトリは指定できません。</param>
        /// <param name="overwrite">コピー先ファイルが上書きできる場合は true。
        /// それ以外の場合は false。</param>
        /// <returns>True:成功 False:失敗</returns>
        public static bool CopyFile(string sourceFileName, string destFileName,
                bool overWrite = false)
        {
            try
            {
                File.Copy(sourceFileName, destFileName, overWrite);
            }
            catch (Exception exception)
            {
                if (exception is TargetShellLibraryException)
                {
                    TargetShellLibrary.PrintException(exception);
                }
                else
                {
                    Console.Error.WriteLine(sourceFileName +
                    ": Error: Failed to back up the file on disk");
                    Console.Error.WriteLine("Failed to auto-format.");
                    if (exception is IOException)
                    {
                        Console.Error.WriteLine($"'{destFileName}' already exists. " +
                            "Please remove '{destFileName}'.");
                    }
                    Console.Error.WriteLine(exception);
                }
                return false;
            }

            return true;
        }

        /// <summary>
        /// ディレクトリの有無をチェックして、なければディレクトリ作成を行う
        /// </summary>
        /// <param name="path1">作成するパス1</param>
        /// <param name="path2">path1に続くパス2</param>
        /// <param name="path3">path2に続くパス3</param>
        /// <returns>true:成功 false:失敗</returns>
        public static bool CreateDirectory(string path1, string path2 = null, string path3 = null)
        {
            var directory = string.Empty;

            // パス1が未設定なら抜ける
            if (string.IsNullOrEmpty(path1))
            {
                return false;
            }

            // ディレクトリにパス1を設定
            directory = path1;

            // パス2を設定しているなら結合する
            if (!string.IsNullOrEmpty(path2))
            {
                directory = Path.Combine(directory, path2);
            }

            // パス3を設定しているなら結合する
            if (!string.IsNullOrEmpty(path3))
            {
                directory = Path.Combine(directory, path3);
            }

            // ディレクトリ有無チェック
            if (!Directory.Exists(directory))
            {
                // ディレクトリが無ければ作成する
                Directory.CreateDirectory(directory);
            }
            return true;
        }

        /// <summary>
        /// 特定文字を最初にヒットした箇所にのみ置き換える
        /// </summary>
        /// <param name="input">置換される文字列</param>
        /// <param name="oldString">検索する文字列</param>
        /// <param name="newString">置換する文字列</param>
        /// <returns>置き換え後の文字列</returns>
        public static string ReplaceFirst(string input, string oldString, string newString)
        {
            var returnString = input;
            // 置き換えする文字が含まれているかチェック
            if (Regex.IsMatch(input, oldString))
            {
                var re = new Regex(oldString);
                returnString = re.Replace(input, newString, 1);
            }
            return returnString;
        }

        /// <summary>
        /// ファイル、ディレクトリが存在するか調べる
        /// </summary>
        /// <param name="path">ファイル、ディレクトリ名</param>
        /// <returns>ファイル、ディレクトリが存在する場合はtrue。それ以外はfalse</returns>
        public static bool IsExistFile(string path)
        {
            if (!File.Exists(path))
            {
                Console.WriteLine("[ERROR]file not found：{0}", path);
                return false;
            }
            return true;
        }

        /// <summary>
        /// 指定したファイルに対してカレントユーザーのアクセス権を設定する
        /// </summary>
        /// <param name="filePath">アクセス権を設定するファイル名のフルパス</param>
        /// <param name="fileSystemRights">設定するアクセス権</param>
        public static void SetAccessControlCurrentUser(string filePath,
                FileSystemRights fileSystemRights)
        {
            try
            {
                // ファイル有無チェック
                if (IsExistFile(filePath))
                {
                    // カレントユーザーを取得
                    var user = Environment.UserDomainName + "\\" + Environment.UserName;

                    // アクセス権を設定
                    FileSystemAccessRule accessRule = new FileSystemAccessRule(user,
                            fileSystemRights, AccessControlType.Allow);
                    var accessControl = File.GetAccessControl(filePath);
                    accessControl.AddAccessRule(accessRule);
                    File.SetAccessControl(filePath, accessControl);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// Nxサーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public static List<DevicePluginInfo> GetDeviceInfoNx()
        {
            var searchDeviceInfoList = new List<DevicePluginInfo>();
            var searchDeviceInfoEdev = new DevicePluginInfo();
            searchDeviceInfoEdev.SupportHardwareType =
                    CommonDeviceConstants.SupportHardwareTypeEdev;
            searchDeviceInfoEdev.SearchDeviceKeyword =
                    CommonDeviceConstants.SearchDeviceKeywordEdev;
            searchDeviceInfoList.Add(searchDeviceInfoEdev);
            var searchDeviceInfoSdev = new DevicePluginInfo();
            searchDeviceInfoSdev.SupportHardwareType =
                    CommonDeviceConstants.SupportHardwareTypeSdev;
            searchDeviceInfoSdev.SearchDeviceKeyword =
                    CommonDeviceConstants.SearchDeviceKeywordSdev;
            searchDeviceInfoList.Add(searchDeviceInfoSdev);
            return searchDeviceInfoList;
        }

        /// <summary>
        /// Sdevサーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public static List<DevicePluginInfo> GetDeviceInfoSdev()
        {
            var searchDeviceInfoList = new List<DevicePluginInfo>();
            var searchDeviceInfoSdev = new DevicePluginInfo();
            searchDeviceInfoSdev.SupportHardwareType =
                    CommonDeviceConstants.SupportHardwareTypeSdev;
            searchDeviceInfoSdev.SearchDeviceKeyword =
                    CommonDeviceConstants.SearchDeviceKeywordSdev;
            searchDeviceInfoList.Add(searchDeviceInfoSdev);
            return searchDeviceInfoList;
        }

        /// <summary>
        /// Edevサーチ条件取得
        /// </summary>
        /// <returns>サーチ条件の文字列</returns>
        public static List<DevicePluginInfo> GetDeviceInfoEdev()
        {
            var searchDeviceInfoList = new List<DevicePluginInfo>();
            var searchDeviceInfoEdev = new DevicePluginInfo();
            searchDeviceInfoEdev.SupportHardwareType =
                    CommonDeviceConstants.SupportHardwareTypeEdev;
            searchDeviceInfoEdev.SearchDeviceKeyword =
                    CommonDeviceConstants.SearchDeviceKeywordEdev;
            searchDeviceInfoList.Add(searchDeviceInfoEdev);
            return searchDeviceInfoList;
        }

        /// <summary>
        /// シリアル番号のログディレクトリ作成
        /// </summary>
        /// <param name="logOutputDir">ログ出力用のディレクトリパス</param>
        /// <param name="serialNumber">シリアル番号</param>
        /// <returns>作成したログディレクトリパス</returns>
        public static string CreateSerialNumberLogDirectory(
                string logOutputDir, string serialNumber)
        {
            var logSubDir = string.Empty;
            // ログ出力先を設定している場合のみログフォルダを作成する
            // logOutputDirが設定されていない場合は空文字を返却する
            if (!string.IsNullOrEmpty(logOutputDir))
            {
                logSubDir = Path.Combine(logOutputDir,
                        serialNumber);
                if (!Directory.Exists(logSubDir))
                {
                    Directory.CreateDirectory(logSubDir);
                }
            }
            return logSubDir;
        }

        /// <summary>
        /// 日付のログディレクトリを生成する
        /// </summary>
        /// <param name="outputDirectory">出力ディレクトリ</param>
        /// <returns>日付のログディレクトリパス</returns>
        public static string CreateDateLogDirectory(string outputDirectory)
        {
            var logOutputDir = string.Empty;
            // ログ出力先を設定している場合のみログディレクトリを作成する
            if (!string.IsNullOrEmpty(outputDirectory))
            {
                if (!Directory.Exists(outputDirectory))
                {
                    Directory.CreateDirectory(outputDirectory);
                }
                var stringNow = DateTime.Now.ToString("yyyyMMddHHmmss");
                logOutputDir = Path.Combine(outputDirectory, stringNow);
                if (!Directory.Exists(logOutputDir))
                {
                    Directory.CreateDirectory(logOutputDir);
                }
            }
            return logOutputDir;
        }

        /// ディレクトリが存在するか調べる
        /// </summary>
        /// <param name="path">ディレクトリ名</param>
        /// <returns>ディレクトリが存在する場合はtrue。それ以外はfalse</returns>
        public static bool IsExistDirectory(string path)
        {
            if (!Directory.Exists(path))
            {
                Console.WriteLine("[ERROR]directory not found：{0}", path);
                return false;
            }
            return true;
        }
    }
}
