﻿#############################################################################
#
# 指定された Confluence ページから変更履歴の収集・出力に必要な設定を読んで
# gatherChangeLog や outputChangeLog から読み込める状態で保持しておく
#
#############################################################################

param
(
    [string]$AuthenticationFile = "",
	[string]$AliasFile = "",
	[string]$ConfigFile = "",
    [string]$User = "",
    [string]$Password = "",
	[string]$aliasConfPageID = "",
    [string]$configConfPageID = ""
)

#----------------  User,Password の決定
if ( "" -eq $User -or "" -eq $Password )
{
    #--- Authentication File が指定されていない場合
    if ( "" -eq $AuthenticationFile )
    {
            $AuthenticationFile = ".\config\Authentication.xml"
    }

	[Xml]$AuthConfig = Get-Content $AuthenticationFile
    # ユーザー名
    if ( "" -eq $User )
    {
        $User = $AuthConfig.Configuration.User
    }
    # パスワード
    if ( "" -eq $Password )
    {
        $Password = $AuthConfig.Configuration.Password
    }
}
if ( "" -eq $User )
{
    Write-Error "User name is not specified."
    exit 1
}
if ( "" -eq $Password )
{
    Write-Error "Password is not specified."
    exit 1
}

#----------------  Aliases.xml から設定ファイルが書かれた Confluence ページの ID を取得
if ( $AliasFile -ne "" )
{
    Write-Host("Alias File : {0}" -f $AliasFile)
    if ( Test-Path -path "$AliasFile" )
    {
        [XML]$ConfluencePage = Get-Content -Encoding UTF8 $AliasFile
        $aliasConfPageID = $ConfluencePage.AliasList.AliasConfPageId
        Write-Host("Alias Confluence Page ID : {0}" -f $aliasConfPageID)
    }
}

if ( $ConfigFile -ne "")
{
    Write-Host("Config File : {0}" -f $ConfigFile)
    if ( Test-Path -path "$ConfigFile" )
    {
        [XML]$ConfluencePage = Get-Content -Encoding UTF8 $ConfigFile
        $configConfPageID = $ConfluencePage.Configuration.ConfigConfPageId
        Write-Host("Config Confluence Page ID : {0}" -f $configConfPageID)
    }
}


# リリース日
$releaseDate = (Get-Date).ToString("yyyy/MM/dd")

#================================================================================
# Confluence からデータを取得して XML オブジェクトに変換して返す
#
function Get-ConfluenceData( $PageID )
{
    #---- Rest URL の作成
    $RestUrl = "http://spdlybra.nintendo.co.jp/confluence/rest/api/content/{0}?expand=body.storage" -f $PageID
    Write-Host $RestUrl

    # このような URL になる
    #      http://spdlybra.nintendo.co.jp/confluence/rest/api/content/123456789?expand=body.storage

    #---- Confluence からデータを取得
    $RestResult = Invoke-RestApi $RestUrl $User $Password
    if ( $null -eq $RestResult )
    {
        Write-Host "** No article matched."
        return
    }

    #---- JSON パース
    $code=@"
static function parseJSON(json)
{
    return eval('(' +json + ')');
}
"@

    $JSONUtil = (Add-Type -Language JScript -MemberDefinition $code -Name "JSONUtil" -PassThru)[1]
    $jsobj = $JSONUtil::parseJSON($RestResult)
    $confStorage = $jsobj['body']['storage']['value']

    $confObject = New-Object System.Xml.XmlDocument
    
    # nbsp 回避のための DTD 宣言、ac 名前空間宣言
    $xmlDefinition = @"
<!DOCTYPE doctypeName [
   <!ENTITY nbsp "&#160;">
]>
<ac:xxx xmlns:ac='dummy'>
"@

    $confStorage = $xmlDefinition + $confStorage + '</ac:xxx>'
    $confObject.LoadXml([string]$confStorage)
    
    return $confStorage
}

#================================================================================
# REST API を投げる
#
function Invoke-RestApi( $Url, $User, $Password )
{
    $Auth = '{0}:{1}' -f $User, $Password
    $AuthBytes = [System.Text.Encoding]::ASCII.GetBytes($Auth)
    $Base64 = [System.Convert]::ToBase64String($AuthBytes)
    $Headers = @{"Authorization" = "Basic $Base64"}

    $request = [System.Net.WebRequest]::Create($url)
    $request.Method = "GET"
    $request.Headers["Authorization"] = "Basic $Base64"

    try
    {
        $response = $request.GetResponse()
        $stream = $response.GetResponseStream()
        $reader   = New-Object "System.IO.StreamReader" $stream
        return $reader.ReadToEnd()
    }
    catch [System.Net.WebException] 
    {
        Write-Error "Error occurred in accessing JIRA server." # サーバアクセス中に例外発生
        return $null
    }
    finally
    {
        if ( $reader ){ $reader.Dispose() }
        if ( $strem  ){ $reader.Dispose() }
        if ( $reader ){ $reader.Dispose() }
    }
}

#================================================================================
# 取得した Confluence の XML オブジェクトから ID が振られているマクロを連想配列に格納する
#
function Get-StructuredMacroMap( $confObject )
{
    $strMacroMap = @{}
    
    # ac:structured-macro の配列
    $acStrMacros = $confObject.GetElementsByTagName("ac:structured-macro")
    
    # Confluence ページ上の ac:structured-macro の配列をループ
    foreach ($acStrMacro in $acStrMacros)
    {
        # div マクロ以外は無視(Scroll Translations など全体を包むマクロによる不具合を考慮）
        if ( $acStrMacro.GetAttribute("ac:name") -ne "div" )
        {
            continue;
        }

        # ac:parameter という名前の要素の配列を得る GetElementsByTagname は XmlNodeList を返す
        $acParams = $acStrMacro.GetElementsByTagName("ac:parameter")
        # id 入っている ac:parameter を格納する変数
        $acId = ""
        
        # このループで、ac:parameter 群の中から ac:name="id" のものを探しその値を $acId に入れてループ完了
        foreach ($acParam in $acParams)
        {    
            if ( $acParam.GetAttribute("ac:name") -eq "id" )
            {
                $acId = $acParam.InnerText
                break;
            }
        }
        
        # ID が空でなかったら、ID をキーにして連想配列に格納
        if ( $acId -ne "")
        {
            $strMacroMap.Add($acId, $acStrMacro.GetElementsByTagName("ac:rich-text-body").Item(0).innerXml)
        }
    }
    return $strMacroMap
}

#================================================================================
# XML の文字列に DOCTYPE 宣言などを付けて XML オブジェクトを返す
#
function Get-XmlObject( $xmlStr )
{
    $xmlDefinition = @"
<!DOCTYPE doctypeName [
   <!ENTITY nbsp "&#160;">
]>
<ac:xxx xmlns:ac='dummy'>
"@

    $xmlStr = $xmlDefinition + $xmlStr + '</ac:xxx>'

    $local:confObject = New-Object System.Xml.XmlDocument
    $local:confObject.LoadXml([string]$xmlStr)

    return $local:confObject
}

#================================================================================
# 変更履歴を収集する修正バージョンを配列で返す
#
function Get-GatherVersions()
{
    $gatherVersions = @()

    # gather のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($aliasMacroMap["gather"])
    $trTags = $confObject.GetElementsByTagName("tr")
    foreach ( $trTag in $trTags )
    {
        $tdTags = $trTag.GetElementsByTagName("td")
        foreach ( $tdTag in $tdTags )
        {
            $gatherVersions += $tdTag.innerText
        }
    }

    return $gatherVersions
}

#================================================================================
# TopVersion の数を返す（TopVersion＝バージョン文字列、日付、バージョン・・の集合）
#
function Get-NumberOfTopVersion_Confluence()
{
    $confMacroId = "changelog_top"
    # 英語なら ID を変える
    if($English -eq $true)
    {
        $confMacroId = "changelog_top_en"
    }
    # changelog_top のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $trTags = $confObject.GetElementsByTagName("tr")

    # テーブルのヘッダー部分を除くため 1 引いて返す
    return $trTags.Count-1
}

#================================================================================
# 指定したインデックスで TopVersion を取得する
#
function Get-TopVersionInfo_Confluence( $index )
{
    $confMacroId = "changelog_top"
    # 英語なら ID を変える
    if($English -eq $true)
    {
        $confMacroId = "changelog_top_en"
    }

    # changelog_top のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $trTags = $confObject.GetElementsByTagName("tr")

    # 行の最大数はヘッダーの一行分を抜いた数
    $maxNum = $trTags.Count - 1

    # インデックスが 0 未満や最大数を超えていたらリターン
    if ( $index -lt 0 -or $maxNum -le $index)
    {
        Write-Host "Index Error"
        return
    }

    $tdTags = $trTags.Item($index + 1).GetElementsByTagName("td")

    # 返り値：バージョンリストを作成
    [array]$versions = @()
    foreach($tdVersion in $tdTags.Item(0).ChildNodes)
    {
        $versions += [string]$tdVersion.innerText
    }

    # 返り値：バージョンの文字列
    $versionString = $tdTags.Item(1).innerText
    # versionString が空文字列ならバージョン番号の一つ目を使用する
    if( $versionString.trim() -eq "")
    {
        $versionString = $versions[0]
    }

    # 返り値：日付を置換
    $packageDate = $tdTags.Item(2).innerText -replace "{{date}}", $releaseDate
    $packageDate = "($packageDate)"
    
    # Write-Host ("Top Version : {0}, {1}, {2}" -f $versions, $versionString, $packageDate)
    return [array]$versions, $versionString, $packageDate
}

#================================================================================
# パッケージの数を返す（パッケージ＝バージョン文字列、日付、バージョン・・の集合）
#
function Get-PackageListPackageNum_Confluence()
{
    $confMacroId = "changelog_each_packages"
    # 英語なら ID を変える
    if($English -eq $true)
    {
        $confMacroId = "changelog_each_packages_en"
    }

    # changelog_each_packages のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $trTags = $confObject.GetElementsByTagName("tr")

    # テーブルのヘッダー部分を除くため 1 引いて返す
    return $trTags.Count-1
}

#================================================================================
# パッケージごとの情報を取得する。ChangelogConfig と違い各パッケージには同じ情報を使う
#
function Get-PackageListPackageInfo_Confluence( $index )
{
    $confMacroId = "changelog_each_packages"
    # 英語なら ID を変える
    if($English -eq $true)
    {
        $confMacroId = "changelog_each_packages_en"
    }

    # changelog_each_packages のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $trTags = $confObject.GetElementsByTagName("tr")

    # 行の最大数はヘッダーの一行分を抜いた数
    $maxNum = $trTags.Count - 1

    # インデックスが 0 未満や最大数を超えていたらリターン
    if ( $index -lt 0 -or $maxNum -le $index)
    {
        Write-Host "Index Error"
        return
    }

    $tdTags = $trTags.Item($index + 1).GetElementsByTagName("td")

    # 返り値：バージョンリストを作成
    [array]$versions = @()
    foreach($tdVersion in $tdTags.Item(0).ChildNodes)
    {
        $versions += [string]$tdVersion.innerText
    }

    # 返り値：バージョンの文字列
    $versionString = $tdTags.Item(1).innerText

    # 返り値：日付を置換
    $packageDate = $tdTags.Item(2).innerText -replace "{{date}}", $releaseDate
    $packageDate = "($packageDate)"
    
    # Write-Host ("Top Version : {0}, {1}, {2}" -f $versions, $versionString, $packageDate)
    return [array]$versions, $versionString, $packageDate
}

#================================================================================
# バージョン文字列のエイリアス情報を作る
#
function Create-VersionAliases_Confluence()
{
    # aliases_version のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($aliasMacroMap["aliases_version"])
    $trTags = $confObject.GetElementsByTagName("tr")

    $aliasRule = @{}
    $fnameRule = @{}
    $aliasRuleArray = @()
    $fnameRuleArray = @()

    for($i=1; $i -lt $trTags.Count; $i++)
    {
        $tdTags = $trTags.Item($i).GetElementsByTagName("td");
        # [0] = alias, [1] = name, [2] = fname
        $aliasRule.Add( $tdTags.Item(0).innerText, $tdTags.Item(1).innerText )
        $fnameRule.Add( $tdTags.Item(1).innerText, $tdTags.Item(2).innerText )
        $aliasRuleArray += $tdTags.Item(0).innerText
        $fnameRuleArray += $tdTags.Item(1).innerText
    }
    
    return $aliasRule, $fnameRule, $aliasRuleArray, $fnameRuleArray
}


#================================================================================
# ID 文字列のエイリアス情報を作る
#
function Create-IdAliases_Confluence(  )
{
    # aliases_id のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($aliasMacroMap["aliases_id"])
    $trTags = $confObject.GetElementsByTagName("tr")

    $aliasRule = @{}

    for($i=1; $i -lt $trTags.Count; $i++)
    {
        $tdTags = $trTags.Item($i).GetElementsByTagName("td");
        # [0] = alias, [1] = idStr
        $aliasRule.Add( $tdTags.Item(0).innerText, $tdTags.Item(1).innerText )
    }

    return $aliasRule
}

#================================================================================
#  エイリアス考慮
#
function Filter-AliasStr( $aliasRule, $a )
{
    if ( $aliasRule.Contains( $a ) )
    {
        return $aliasRule[$a]
    }
    else
    {
        return $a
    }
}

#================================================================================
#  リリース日を得る（日英指定可）
#
function Get-ReleaseDate()
{
    $confMacroId = "release_date"
    # 英語なら ID を変える
    if($English -eq $true)
    {
        $confMacroId = "release_date_en"
    }

    # aliases_version のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $tdTags = $confObject.GetElementsByTagName("td")

    return $tdTags.Item(0).InnerText -replace "{{date}}", $releaseDate
}

#================================================================================
#  メジャーバージョンのリストを得る
#
function Get-MajorVersionList()
{
	$trTags = _Get-TableRows "changelog_major_versions"
    $majorVersions = @()

    for($i=1; $i -lt $trTags.Count; $i++)
    {
        $tdTags = $trTags[$i].GetElementsByTagName("td");
        # [0] = Major Version
        $majorVersions += $tdTags.Item(0).innerText
    }

    return $majorVersions
}

#================================================================================
#  メジャーバージョン毎の変更履歴リストの長さを得る
#
function Get-EachMajorVersionsListNum ( $majorVersion )
{
	$majorVersion = $majorVersion -replace "\.", "_"
	$trTags = _Get-TableRows ("changelog_version_" + $majorVersion)

	return $trTags.Count - 1
}

#================================================================================
#  メジャーバージョン毎の変更履歴リストを得る
#
function Get-EachMajorVersionsList( [string]$majorVersion, [int]$index)
{
	$majorVersion = $majorVersion -replace "\.", "_"
	$trTags = _Get-TableRows ("changelog_version_" + $majorVersion)

    # 行の最大数はヘッダーの一行分を抜いた数
    $maxNum = $trTags.Count - 1

    # インデックスが 0 未満や最大数を超えていたらリターン
    if ( $index -lt 0 -or $maxNum -le $index)
    {
        Write-Host "Index Error"
        return
    }

	# 指定されたインデックスの行の td タグを配列で取得
    $tdTags = $trTags[$index + 1].GetElementsByTagName("td")

    # 返り値：バージョンリストを作成
    [array]$versions = @()
    foreach($tdVersion in $tdTags.Item(0).ChildNodes)
    {
        $versions += [string]$tdVersion.innerText
    }

    # 返り値：バージョンの文字列
    $versionString = $tdTags.Item(1).innerText
	# null, empty, nbsp なら Version で置換する
	if ([string]::IsNullOrEmpty($versionString) -or $versionString -match "^ *$" )
	{
		$versionString = $versions[0]
	}

    # 返り値：日付を置換
    $packageDate = $tdTags.Item(2).innerText -replace "{{date}}", $releaseDate
    
    return [array]$versions, $versionString, $packageDate
}

#================================================================================
#  古い履歴のリストの長さを得る
#
function Get-OldChangelogListNum()
{
	$trTags = _Get-TableRows "changelog_old"

	return $trTags.Count - 1
}

#================================================================================
#  古い履歴を得る
#
function Get-OldChangelogList( $index )
{
	$trTags = _Get-TableRows "changelog_old"

    # 行の最大数はヘッダーの一行分を抜いた数
    $maxNum = $trTags.Count - 1

    # インデックスが 0 未満や最大数を超えていたらリターン
    if ( $index -lt 0 -or $maxNum -le $index)
    {
        Write-Host "Index Error"
        return
    }

	# 指定されたインデックスの行の td タグを配列で取得
    $tdTags = $trTags[$index + 1].GetElementsByTagName("td")

    # 返り値：バージョンリストを作成
    [array]$versions = @()
    foreach($tdVersion in $tdTags.Item(0).ChildNodes)
    {
        $versions += [string]$tdVersion.innerText
    }

    # 返り値：バージョンの文字列
    $versionString = $tdTags.Item(1).innerText
	# null, empty, nbsp なら Version で置換する
	if ([string]::IsNullOrEmpty($versionString) -or $versionString -match "^ *$" )
	{
		$versionString = $versions[0]
	}

    # 返り値：日付を置換
    $packageDate = $tdTags.Item(2).innerText -replace "{{date}}", $releaseDate
    
    return [array]$versions, $versionString, $packageDate
}

#================================================================================
#  内部向けの汎用関数：ID をもとにテーブルの各行の配列を返す
#
function _Get-TableRows( [string]$macroId )
{
	$confMacroId = $macroId
	# 英語なら ID を変える
	if($English -eq $true)
	{
		$confMacroId = $confMacroId + "_en"
	}

	# changelog_major_versions のテーブル内容の XML オブジェクトを取得
    $confObject = Get-XmlObject($configMacroMap[$confMacroId])
    $trTags = $confObject.GetElementsByTagName("tr")

	return $trTags
}


#================================================================================
#  メイン
#

# Alias 情報の読み込み
if ( $aliasConfPageID -ne "")
{
    [System.Xml.XmlDocument]$confObject = Get-ConfluenceData ( $aliasConfPageID )
    $aliasMacroMap = Get-StructuredMacroMap ( $confObject )
}
Write-Host ("Page ID : {0}" -f $aliasConfPageID)

# Config 情報の読み込み
if ( $configConfPageID -ne "")
{
    [System.Xml.XmlDocument]$confObject = Get-ConfluenceData ( $configConfPageID )
    $configMacroMap = Get-StructuredMacroMap ( $confObject )
}
Write-Host ("Page ID : {0}" -f $configConfPageID)

<#

$majorVersionList =  Get-MajorVersionList("")
Write-Host ("
	Get Major Version list {0}" -f $majorVersionList.Count)

foreach ( $majorVersion in $majorVersionList)
{
	$num = Get-EachMajorVersionsListNum( $majorVersion )
	Write-Host ("  Changelog Version Count : {0}" -f $num)
	#for ( $index = 0; $index -lt $num; $index++)
	#{
	#	[array]$versions, $versionString, $packageDate = Get-EachMajorVersionsList $majorVersion $index
	#	Write-Host ("    VVP : {0}, {1}, {2}" -f $versions.Count, $versionString, $packageDate)
	#}

	# メジャーバージョンごとに最新バージョンを表示
	[array]$versions, $versionString, $packageDate = Get-EachMajorVersionsList $majorVersion 0
	Write-Host ("    VVP : {0}, {1}, {2}" -f $versions.Count, $versionString, $packageDate)
}

Write-Host("=====================")

$num = Get-GetOldChangelogListNum
Write-Host ("  Old Changelog Version Count : {0}" -f $num)
for ( $index = 0; $index -lt $num; $index++)
{
	[array]$versions, $versionString, $packageDate = Get-GetOldChangelogList $index
	# Write-Host ("    OLD : {0}, {1}, {2}" -f $versions.Count, $versionString, $packageDate)
}
#>