#================================================================
# doxygen tagfile からリンク情報を抜き出して XML を作成する
#================================================================
#
#    COMMAND [-TagFile <tagfile>] [-OutFile <outputJs>] [-OutHtml <htmlFile>] [-Prefix <urlPrefix>] [-ApiDirectory <folder>] [-Package <packageName>] [-ExtraLink <extraLinkFile>] [-AliasLink <aliasFile>] [-Detail|-Quiet]
#
#    -TagFile <tagfile> はタグファイルの指定。
#    省略するとカレントの tagFile.xml を対象とする。
#
#    -OutFile <outputJs> は出力する XML ファイル名。
#    省略するとカレントの TagInfo.xml となる。
#
#    -OutHtml <outputHtml> を指定するとタグの 一覧 HTML を作成する。
#    -HtmlPrefix <prefix> はリンクのためにパスの前に付けられる。
#
#    -ExtraLink <extraLinkFile> は、追加のリンク情報を出力ファイルに
#    加えるためのものです。省略時は読み込みません。
#    なお、通常は config/ExtraLink.xml を使用します。
#
#    -AliasLink <aliasFile> は、表記を別の文字列に読み替えて
#    リンク情報を追加するためのものです。
#    例えば nn::gfx::TCommandBuffer は履歴ではしばしば nn::gfx::CommandBuffer と書かれます。
#    (未実装)
#
#    -Detail 動作の情報を詳細に表示する。省略化で、省略時は詳細を表示しない。
#    Detail と Quiet は片方しか指定できません。
#
#    -Quiet  動作の情報の表示を出来るだけ抑制する。エラーや警告だけ表示する。
#    Detail と Quiet は片方しか指定できません。
#

param
(
    [String]$TagFile = "tagFile.xml",
    [String]$OutFile = "TagInfo.xml",
    [String]$OutHtml = "",
    [String]$HtmlPrefix = "",
    [String]$ApiDirectory = "",
    [String]$Package = "",
    [String]$ExtraLink = "",
    [String]$AliasLink = "",
    [switch]$Detail,
    [switch]$Quiet
)
#----------------------------------------------------------------
$script:outLine = @()
$script:Info = @()
$script:AliasList = @()

#---------------- 表示レベル
$LogLevel = 1
if ( $Detail -eq $true -and $Quiet -eq $true )
{
    Write-Host "*** Error: Cannot -Detail and -Quiet options are specified at a time."
}
if ( $Detail -eq $true ){ $LogLevel = 2 }
if ( $Quiet  -eq $true ){ $LogLevel = 0 }

function Write-HostLog( $level, $str )
{
    if ( $level -eq "Detail" )
    {
        if ( $LogLevel -ge 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "Log" )
    {
        if ( $LogLevel -ge 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "Error" )
    {
        Write-Host $str
    }
    elseif ( $level -eq "DetailOnly" )
    {
        if ( $LogLevel -eq 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "LogOnly" )
    {
        if ( $LogLevel -eq 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "QuietOnly" )
    {
        if ( $LogLevel -eq 0 ){ Write-Host $str }
    }
}
function Write-Error( $str )
{
    Write-HostLog "Error" ("*** Error: "+$str)
}
function Write-Warning( $str )
{
    Write-HostLog "Error" ("*** Warning: "+$str)
}
function Write-Detail( $str )
{
    Write-HostLog "Detail" $str
}
function Write-Log( $str )
{
    Write-HostLog "Log" $str
}

#----------------------------------------------------------------
# 情報追加
function Add-UrlLink( $name, $html, $label )
{
    # $script:outLine += "    SetUrl( preUrl, '{0}', '{1}' )" -f $name, $url
}

#----------------------------------------------------------------
Function Create-JsClassName( $ns, $nm )
{
    if ( $ns -eq "" )
    {
#        return "ApiLink_{0}" -f ($nm -replace ":","_")
         return $nm
    }
    else
    {
#        return "ApiLink_{0}__{1}" -f ($ns -replace ":","_"), ($nm -replace ":","_")
         return $ns+"::"+$nm
    }
}

function Proc-Class( $classname, $filename, $classType )
{
    Write-Detail( "CLASS: {0} file={1}" -f $classname, $filename )
    $jsClassName = Create-JsClassName "" $classname
    Add-LinkInfo "class" $jsclassName $filename ""
}


function Proc-Member( $mem, $namespace )
{
    switch( $Mem.member.kind )
    {
        "define"
        {
            Write-Detail( "DEFINE: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName "" $Mem.member.name
            Add-LinkInfo "define" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "enumeration"
        {
            Write-Detail( "ENUM: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "enum" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "enumvalue"
        {
            Write-Detail( "ENUM VAL: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "enumval" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "friend"
        {
            Write-Detail( "FRIEND: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "friend" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "function"
        {
            Write-Detail( "FUNC: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "func" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "typedef"
        {
            Write-Detail( "TYPEDEF {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "typedef" $className $Mem.member.anchorfile $Mem.member.anchor
        }
        "variable"
        {
            Write-Detail( "VAR  {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            $className = Create-JsClassName $namespace $Mem.member.name
            Add-LinkInfo "var" $className $Mem.member.anchorfile $Mem.member.anchor
        }
    }
    return
}

#----------------------------------------------------------------
function Add-ExtraLink()
{

    if ( $ExtraLink -eq "" )
    {
        return
    }
    [XML]$Tags = Get-Content -Encoding UTF8 $ExtraLink

    for( $i=0; $i -lt $Tags.ExtraInformation.Link.Length; $i++ )
    {
        $i1 = $Tags.ExtraInformation.Link[$i].Type
        $i2 = $Tags.ExtraInformation.Link[$i].Name
        $i3 = $Tags.ExtraInformation.Link[$i].Html
        $i4 = $Tags.ExtraInformation.Link[$i].Label

        $tmp = @{ type=$i1; name=$i2; html=$i3; label=$i4 }
        $script:Info += $tmp

        Write-Detail( "EXTRALINK: {0} to {1}" -f $i2, $i3 )
    }
}

#----------------------------------------------------------------
function Add-AliasLink()
{
    if ( $AliasLink -eq "" )
    {
        return
    }
    [XML]$Tags = Get-Content -Encoding UTF8 $AliasLink

    for( $i=0; $i -lt $Tags.AliasInformation.Link.Length; $i++ )
    {
        $i1 = $Tags.AliasInformation.Link[$i].Name.InnerText
        $i2 = $Tags.AliasInformation.Link[$i].Alias.InnerText

        $tmp = @{ name=$i1; alias=$i2 }
        $script:AliasList += $tmp

        # Write-Detail( "ALIAS: {0} to {1}" -f $i1, $i2 )
    }
}

#----------------------------------------------------------------
# リンク情報追加
# そしてエイリアスを考慮して必要なら追加
function Add-LinkInfo( $type, $name, $html, $label )
{
    $tmp = @{ type=$type; name=$name; html=$html; label=$label }
    $script:Info += $tmp

    foreach( $info in $script:AliasList )
    {
        if ( ($name -replace " ","") -match "(?x:" + $info.name + ")" )
        {
           # 置き換えたデータも追加する
           $aliasedName = (($name -replace " ","") -replace $matches[0],$info.alias)
           $tmp = @{ type=$type; name=$aliasedName; html=$html; label=$label }
           $script:Info += $tmp

           Write-Detail( "Alias: {0} to {1}" -f $name, $aliasedName )
        }
    }
}

#----------------------------------------------------------------
#---- 追加リンク情報
if ( $ExtraLink -ne "" )
{
    if ( -not ( Test-Path -path $ExtraLink ) )
    {
        Write-Error( "ExtraLink file $ExtraLink not found." )
        exit 1
    }
}

#---- エイリアス情報
if ( $AliasLink -ne "" )
{
    if ( -not ( Test-Path -path $AliasLink ) )
    {
        Write-Error( "AliasLink file $AliasLink not found." )
        exit 1
    }
}

#---- エイリアス情報読み込み
Add-AliasLink

#---- Tagファイル読み込み
if ( $TagFile -eq "" )
{
    Write-Error( "Tagfile is not specified." )
    exit 1
}
if ( -not ( Test-Path -path $TagFile ) )
{
    Write-Error( "Tagfile $TagFile not found." )
    exit 1
}
[Xml]$Tags = Get-Content -Encoding UTF8 $TagFile
$Navi = $Tags.CreateNavigator()
$Compounds = $Navi.Select("/tagfile/compound[*]")

$n = 0
foreach ( $Compound in $Compounds )
{
    [Xml]$Com = $Compound.OuterXml

    if ( $Com.compound.kind -eq "class" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "class"
    }
    elseif ( $Com.compound.kind -eq "struct" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "struct"
    }
    elseif ( $Com.compound.kind -eq "namespace" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "namespace"
    }

    #---- namespace を記録
    if ( $Com.compound.namespace -eq $null )
    {
        $namespace = ""
    }
    else
    {
        if ( $Com.compound.namespace.GetType().BaseType -eq [System.Array] )
        {
            $namespaceLen = $Com.compound.namespace.Length
            $namespace = $Com.compound.namespace[ $namespaceLen - 1 ]
        }
        else
        {
            $namespace = $Com.compound.namespace
        }
    }

    #---- member タグを順次処理する
    $Members = $Compound.Select("member")
    foreach ( $Member in $Members )
    {
        if ( $Com.compound.kind -eq "file" )
        {
            [Xml]$Mem = $Member.OuterXml
            Proc-Member $Mem $namespace $Com.kind
        }
        else
        {
            [Xml]$Mem = $Member.OuterXml
            Proc-Member $Mem $Com.compound.name $Com.compound.kind
        }

        $n ++
    }
}

#---- 別ファイルからの追加情報
Add-ExtraLink

#---- ソート
$sortKey=@{Expression={$_.Item("name")};ascending=$true}
$Info_sorted = ( $script:Info | Sort-Object ($sortKey) -unique )

#----------------
# 結果出力
New-Item -force -ItemType file $OutFile | Out-Null
"<?xml version='1.0' encoding='UTF-8' standalone='yes' ?>" | Out-File -Encoding UTF8 $OutFile -Append
"<TagInformation>" | Out-File -Encoding UTF8 $OutFile -Append

# パッケージ名
if ( $Package -eq "" )
{
    $Package = "NintendoSDK"
}
"<Package>$Package</Package>" | Out-File -Encoding UTF8 $OutFile -Append

# ディレクトリ
if ( $ApiDirectory -eq "" )
{
    $ApiDirectory = "Api/Html/"
}
elseif ( $ApiDirectory -match "[^/]$" )
{
    $ApiDirectory = $ApiDirectory + "/"
}

"<ApiDirectory>$ApiDirectory</ApiDirectory>" | Out-File -Encoding UTF8 $OutFile -Append

# 日付
$dateStr = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
"<Date>$dateStr</Date>" | Out-File -Encoding UTF8 $OutFile -Append

"<LinkList>" | Out-File -Encoding UTF8 $OutFile -Append
foreach ( $info in $Info_sorted )
{
    ( "<Link><Type>{0}</Type><Name><![CDATA[{1}]]></Name><Html>{2}</Html><Label>{3}</Label></Link>" `
        -f $info.type, $info.name, $info.html, $info.label) | Out-File -Encoding UTF8 $OutFile -Append
}
"</LinkList>" | Out-File -Encoding UTF8 $OutFile -Append
"</TagInformation>" | Out-File -Encoding UTF8 $OutFile -Append

#---- 必要なら HTML出力
if ( $OutHtml -ne "" )
{
New-Item -force -ItemType file $OutHtml | Out-Null
"<html>" | Out-File -Encoding UTF8 $OutHtml -Append
"<body>" | Out-File -Encoding UTF8 $OutHtml -Append

foreach ( $info in $Info_sorted )
{
    if ( $info.label -eq "" )
    {
        $url = $info.html
        $path = $HtmlPrefix + $ApiDirectory + $info.html
    }
    else
    {
        $url = $info.html + "#" + $info.label
        $path = $HtmlPrefix + $ApiDirectory + $info.html + "#" + $info.label
    }
   ( "<a href=`"{3}`">{0}</a> ({1} , <small>{2}</small>)<br>" -f $info.name, $info.type, $url, $path ) | Out-File -Encoding UTF8 $OutHtml -Append
}

"</body>" | Out-File -Encoding UTF8 $OutHtml -Append
"</html>" | Out-File -Encoding UTF8 $OutHtml -Append
}

Write-Detail( "Done. $n items." )
