#================================================================
# doxygen tagfile から タグ情報 JSファイルを作成する
#================================================================
#
#    COMMAND [-TagFile <tagfile>] [-OutFile <outputJs>] [-OutHtml <outputHtml>] [-UrlPrefix <urlPrefix>] [-English] [-Detail]
#
#    -TagFile <tagfile> はタグファイルの指定。
#    省略するとカレントの tagFile.xml を対象とする。
#
#    -OutFile <outputJs> は出力する JavaScriptファイル名。
#    省略するとカレントの tag.js となる。
#
#    -OutHtml <outputHtml>を指定するとタグの 一覧 HTML を作成する。
#    その際、指定があれば <urlPrefix> をプレフィックスとして付加する。
#    このプレフィックスはここれではこの一覧に使用されるだけである。
#

param
(
    [String]$TagFile = "tagFile.xml",
    [String]$OutFile = "tag.js",
    [String]$OutHtml,
    [String]$UrlPrefix,
    [switch]$English,
    [switch]$Detail
)
#----------------------------------------------------------------
$outLine = @()

#----------------------------------------------------------------
# UrlPrefix 指定
if ( $UrlPrefix -eq "" )
{
    $UrlPrefix = "./"
}
if ( $UrlPrefix -match "[^/]$" )
{
    $UrlPrefix = $UrlPrefix + "/"
}

Function Create-JsClassName( $ns, $nm )
{
    if ( $ns -eq "" )
    {
        return "ApiLink_{0}" -f ($nm -replace ":","_")
    }
    else
    {
        return "ApiLink_{0}__{1}" -f ($ns -replace ":","_"), ($nm -replace ":","_")
    }
}

function Proc-Class( $classname, $filename, $classType )
{
    if ( $Detail -eq $true )
    {
        Write-Host ("CLASS: {0} file={1}" -f $classname, $filename)
    }
    if ( $OutHtml -ne "" )
    {
        "{2}:  <a href=`"{3}{1}`">{0}</a><br>" -f $classname, $filename, $classType.ToUpper(), $urlPrefix | Out-File $OutHtml -Append
    }
    $jsClassName = Create-JsClassName "" $classname
    $script:outLine += "    SetUrl( preUrl, '{0}', '{1}' )" -f $jsClassName, $filename
}


function Proc-Member( $mem, $namespace, $comKind, $comName )
{
    switch( $Mem.member.kind ){
        "define"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ("DEFINE: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "DEFINE:  <a href=`"{4}{3}{1}#{2}`">{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName "" $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
        "enumeration"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ( "ENUM: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "ENUM:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor, $urlPreFix
        }
        "enumvalue"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ( "ENUM VAL: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "ENUM VAL:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
        "friend"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ( "FRIEND: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "FRIEND:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
        "function"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ("FUNC: {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "FUNC:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
        "typedef"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ("TYPEDEF {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "TYPEDEF:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
        "variable"
        {
            if ( $Detail -eq $true )
            {
                Write-Host ("VAR  {3}::{0} file={1} anc={2} " -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace )
            }
            if ( $OutHtml -ne "" )
            {
                "VAR:  <a href=`"{4}{1}#{2}`">{3}::{0}</a><br>" -f $Mem.member.name, $Mem.member.anchorfile, $Mem.member.anchor, $namespace, $urlPrefix | Out-File $OutHtml -Append
            }
            $className = Create-JsClassName $namespace $Mem.member.name
            $script:outLine += "    SetUrl( preUrl, '{0}', '{1}#{2}' )" -f $className, $Mem.member.anchorfile, $Mem.member.anchor
        }
    }
    return
}

#----------------------------------------------------------------
if ( $TagFile -eq "" )
{
    Write-Host "*** Tagfile $TagFile not found."
    exit 1
}
[Xml]$Tags = Get-Content -Encoding UTF8 $TagFile
$Navi = $Tags.CreateNavigator()
$Compounds = $Navi.Select("/tagfile/compound[*]")


#---- 出力ファイル
if ( $OutHtml -ne "" )
{
    New-Item -force -ItemType file $OutHtml | Out-Null
}

$n = 0
foreach ( $Compound in $Compounds )
{
    [Xml]$Com = $Compound.OuterXml

    if ( $Com.compound.kind -eq "class" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "class"
    }
    elseif ( $Com.compound.kind -eq "struct" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "struct"
    }
    elseif ( $Com.compound.kind -eq "namespace" )
    {
        Proc-Class $Com.compound.name $Com.compound.filename "namespace"
    }

    #---- namespace を記録
    if ( $Com.compound.namespace -eq $null )
    {
        $namespace = ""
    }
    else
    {
        if ( $Com.compound.namespace.GetType().BaseType -eq [System.Array] )
        {
            $namespaceLen = $Com.compound.namespace.Length
            $namespace = $Com.compound.namespace[ $namespaceLen - 1 ]
        }
        else
        {
            $namespace = $Com.compound.namespace
        }
    }

    #---- member タグを順次処理する
    $Members = $Compound.Select("member")
    foreach ( $Member in $Members )
    {
        if ( $Com.compound.kind -eq "file" )
        {
            [Xml]$Mem = $Member.OuterXml
            Proc-Member $Mem $namespace $Com.kind
        }
        else
        {
            [Xml]$Mem = $Member.OuterXml
            Proc-Member $Mem $Com.compound.name $Com.compound.kind
        }

        $n ++
    }
}

#---- ソート
$outLine = $outLine | Sort-Object | Get-Unique
if ( $outLine -eq $null )
{
    $outLine = @()
}

#---- テンプレート読み込み
$scriptPath = Split-Path -parent $MyInvocation.MyCommand.Definition

if ( $English -eq $false )
{
    $template = Get-Content -Encoding UTF8 -Path "$scriptPath\template\ja\apiLink_reference.js"
}
else
{
    $template = Get-Content -Encoding UTF8 -Path "$scriptPath\template\en\apiLink_reference.js"
}
$template = $template -replace "_\$\$@\(URL_PATTERN\)", [string]::join("`r`n", $outLine)

New-Item -force -ItemType file $OutFile | Out-Null
$template | Out-File -Encoding UTF8 $OutFile


if ( $Detail -eq $true )
{
    Write-Host "Done. $n items."
}
