﻿namespace PackageCreator
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using YamlDotNet.RepresentationModel;

    /// <summary>
    /// バンドルファイルを読み込むクラスです。
    /// </summary>
    public class BundleDefinition
    {
        #region コンストラクタ
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="bundlePath">読み込むバンドルファイルのパス</param>
        public BundleDefinition(string bundlePath, string rootPath, IFileGetter fileGetter)
        {
            this.BundleDefintionFilePath = bundlePath;
            this.Read(this.BundleDefintionFilePath, rootPath, fileGetter);
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// バンドルファイルのパスを表します。
        /// </summary>
        public string BundleDefintionFilePath { get; private set; }

        /// <summary>
        /// 公開タイプを表します。
        /// </summary>
        public PackageType.Type Type { get; private set; }

        /// <summary>
        /// バンドルに含まれるファイルリストを表します。
        /// </summary>
        public IEnumerable<string> Files { get; private set; }

        /// <summary>
        /// バンドルが内包するバンドルを示します。
        /// </summary>
        public IEnumerable<string> Bundles { get; private set; }

        /// <summary>
        /// 発見できなかったファイルのリストです。
        /// </summary>
        public IReadOnlyList<string> UnresolvedPaths { get; private set; }
        #endregion

        #region ワイルドカードの展開
        private IEnumerable<string> ExpandWildCard(string filepath, string rootPath, IFileGetter fileGetter)
        {
            // FileTree.GetFiles の結果は絶対パスのため Substring(rootPath.Length).TrimStart('\\') で相対パスに変換
            return from path in fileGetter.GetFiles(Path.Combine(rootPath, filepath))
                   select path.Substring(rootPath.Length).TrimStart('\\');
        }
        #endregion

        #region バンドルファイルの読み込み
        private void Read(string bundlePath, string rootPath, IFileGetter fileGetter)
        {
            try
            {
                Dictionary<string, object> data;
                using (StreamReader reader = new StreamReader(bundlePath))
                {
                    data = YamlDataReader.Read(reader, "Bundle");
                }

                this.Type = PackageType.Convert((string)data["Type"]);
                this.Bundles = data.ContainsKey("Bundles") ? (data["Bundles"] as List<object>).ConvertFromSequencialNode() : new List<string>();

                var fileList = (data["Files"] as List<object>).ConvertFromSequencialNode();
                var excludeList = data.ContainsKey("ExcludeName")
                    ? (data["ExcludeName"] as List<object>).ConvertFromSequencialNode().Select(path => path.Replace('/', Path.DirectorySeparatorChar)).ToArray()
                    : Enumerable.Empty<string>();

                var files = new List<string>();
                var unresolved = new List<string>(0);
                foreach (var pattern in fileList)
                {
                    var expanded = from path in ExpandWildCard(pattern, rootPath, fileGetter)
                                   where !excludeList.Any(path.Contains)
                                   select path;
                    var expanded2 = expanded.ToList();
                    if (expanded2.Count == 0)
                    {
                        unresolved.Add(pattern);
                    }
                    files.AddRange(expanded2);
                }
                this.Files = files;
                this.UnresolvedPaths = unresolved;
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: Bundle definition file has invalid format.");
                Console.Error.WriteLine($"\tFailed to read '${this.BundleDefintionFilePath}'.");
                throw;
            }
        }
        #endregion
    }
}
