﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using System.IO.Ports;

namespace NxAgingHelper
{
    public partial class Form1 : Form
    {
        private AgingManager m_AgingManager;
        private UartLogger m_UartLogger;
        private StringCircularBuffer m_UartLogBuffer;
        private RegExWatcher m_RegExWatcher;

        public Form1()
        {
            InitializeComponent();

            // COM ポート初期化
            RefreshComPortList();

            // ログファイル名初期化
            BaseLogFilePathTextBox.Text = string.Format("{0}_{1}_", DateTime.Now.ToString("yyyyMMdd"), Environment.UserName);

            // 正規表現系の UI 初期化
            InitializeRegExUI();

            m_AgingManager = new AgingManager();
            m_AgingManager.ExceptionHandleEvent += e =>
            {
                MessageBox.Show(e.Message);
            };

            m_UartLogger = new UartLogger();
            m_UartLogger.ExceptionHandleEvent += e =>
            {
                MessageBox.Show(e.Message);
            };
            m_UartLogger.ReceiveHandleEvent += s =>
            {
                AddUartText(s);
            };
            m_RegExWatcher = new RegExWatcher();
            m_UartLogger.ReceiveHandleEvent += s =>
            {
                WatchLogWithRegEx(s);
            };

            m_UartLogBuffer = new StringCircularBuffer();

            StopEventButton.Enabled = false;
            StopLoggingButton.Enabled = false;
        }

        private void RefreshComPortList()
        {
            // コンボボックスに COM ポート一覧を設定
            var portNames = SerialPort.GetPortNames();
            McciComboBox.Items.Clear();
            McciComboBox.Items.AddRange(portNames);
            EdevComboBox.Items.Clear();
            EdevComboBox.Items.AddRange(portNames);
            HdmiComboBox.Items.Clear();
            HdmiComboBox.Items.AddRange(portNames);
        }

        private delegate void AddUartTextCallback(string text);
        private void AddUartText(string text)
        {
            if (this.InvokeRequired)
            {
                m_UartLogBuffer.Append(text);
                var d = new AddUartTextCallback(AddUartText);
                this.Invoke(d, new object[] { text });
            }
            else
            {
                UartLogTextBox.Text = m_UartLogBuffer.Get();
                UartLogTextBox.SelectionStart = UartLogTextBox.Text.Length;
                UartLogTextBox.ScrollToCaret();
            }
        }

        private void Refresh_Click(object sender, EventArgs e)
        {
            RefreshComPortList();
        }

        private void StartButton_Click(object sender, EventArgs e)
        {
            // エラーチェック
            if (McciCheckBox.Checked && McciComboBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("MCCI 3101 を接続している COM ポートを選択してください。");
                return;
            }
            if (HdmiComPortCheckBox.Checked && HdmiComboBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("Hdmi Switcher を接続している COM ポートを選択してください。");
                return;
            }
            if (HdmiIpAddresCheckBox.Checked && HdmiIpTextBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("Hdmi Switcher の IP アドレスを入力してください。");
                return;
            }
            if (ConfigurationFilePathTextBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("Configuration File Path を入力してください。");
                return;
            }
            if ((SleepCheckBox.Checked || SdevUsbCheckBox.Checked) && SdevIpTextBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("SDEV IP Address を入力してください。");
                return;
            }
            if (BaseLogFilePathTextBox.Text.Equals(string.Empty))
            {
                MessageBox.Show("Log File Path を入力してください。");
                return;
            }

            // Configuration 読み込み
            var parser = new CommandYamlParser();

            // チェックボックスをチェックしていない場合は、パースの段階で各コマンドを無効化
            parser.EnableMcci = McciCheckBox.Checked;
            parser.EnableHdmi = HdmiIpAddresCheckBox.Checked || HdmiComPortCheckBox.Checked;
            parser.EnablePowerUsb = PowerUsbCheckBox.Checked;
            parser.EnableSdevUsb = SdevUsbCheckBox.Checked;
            parser.EnablePowerButton = SleepCheckBox.Checked;
            parser.EnableBatteryLevelEmulation = BatteryLevelEmulationCheckBox.Checked;

            ExecuteCommand command;
            try
            {
                command = parser.Parse(ConfigurationFilePathTextBox.Text);
            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message);
                return;
            }

            // AgingManager のプロパティ設定
            m_AgingManager.UseHdmi = HdmiIpAddresCheckBox.Checked || HdmiComPortCheckBox.Checked;
            m_AgingManager.UseMcci = McciCheckBox.Checked;
            m_AgingManager.UsePowerUsb = PowerUsbCheckBox.Checked;
            m_AgingManager.UseSdev = SdevRadioButton.Checked && (SdevUsbCheckBox.Checked || SleepCheckBox.Checked || BatteryLevelEmulationCheckBox.Checked);
            m_AgingManager.UseBattery = BatteryLevelEmulationCheckBox.Checked;

            var useHdmiComPort = !HdmiIpAddresCheckBox.Checked && HdmiComPortCheckBox.Checked;

            // エージング開始
            try
            {
                m_AgingManager.Start(command, EventLogFilePathTextBox.Text, SdevIpTextBox.Text, McciComboBox.Text, useHdmiComPort ? HdmiComboBox.Text : HdmiIpTextBox.Text);
            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message);

                // エージング開始後に例外が発生した際、正しくリソースを解放する処理を実装できていないので、
                // アプリケーションが不正な状態となる。そのため、アプリケーションを丸ごと終了させる。
                //
                // TODO: 正しいリソース解放の実装
                Application.Exit();
            }

            // ウィンドウタイトル変更
            this.Text = "[Now Aging...] NX Aging Helper";

            // Start / Stop ボタンの Enable 切り替え
            StartEventButton.Enabled = false;
            StopEventButton.Enabled = true;

            // 正規表現系のリフレッシュ
            RefreshRegExUI();
        }

        private void StopButton_Click(object sender, EventArgs e)
        {
            // ウィンドウタイトル変更
            this.Text = "[Finishing...] Device Driver Aging Helper";

            // エージング終了
            m_AgingManager.Stop();

            // ウィンドウタイトル変更
            this.Text = "NX Aging Helper";

            // Start / Stop ボタンの Enable 切り替え
            StartEventButton.Enabled = true;
            StopEventButton.Enabled = false;
        }

        private void BaseLogFilePathTextBox_TextChanged(object sender, EventArgs e)
        {
            var input = BaseLogFilePathTextBox.Text;
            EventLogFilePathTextBox.Text = input + "_Event.log";
            UartLogFilePathTextBox.Text = input + "_Uart.log";

            bool isSdev = input.Contains("SDEV") && !input.Contains("EDEV");
            bool isEdev = !input.Contains("SDEV") && input.Contains("EDEV");

            // ラジオボタンが未選択なら、LogFilePath から設定内容を推測
            if (!SdevRadioButton.Checked && !EdevRadioButton.Checked)
            {
                if (isSdev)
                {
                    SdevRadioButton.Checked = true;
                    EdevRadioButton.Checked = false;
                    SetUIForSdev();
                }
                if (isEdev)
                {
                    SdevRadioButton.Checked = false;
                    EdevRadioButton.Checked = true;
                    SetUIForEdev();
                }
            }

            // チェックボックスがすべて未選択なら、LogFilePath から設定内容を推測
            if (!IsAnyCheckBoxChecked())
            {
                if (input.Contains("DOCK"))
                {
                    if (isSdev)
                    {
                        SdevUsbCheckBox.Checked = true;
                    }
                    else
                    {
                        McciCheckBox.Checked = true;
                    }
                }
                if (input.Contains("SLEEP") && isSdev)
                {
                    SleepCheckBox.Checked = true;
                }
                if (input.Contains("HDMI"))
                {
                    HdmiIpAddresCheckBox.Checked = true;
                }
            }
        }

        private bool IsAnyCheckBoxChecked()
        {
            return
                McciCheckBox.Checked ||
                HdmiIpAddresCheckBox.Checked ||
                HdmiComPortCheckBox.Checked ||
                PowerUsbCheckBox.Checked ||
                SdevUsbCheckBox.Checked ||
                SleepCheckBox.Checked;
        }

        private void SetUIForSdev()
        {
            SdevUsbCheckBox.Enabled = true;
            SleepCheckBox.Enabled = true;
            SdevIpTextBox.Enabled = true;
            BatteryLevelEmulationCheckBox.Enabled = true;
            EdevComboBox.Enabled = false;
        }

        private void SetUIForEdev()
        {
            SdevUsbCheckBox.Enabled = false;
            SleepCheckBox.Enabled = false;
            SdevIpTextBox.Enabled = false;
            BatteryLevelEmulationCheckBox.Enabled = false;
            EdevComboBox.Enabled = true;
        }

        private void SdevRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (SdevRadioButton.Checked)
            {
                SetUIForSdev();
            }
        }

        private void EdevRadioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (EdevRadioButton.Checked)
            {
                SetUIForEdev();
            }
        }

        private void StartLoggingButton_Click(object sender, EventArgs e)
        {
            if (!SdevRadioButton.Checked && !EdevRadioButton.Checked)
            {
                MessageBox.Show("SDEV または EDEV を選択してください。");
                return;
            }

            if (SdevRadioButton.Checked)
            {
                if (SdevIpTextBox.Text.Equals(string.Empty))
                {
                    MessageBox.Show("SDEV IP Address を入力してください。");
                    return;
                }

                try
                {
                    m_UartLogger.StartForSdev(SdevIpTextBox.Text, UartLogFilePathTextBox.Text);
                }
                catch (Exception exception)
                {
                    MessageBox.Show(exception.Message);
                    return;
                }
            }
            else if (EdevRadioButton.Checked)
            {
                if (EdevComboBox.Text.Equals(string.Empty))
                {
                    MessageBox.Show("EDEV COM Port を選択してください。");
                    return;
                }
                try
                {
                    m_UartLogger.StartForEdev(EdevComboBox.Text, UartLogFilePathTextBox.Text);
                }
                catch (Exception exception)
                {
                    MessageBox.Show(exception.Message);
                    return;
                }
            }

            StartLoggingButton.Enabled = false;
            StopLoggingButton.Enabled = true;
        }

        private void StopLoggingButton_Click(object sender, EventArgs e)
        {
            m_UartLogger.Stop();

            StartLoggingButton.Enabled = true;
            StopLoggingButton.Enabled = false;
        }

        private void HdmiCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            System.Windows.Forms.CheckBox checkBox;
            if (sender.GetType() == HdmiIpAddresCheckBox.GetType())
            {
                if (HdmiIpAddresCheckBox.Checked && HdmiComPortCheckBox.Checked)
                {
                    checkBox = (System.Windows.Forms.CheckBox)sender;
                    if (checkBox.Name == HdmiIpAddresCheckBox.Name)
                    {
                        HdmiComPortCheckBox.Checked = !HdmiIpAddresCheckBox.Checked;
                    }
                    else if (checkBox.Name == HdmiComPortCheckBox.Name)
                    {
                        HdmiIpAddresCheckBox.Checked = !HdmiComPortCheckBox.Checked;
                    }
                }
                HdmiIpTextBox.Enabled = !HdmiComPortCheckBox.Checked;
                HdmiComboBox.Enabled = !HdmiIpAddresCheckBox.Checked;
            }
        }

        // 正規表現系の変数
        private const int RegExItemNum = 3;
        private const int MatchFuncNum = 1;
        private System.Windows.Forms.CheckBox[] m_RegExCheckBoxes;
        private System.Windows.Forms.TextBox[] m_RegExTextBoxes;
        private System.Windows.Forms.ComboBox[] m_RegExComboBoxes;
        private string[] m_RegExActionString;
        private RegExWatcher.MatchFunction[] m_RegExMatchFuncs;
        private Color m_RegExDefaultBgColor;

        // 正規表現系の UI の初期化
        private void InitializeRegExUI()
        {
            m_RegExCheckBoxes = new System.Windows.Forms.CheckBox[RegExItemNum];
            m_RegExTextBoxes = new System.Windows.Forms.TextBox[RegExItemNum];
            m_RegExComboBoxes = new System.Windows.Forms.ComboBox[RegExItemNum];

            // UI 要素と対応付ける（UI を増やしたらここも追加する）
            m_RegExCheckBoxes[0] = RegExWatchCheckBox1;
            m_RegExTextBoxes[0] = RegExWatchTextBox1;
            m_RegExComboBoxes[0] = RegExWatchComboBox1;
            m_RegExCheckBoxes[1] = RegExWatchCheckBox2;
            m_RegExTextBoxes[1] = RegExWatchTextBox2;
            m_RegExComboBoxes[1] = RegExWatchComboBox2;
            m_RegExCheckBoxes[2] = RegExWatchCheckBox3;
            m_RegExTextBoxes[2] = RegExWatchTextBox3;
            m_RegExComboBoxes[2] = RegExWatchComboBox3;

            // マッチしたときアクション名（アクションを増やしたら追加する）
            m_RegExActionString = new String[MatchFuncNum];
            m_RegExActionString[0] = "Stop Aging.";

            // コンボボックス（ドロップダウンリスト）に登録
            foreach (ComboBox regExComboBox in m_RegExComboBoxes)
            {
                regExComboBox.Items.Clear();
                foreach (String actionString in m_RegExActionString)
                {
                    regExComboBox.Items.Add(actionString);
                }
                regExComboBox.SelectedIndex = 0;
                regExComboBox.Refresh();
            }
            // ドロップダウンリストと関数の対応付け（アクションを増やしたらここも追加する）
            m_RegExMatchFuncs = new RegExWatcher.MatchFunction[m_RegExComboBoxes[0].MaxDropDownItems];
            m_RegExMatchFuncs[m_RegExComboBoxes[0].Items.IndexOf(m_RegExActionString[0])] = RegExWatch_StopAging;

            // 初期背景色の取得
            m_RegExDefaultBgColor = m_RegExComboBoxes[0].BackColor;
        }

        // 正規表現系 UI のリフレッシュ（エージング実行ごとに行うリセット）
        private void RefreshRegExUI()
        {
            // 背景色を元に戻す（合致した場合に背景色を変えたのを元に戻す）
            foreach(TextBox regExTextBox in m_RegExTextBoxes)
            {
                regExTextBox.BackColor = m_RegExDefaultBgColor;
            }
            m_RegExWatcher.Refresh();
        }

        // 各行のチェックボックスを変更した際の処理
        private void RegExWatchCheckBox1_CheckedChanged(object sender, EventArgs e)
        {
            ControlRegExWatch(0);
        }
        private void RegExWatchCheckBox2_CheckedChanged(object sender, EventArgs e)
        {
            ControlRegExWatch(1);
        }
        private void RegExWatchCheckBox3_CheckedChanged(object sender, EventArgs e)
        {
            ControlRegExWatch(2);
        }
        // チェックボックスの有効・無効に対応した処理を実行する
        private void ControlRegExWatch(int id)
        {
            if (m_RegExCheckBoxes[id].Checked)
            {
                // チェックが入った

                // 正規表現欄が空の場合は対象外
                if (m_RegExTextBoxes[id].Text == "")
                {
                    // チェックボックスを空にして抜ける（チェックボックスをチェックできない、と見える）
                    m_RegExCheckBoxes[id].Checked = false;
                    return;
                }

                // 正規表現と対応する処理を登録する
                m_RegExWatcher.RegisterRegExFunc(id, m_RegExTextBoxes[id].Text, m_RegExMatchFuncs[m_RegExComboBoxes[id].SelectedIndex]);

                // コントロールの制御を無効化する
                m_RegExTextBoxes[id].Enabled = false;
                m_RegExComboBoxes[id].Enabled = false;
            }
            else
            {
                // チェックが外れた
                // 対象 ID の正規表現情報を削除する
                m_RegExWatcher.ReleaseRegExFunc(id);

                // コントロールの制御を有効化する
                m_RegExTextBoxes[id].Enabled = true;
                m_RegExComboBoxes[id].Enabled = true;
            }
        }

        // ログが出力されたら呼び出されるハンドラ
        private delegate void WatchLogWithRegExCallback(string text);
        private void WatchLogWithRegEx(string text)
        {
            // コントロール権限の都合から呼び出しスレッドを変更する
            if (this.InvokeRequired)
            {
                var d = new WatchLogWithRegExCallback(WatchLogWithRegEx);
                this.Invoke(d, new object[] { text });
            }
            else
            {
                // 正規表現とのマッチングと、それに応じた処理を実行する
                m_RegExWatcher.WatchLogWithRegEx(text);
            }
        }

        // エージングを停止させる
        // 内容は基本的に StopButton_Click の内容のコピー
        private void RegExWatch_StopAging(int matchId)
        {
            // マッチした正規表現欄の背景色を変える
            m_RegExTextBoxes[matchId].BackColor = Color.Red;

            // エージングを停止させる
            if (m_AgingManager.IsStarted)
            {
                // ウィンドウタイトル変更
                this.Text = "[Finishing...] Device Driver Aging Helper";

                // エージング終了
                m_AgingManager.Stop();

                // ウィンドウタイトル変更
                this.Text = "NX Aging Helper";

                // Start / Stop ボタンの Enable 切り替え
                StartEventButton.Enabled = true;
                StopEventButton.Enabled = false;
            }
        }
    }
}
