﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.Shell.Interop;
using Nintendo.NintendoSdkVsExtension.Base;
using Nintendo.NintendoSdkVsExtension.Models;
using Nintendo.NintendoSdkVsExtension.Resources;
using Nintendo.NintendoSdkVsExtension.Shell;
using Nintendo.NintendoSdkVsExtension.VcAccessors;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Logic
{
    public class FailedToApplyConfiguration
    {
        public VcProjectState Project { get; }
        public SdkConfigurationSettingsStore ExpectedConfiguration { get; }
        public ConfigurationPair ConfigurationPair { get; }
        public string ActualSdkRoot { get; }
        public string ActualSpec { get; }
        public string ActualBuildType { get; }

        public FailedToApplyConfiguration(
            VcProjectState expectedProject,
            ConfigurationPair configurationPair,
            SdkConfigurationSettingsStore expectedConfiguration,
            string actualSdkRoot,
            string actualSpec,
            string actualBuildType)
        {
            this.Project = expectedProject;
            this.ConfigurationPair = configurationPair;
            this.ExpectedConfiguration = expectedConfiguration;
            this.ActualSdkRoot = actualSdkRoot;
            this.ActualSpec = actualSpec;
            this.ActualBuildType = actualBuildType;
        }
    }

    // TODO: 雑多すぎるのを修正
    public static class SdkConfigurationManagerUtil
    {
        public static Models.SdkSolution ToSolutionModel(IEnumerable<VcProjectState> vcProjects)
        {
            return new Models.SdkSolution(
                vcProjects.Select(x => new SdkVcProject(x.ProjectFilePath, x.Settings)).ToArray());
        }

        public static IEnumerable<VcProjectState> ReadProjectSettings(IServiceProvider serviceProvider)
        {
            return VsSolutionUtil.EnumerateVcxproj(serviceProvider)
                .Select(h => new VcProjectState(VsHierarchyUtil.GetProjectPath(h), ReadProjectSettingsStore(h)))
                .ToArray();
        }

        public static IEnumerable<VcProjectState> ReadProjectSettingsFromVcxproj(IServiceProvider serviceProvider)
        {
            return VsSolutionUtil.EnumerateVcxproj(serviceProvider)
                .Select(h => new VcProjectState(VsHierarchyUtil.GetProjectPath(h), VcProjectConfigurationUtil.ReadSdkProjectSettingsStoreFromVcxproj(h)))
                .ToArray();
        }

        public static SdkProjectSettingsStore ReadProjectSettingsStore(IVsHierarchy h)
        {
            var projectPath = VsHierarchyUtil.GetCanonicalName(h);
            string projectXmlPath = GetProjectXmlPath(projectPath);
            // .nnsdk.xml があれば .nnsdk.xml から取得する
            if (File.Exists(projectXmlPath))
            {
                var settingsFromProject = VcProjectConfigurationUtil.ReadSdkProjectSettingsStoreFromVcxproj(h);
                var settingsFromXml = LoadProjectSettingsStore(projectXmlPath); // TODO: エラーハンドリング

                return MergeProjectSettings(settingsFromProject, settingsFromXml);
            }
            else
            {
                return VcProjectConfigurationUtil.ReadSdkProjectSettingsStoreFromVcxproj(h);
            }
        }

        // TODO: このロジックを分類
        private static string GetProjectXmlPath(string projectPath)
        {
            return Path.ChangeExtension(projectPath, ".nnsdk.xml");
        }

        /// <summary>
        /// プロジェクトから取得できない値 (IsSdkEnabled でない構成の値) を、 nnsdk.xml から取得した値にする
        /// </summary>
        private static SdkProjectSettingsStore MergeProjectSettings(SdkProjectSettingsStore settingsFromProject, SdkProjectSettingsStore settingsFromXml)
        {
            var mergedSettings = new SdkProjectSettingsStore(settingsFromProject);

            foreach (var c in settingsFromProject.Configurations)
            {
                var configurationPair = c.Key;
                var configurationSettingsFromProject = c.Value;
                if (!configurationSettingsFromProject.IsSdkEnabled
                    && settingsFromXml.Configurations.ContainsKey(configurationPair))
                {
                    mergedSettings.RemoveConfiguration(configurationPair);
                    mergedSettings.AddConfiguration(new SdkConfigurationSettingsStore(settingsFromXml.Configurations[configurationPair]));
                }
                else
                {
                    // settingsFromXmlsettingsFromProject のものを採用
                }
            }
            return mergedSettings;
        }

        private static SdkProjectSettingsStore LoadProjectSettingsStore(string projectXmlPath)
        {
            var serializer = new Serialization.V1.SdkProjectSettingsSerializer();
            using (var fs = File.OpenRead(projectXmlPath))
            {
                return serializer.Deserialize(fs).ToStore();
            }
        }

        // FIXME: 途中で失敗したらどうすんのこれ
        public static void SaveProjectSettings(Models.SdkSolution solution)
        {
            // プロジェクトに反映する
            SdkConfigurationManagerUtil.ApplyConfigurationsToVcxproj(solution);

            // XML に出す
            foreach (var p in solution.VcProjects)
            {
                var projectXmlPath = GetProjectXmlPath(p.ProjectFilePath);
                if (File.Exists(projectXmlPath) || p.ProjectConfigurations.Any(x => x.IsSdkEnabled))
                {
                    SdkConfigurationManagerUtil.SaveProjectSettingsStore(GetProjectXmlPath(p.ProjectFilePath), p.BackingStore);
                }
            }
        }

        private static void SaveProjectSettingsStore(string projectXmlPath, SdkProjectSettingsStore settings)
        {
            var serializer = new Serialization.V1.SdkProjectSettingsSerializer();
            using (var ms = new MemoryStream())
            {
                serializer.Serialize(ms, new Serialization.V1.SerializationSdkProjectSettings(settings));
                File.WriteAllBytes(projectXmlPath, ms.ToArray());
            }
        }

        public static bool IsUpgradingRequired(IServiceProvider serviceProvider)
        {
            return VsSolutionUtil.EnumerateVcxproj(serviceProvider)
                .Any(x => IsUpgradingRequired(x));
        }

        public static bool IsUpgradingRequired(IVsHierarchy vcxprojHierarchy)
        {
            var vcp = new VcAccessors.VcProjectAccessor(vcxprojHierarchy);
            foreach (var c in vcp.Configurations)
            {
                if (c.PropertySheets.Any(p => IsV0SdkPropertySheet(p.Name)))
                {
                    return true;
                }
            }
            return false;
        }

        public static bool IsV0SdkPropertySheet(string name)
        {
            return
                name.StartsWith("NintendoSdkVcProjectSettings") ||
                name.StartsWith("NintendoSdkSpec_") ||
                name.StartsWith("NintendoSdkBuildType_");
        }

        public static IEnumerable<FailedToApplyConfiguration> EnumerateFailedToApplyConfigurations(
            IEnumerable<VcProjectState> expectedSettings, IEnumerable<IVsHierarchy> projectHierarchies)
        {
            var vcProjects = projectHierarchies.Select(x => new VcProjectAccessor(x)).ToArray();
            foreach (var expectedVcProject in expectedSettings)
            {
                var currentVcProject = vcProjects.FirstOrDefault(
                    x => x.ProjectFilePath == expectedVcProject.ProjectFilePath);
                if (currentVcProject == null)
                {
                    continue;
                }

                foreach (var expectedConfiguration in expectedVcProject.Settings.Configurations.Values)
                {
                    var currentConfiguration = currentVcProject.Configurations.FirstOrDefault(
                        x => x.ConfigurationPair == expectedConfiguration.ConfigurationPair);
                    if (currentConfiguration == null)
                    {
                        continue;
                    }

                    var currentSdkRoot = currentConfiguration.GetEvaluatedPropertyValue("NintendoSdkRoot");
                    var currentSpec = currentConfiguration.GetEvaluatedPropertyValue("NintendoSdkSpec");
                    var currentBuildType = currentConfiguration.GetEvaluatedPropertyValue("NintendoSdkBuildType");
                    // TODO: SdkRoot の判定はこれでよいのか
                    if (expectedConfiguration.IsSdkEnabled &&
                            (EnsureEndsWithPathSeparator(expectedConfiguration.SdkRoot.ToEvaluatedPath()) != currentConfiguration.GetEvaluatedPropertyValue("NintendoSdkRoot") ||
                            expectedConfiguration.Spec != currentSpec ||
                            expectedConfiguration.BuildType.ToString() != currentBuildType))
                    {
                        yield return new FailedToApplyConfiguration(
                            expectedVcProject,
                            expectedConfiguration.ConfigurationPair,
                            new SdkConfigurationSettingsStore(expectedConfiguration),
                            currentSdkRoot,
                            currentSpec,
                            currentBuildType);
                    }
                }
            }
        }

        private static string EnsureEndsWithPathSeparator(string directory)
        {
            if (string.IsNullOrEmpty(Path.GetFileName(directory)))
            {
                return directory;
            }
            else
            {
                return directory + '\\';
            }
        }

        // FIXME: 途中で失敗したらどうすんのこれ
        public static void UpgradePropertySheets(IServiceProvider serviceProvider, SdkSolution solution)
        {
            foreach (var h in VsSolutionUtil.EnumerateVcxproj(serviceProvider))
            {
                UpgradePropertySheets(h, solution);
            }
        }

        private static void UpgradePropertySheets(IVsHierarchy h, SdkSolution solution)
        {
            var a = new VcAccessors.VcProjectAccessor(h);
            var vcProjectModel = solution.VcProjects.Single(x => x.ProjectFilePath == a.ProjectFilePath);
            var isSdkEnabled = vcProjectModel.ProjectConfigurations.Any(x => x.IsSdkEnabled);
            if (isSdkEnabled)
            {
                // あらかじめ ImportNintendoSdk.props を生成しておく
                WriteImportNintendoSdkProps(Path.Combine(Path.GetDirectoryName(vcProjectModel.ProjectFilePath), "ImportNintendoSdk.props"));
            }

            foreach (var c in a.Configurations)
            {
                var configurationModel = vcProjectModel.ProjectConfigurations.Single(x => x.Configuration == c.ConfigurationName && x.Platform == c.PlatformName);
                var propsToRemove = c.PropertySheets.Where(x => SdkConfigurationManagerUtil.IsV0SdkPropertySheet(x.Name)).ToArray();
                foreach (var p in propsToRemove)
                {
                    c.RemovePropertySheet(p);
                }

                var importNintendoSdkProps = c.PropertySheets.Where(x => x.Name == "ImportNintendoSdk").FirstOrDefault();
                if (configurationModel.IsSdkEnabled)
                {
                    if (importNintendoSdkProps == null)
                    {
                        c.AddPropertySheet("ImportNintendoSdk.props");
                    }
                }
                else
                {
                    if (importNintendoSdkProps != null)
                    {
                        c.RemovePropertySheet(importNintendoSdkProps);
                    }
                }
            }
        }

        // FIXME: 途中で失敗したらどうすんのこれ
        public static void ApplyConfigurationsToVcxproj(SdkSolution solution)
        {
            var projectCollection = new Microsoft.Build.Evaluation.ProjectCollection();
            foreach (var vcProject in solution.VcProjects)
            {
                ApplyConfigurationsToVcxproj(projectCollection, vcProject);
            }
            projectCollection.UnloadAllProjects();
        }

        private static void ApplyConfigurationsToVcxproj(Microsoft.Build.Evaluation.ProjectCollection projectCollection, SdkVcProject vcProject)
        {
            // VS バージョンごとに MSBuild のバージョンが異なるが、
            // MSBuild 4.0 を参照しておけば、バインドリダイレクトで VS が使用しているバージョンが使われる
            using (var project = new MsBuildConstructionProjectAccessor(vcProject.ProjectFilePath, projectCollection))
            {
                foreach (var configuration in vcProject.ProjectConfigurations)
                {
                    var conditionEvaluator = new MsBuildConditionEvaluator(configuration.Configuration, configuration.Platform);

                    if (configuration.IsSdkEnabled)
                    {
                        // configuration の構成の "Configuration" プロパティグループを取得
                        var targetGroups = project.GetPropertyGroups(conditionEvaluator, BaseConstants.ProjectGroupConfigurationLabel);

                        // プロパティグループが複数ある（Visual Studio から編集するとそうはならないはず）場合
                        // 最初のプロパティグループに反映する。
                        var targetGroup = targetGroups.FirstOrDefault();
                        if (targetGroup == null)
                        {
                            // 存在しなければ反映をあきらめる。後段で反映できたことのチェックを行うので、そこで警告ダイアログが表示される。
                            Global.Logger.WriteGeneralLog(
                                Strings.FailedToApplyConfiguration_MissingPropertyGroupElement,
                                vcProject.ProjectFilePath,
                                configuration.Configuration,
                                configuration.Platform);
                            continue;
                        }

                        // NintendoSDK のプロパティを設定
                        targetGroup.SetProperty("NintendoSdkRoot", configuration.SdkRoot.ToUnevaluatedPath());
                        targetGroup.SetProperty("NintendoSdkSpec", configuration.Spec);
                        targetGroup.SetProperty("NintendoSdkBuildType", configuration.BuildType.ToString());

                        // 余計なプロパティグループから NintendoSDK のプロパティを削除
                        foreach (var extraGroup in targetGroups.Skip(1))
                        {
                            extraGroup.TryRemoveProperty("NintendoSdkRoot");
                            extraGroup.TryRemoveProperty("NintendoSdkSpec");
                            extraGroup.TryRemoveProperty("NintendoSdkBuildType");
                        }
                    }
                    else
                    {
                        var targetGroups = project.GetPropertyGroups(conditionEvaluator, BaseConstants.ProjectGroupConfigurationLabel);

                        foreach (var targetGroup in targetGroups)
                        {
                            // NintendoSDK のプロパティを削除
                            targetGroup.TryRemoveProperty("NintendoSdkRoot");
                            targetGroup.TryRemoveProperty("NintendoSdkSpec");
                            targetGroup.TryRemoveProperty("NintendoSdkBuildType");
                        }
                    }
                }

                project.Save(vcProject.ProjectFilePath);
            }
        }

        private static void WriteImportNintendoSdkProps(string path)
        {
            var assembly = Assembly.GetExecutingAssembly();
            using (var res = assembly.GetManifestResourceStream("Nintendo.NintendoSdkVsExtension.Resources.ImportNintendoSdk.props"))
            using (var fs = File.Create(path))
            {
                res.CopyTo(fs);
            }
        }
    }
}
