﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <utility>

#include "./../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_AhoCorasick.h"

// AhoCorasickBuilder は nn::ngc::detail::AhoCorasick と friend クラスである必要があるため、
// 名前空間は nn::ngc::detail でなければならない
namespace nn { namespace ngc { namespace detail {

/**
 * @brief   AC法で用いるインデックス(オートマトン)を作成します。
 * @details AddWord()関数等で検出対象の文字列（パターン）を追加していき、Build()関数でAhoCorasickオブジェクトを生成します。 作成されたAhoCorasickオブジェクトはユーザーがdeleteする必要があります。
 */
class AhoCorasickBuilder final
{
public:
    AhoCorasickBuilder() NN_NOEXCEPT : m_pPrv(NULL) {}
    ~AhoCorasickBuilder() NN_NOEXCEPT;
    /**
     * @brief       オブジェクトを初期化します。成功した場合はtrueを返します。
     */
    bool Init() NN_NOEXCEPT;
    /**
     * @brief       検出対象の文字列を追加します。
     * @param[in]   str 文字列
     * @return      成功した場合は true
     */
    bool AddWord(const char* str) NN_NOEXCEPT;
    /**
     * @brief       検出対象のデータを追加します。
     * @param[in]   p   データへのポインタ
     * @param[in]   n   データサイズ
     * @return      成功した場合は true
     */
    bool AddPattern(const void* p, size_t n) NN_NOEXCEPT;
    /**
     * @brief       検出対象の文字列の集合が入った配列から文字列を追加します。文字列の区切りは改行(CRLFかLF)である必要があります。
     * @param[in]   str 文字列へのポインタ
     * @param[in]   len 文字列の長さ(strlenの結果)
     * @return      成功した場合は true
     */
    bool AddWords(const char* str, size_t len) NN_NOEXCEPT;
    bool AddWords(const char* str) NN_NOEXCEPT
    {
        return AddWords(str, nlib_strlen(str));
    }
    /**
     * @brief   AhoCorasick オブジェクトを作成します。追加された文字列(パターン)を検出するAC法のためのオートマトンを構築します。
     * @return  AhoCorasick オブジェクトへのポインタ
     */
    AhoCorasick* Build() NN_NOEXCEPT;
    /**
     * @brief           ユーザー定義のコールバック関数です。
     * @param[in]       pFirst      検出語の先頭を指すポインタ
     * @param[in]       pLast       検出語の直後を指すポインタ
     * @param[in]       nodeid      検出した語のID(連続値ではない)
     * @param[in,out]   pUserObj    ユーザーデータ
     * @return          trueの場合は引き続き解析を続ける。falseの場合は処理を打ち切り
     * @details         文字列(パターン)を検出した場合に呼び出されます。 ユーザーはそれに対応した処理を記述します。
     */
    typedef bool (*MatchCallback)(const char* pFirst, const char* pLast, uint32_t nodeid,
                                  void* pUserObj);
    /**
     * @brief           文字列を検査して検出対象の文字列を検出します。
     * @param[in]       pDoc        ヌル終端する文字列
     * @param[in]       callback    語が検出された場合に呼び出されるコールバック関数
     * @param[in,out]   pUserObj    ユーザーデータ
     */
    void MatchByBuilder(const char* pDoc, MatchCallback callback, void* pUserObj) NN_NOEXCEPT;
    void MatchByBuilder(const char* pDoc, MatchCallback callback) NN_NOEXCEPT
    {
        MatchByBuilder(pDoc, callback, NULL);
    }
    /**
     * @brief       登録された文字列(パターン)の数を取得します。
     */
    size_t GetNumWords() const NN_NOEXCEPT;
    /**
     * @brief       登録された文字列(パターン)の総量(バイト単位)を取得します。
     */
    size_t GetNumBytes() const NN_NOEXCEPT;
    /**
     * @brief       作成されたオートマトンのノード数を取得します。
     */
    size_t GetNumNodes() const NN_NOEXCEPT;

private:
    bool SortNodes() NN_NOEXCEPT;
    struct BuildFailureArcTh;
    struct BuildReportTreeHolderTh;
    struct BuildFailureTreeHolderTh;
    struct BuildGotoArcHolderTh;
    struct BuildLenHolderTh;
    struct SortNodesTh;

private:
    struct AhoCorasickBuilderPrivate;
    AhoCorasickBuilderPrivate* m_pPrv;
    AhoCorasickBuilder(const AhoCorasickBuilder&) = delete;
    void operator=(const AhoCorasickBuilder&) = delete;
};

}}} // nn::ngc::detail

