﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Xml;

namespace IFFRefGen.Src
{
    // simpleType
    public class XsdSimpleType : XsdItem, IXsdType
    {
        #region 初期化
        // コンストラクタ
        public XsdSimpleType(XmlElement element, string filePath)
            : base(element, filePath)
        {
        }

        // プリミティブ型用のコンストラクタ
        public XsdSimpleType(string name)
            : base(name)
        {
            if (!name.StartsWith(GXsd.Namespace)) { Error("不明な型を参照しています"); }
            this.PrimitiveType = this;

            // 各種制約の初期化
            _enumerations = new List<string>();
            _minInclusive = string.Empty;
            _maxInclusive = string.Empty;
            _minExclusive = string.Empty;
            _maxExclusive = string.Empty;
            _pattern = string.Empty;
            _whiteSpace = string.Empty;
            _listLength = string.Empty;
            _isList = false;
        }

        public override void Initialize()
        {
            if (this.IsInitialized) { return; }
            base.Initialize();

            int elementCount = 0;

            foreach (XmlNode node in this.Element.ChildNodes)
            {
                if (node.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement element = (XmlElement)node;
                if (element.LocalName == "restriction")
                {
                    InitializeBaseType(element.GetAttribute("base"));
                    InitializeRestriction(element);
                }
                else if (element.LocalName == "list")
                {
                    InitializeBaseType(element.GetAttribute("itemType"));
                    // リスト制約の追加
                    _isList = true;
                }
                else { Debug.Assert(false, "Assertion (toDo)"); }
                elementCount++;
            }
            Debug.Assert(elementCount == 1, "Assertion (toDo)");
        }

        // 基本型の初期化
        private void InitializeBaseType(string typeName)
        {
            XsdSimpleType parent = GXsd.FindSimpleType(typeName);
            if (parent == null)
            {
                // 新しいプリミティブ型を発見
                parent = new XsdSimpleType(typeName);
                GXsd.SimpleTypes.Add(parent);
            }

            parent.Initialize();
            this.BaseType = parent;
            this.PrimitiveType = parent.PrimitiveType;
        }

        // 制約の初期化
        private void InitializeRestriction(XmlElement element)
        {
            foreach (XmlNode childNode in element.ChildNodes)
            {
                if (childNode.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement childElement = (XmlElement)childNode;
                switch (childElement.LocalName)
                {
                    case "enumeration":
                        InitializeEnumeration(childElement);
                        break;
                    case "minInclusive":
                        InitializeMinInclusive(childElement);
                        break;
                    case "maxInclusive":
                        InitializeMaxInclusive(childElement);
                        break;
                    case "minExclusive":
                        InitializeMinExclusive(childElement);
                        break;
                    case "maxExclusive":
                        InitializeMaxExclusive(childElement);
                        break;
                    case "pattern":
                        InitializePattern(childElement);
                        break;
                    case "whiteSpace":
                        InitializeWhiteSpace(childElement);
                        break;
                    case "length":
                        InitializeLength(childElement);
                        break;
                    default:
                        // minLength, maxLength は hexBinary のみなので、とりあえず非対応
                        // 文字列、hexBinary の Length、minLength、maxLength が必要なら対応
                        // Debug.WriteLine(childNode.LocalName + " " + this);
                        Error("サポートしていない restriction です " + childElement.LocalName);
                        break;
                }
            }
        }

        // 使用フラグを立てる
        public override void Use()
        {
            if (this.IsUsed) { return; }
            base.Use();
            if (this.HasBaseType) { this.BaseType.Use(); }
        }
        #endregion

        // 文字列化
        public override string ToString()
        {
            if (this.IsPrimitive)
            {
                return string.Format("P  {0}", this.Name);
            }
            else
            {
                return string.Format(
                    "{0} {1} {2} : {3}",
                    this.IsList ? "SL" : "S ",
                    this.PrimitiveType.Name,
                    this.Name,
                    this.BaseType.Name);
            }
        }

        // プリミティブか、プリミティブ型は親が null
        public bool IsPrimitive { get { return (this.BaseType == null); } }
        // 文字列か
        public bool IsString { get { return GXsd.IsString(this.PrimitiveType.Name); } }
        // 数値か
        public bool IsNumber { get { return GXsd.IsNumber(this.PrimitiveType.Name); } }

        // 基本型を持っているか
        public bool HasBaseType { get { return (this.BaseType != null); } }
        // 基本型
        public XsdSimpleType BaseType { get; protected set; }
        // プリミティブ型
        public XsdSimpleType PrimitiveType { get; protected set; }

        #region enumeration 列挙
        // enumeration
        private void InitializeEnumeration(XmlElement element)
        {
            if (!this.IsString) { Error("文字列以外に enumeration は指定できません"); }
            if (_enumerations == null)
            {
                // 用途があれば許可しても良い？
                if (this.IsEnumeration) { Error("既に enumeration を指定しています"); }
                _enumerations = new List<string>();
            }
            string value = element.GetAttribute("value");

            Debug.Assert(value != null, "Assertion (toDo)");
            _enumerations.Add(value);
            //Debug.WriteLine(this + " " + value);
        }
        public List<string> Enumerations
        {
            get
            {
                if (_enumerations != null) { return _enumerations; }
                return this.BaseType.Enumerations;
            }
        }
        public bool IsEnumeration { get { return (this.Enumerations.Count > 0); } }
        private List<string> _enumerations;
        #endregion

        #region minInclusive 値を含む最小値
        private void InitializeMinInclusive(XmlElement element)
        {
            if (!this.IsNumber) { Error("数値型以外に minInclusive は指定できません"); }
            // 用途があれば許可しても良い？
            if (this.HasMinInclusive) { Error("既に minInclusive を指定しています"); }

            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _minInclusive = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string MinInclusive
        {
            get
            {
                if (_minInclusive != null) { return _minInclusive; }
                return this.BaseType.MinInclusive;
            }
        }
        public bool HasMinInclusive { get { return (this.MinInclusive != string.Empty); } }
        private string _minInclusive;
        #endregion

        #region maxInclusive 値を含む最大値
        private void InitializeMaxInclusive(XmlElement element)
        {
            if (!this.IsNumber) { Error("数値型以外に maxInclusive は指定できません"); }
            // 用途があれば許可しても良い？
            if (this.HasMaxInclusive) { Error("既に maxInclusive を指定しています"); }
            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _maxInclusive = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string MaxInclusive
        {
            get
            {
                if (_maxInclusive != null) { return _maxInclusive; }
                return this.BaseType.MaxInclusive;
            }
        }
        public bool HasMaxInclusive { get { return (this.MaxInclusive != string.Empty); } }
        private string _maxInclusive;
        #endregion

        #region minExclusive 値を含まない最小値
        private void InitializeMinExclusive(XmlElement element)
        {
            if (!this.IsNumber) { Error("数値型以外に minExclusive は指定できません"); }
            // 用途があれば許可しても良い？
            if (this.HasMinExclusive) { Error("既に minExclusive を指定しています"); }

            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _minExclusive = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string MinExclusive
        {
            get
            {
                if (_minExclusive != null) { return _minExclusive; }
                return this.BaseType.MinExclusive;
            }
        }
        public bool HasMinExclusive { get { return (this.MinExclusive != string.Empty); } }
        private string _minExclusive;
        #endregion

        #region maxExclusive 値を含まない最大値
        private void InitializeMaxExclusive(XmlElement element)
        {
            if (!this.IsNumber) { Error("数値型以外に maxExclusive は指定できません"); }
            // 用途があれば許可しても良い？
            if (this.HasMaxExclusive) { Error("既に maxExclusive を指定しています"); }
            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _maxExclusive = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string MaxExclusive
        {
            get
            {
                if (_maxExclusive != null) { return _maxExclusive; }
                return this.BaseType.MaxExclusive;
            }
        }
        public bool HasMaxExclusive { get { return (this.MaxExclusive != string.Empty); } }
        private string _maxExclusive;
        #endregion

        #region pattern 正規表現
        private void InitializePattern(XmlElement element)
        {
            if (this.HasPattern) { Error("既に pattern を指定しています"); }
            string value = element.GetAttribute("value");
            // 制御文字（&#x09;&#x0a;&#x0d;）をそのまま文字として読み込めない
            // XML実体参照文字は以下のコードで保持できる
            //value = element.Attributes[0].InnerXml;
            Debug.Assert(value != null, "Assertion (toDo)");
            _pattern = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string Pattern
        {
            get
            {
                if (_pattern != null) { return _pattern; }
                return this.BaseType.Pattern;
            }
        }
        public bool HasPattern { get { return (this.Pattern != string.Empty); } }
        private string _pattern;
        #endregion

        #region whiteSpace 空白の扱い
        private void InitializeWhiteSpace(XmlElement element)
        {
            if (!this.IsString) { Error("文字列型以外に whiteSpace は指定できません"); }
            // 用途があれば許可しても良いかも
            if (this.HasWhiteSpace) { Error("既に whiteSpace を指定しています"); }
            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _whiteSpace = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string WhiteSpace
        {
            get
            {
                if (_whiteSpace != null) { return _whiteSpace; }
                return this.BaseType.WhiteSpace;
            }
        }
        public bool HasWhiteSpace { get { return (this.WhiteSpace != string.Empty); } }
        private string _whiteSpace;
        #endregion

        #region length リストや文字列の長さ（現状リストのみ）
        private void InitializeLength(XmlElement element)
        {
            // 文字列も必要になれば許可する？許可する場合はこの中で分岐
            // ListLength と StringLength に、hexBinary もあるので ValueLength かも
            // その場合、文字列の Length とリストの Length を別に扱う必要がある
            if (!this.IsList) { Error("リスト以外に length は指定できません"); }
            // 用途があれば許可しても良いかも
            if (this.HasListLength) { Error("既に length を指定しています"); }
            string value = element.GetAttribute("value");
            Debug.Assert(value != null, "Assertion (toDo)");
            _listLength = value;
            //Debug.WriteLine(this + " " + value);
        }
        public string ListLength
        {
            get
            {
                if (_listLength != null) { return _listLength; }
                return this.BaseType.ListLength;
            }
        }
        public bool HasListLength { get { return (this.ListLength != string.Empty); } }
        private string _listLength;
        #endregion

        #region list リスト
        public bool IsList
        {
            get { return _isList.HasValue ? _isList.Value : this.BaseType.IsList; }
        }
        private bool? _isList;
        #endregion
    }
}
