﻿using System;
using System.Collections.Generic;
using System.Xml;

namespace IFFRefGen.Src
{
    // complexType
    public class XsdComplexType : XsdItem, IXsdType
    {
        #region 初期化
        // コンストラクタ
        public XsdComplexType(XmlElement element, string filePath)
            : base(element, filePath)
        {
            //Debug.WriteLine("\n" + element.InnerXml);
        }

        // 初期化
        public override void Initialize()
        {
            if (this.IsInitialized) { return; }
            base.Initialize();

            foreach (XmlNode node in this.Element.ChildNodes)
            {
                if (node.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement element = (XmlElement)node;
                switch (element.LocalName)
                {
                    case "simpleContent":
                        InitializeSimpleContent(element);
                        break;
                    case "sequence":
                        this.Content = new XsdContentSequence(element, this);
                        break;
                    case "choice":
                        this.Content = new XsdContentChoice(element, this);
                        break;
                    case "attributeGroup":
                        InitializeAttributeGroup(element);
                        break;
                    case "attribute":
                        InitializeAttribute(element);
                        break;
                    default:
                        // complexContent と all は未サポート
                        Error("未サポートの要素です " + element.LocalName);
                        break;
                }
            }
            if (this.Content != null) { this.Content.Initialize(); }
        }

        // simpleContent の初期化
        public void InitializeSimpleContent(XmlElement simpleContentElement)
        {
            foreach (XmlNode node in simpleContentElement.ChildNodes)
            {
                if (node.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement element = (XmlElement)node;
                if (element.LocalName != "extension")
                {
                    Error("simpleContent は extension しかサポートしません");
                }

                string baseTypeName = element.GetAttribute("base");
                if (baseTypeName == string.Empty) { Error("extension に base 属性がありません"); }

                if (this.BaseType != null) { Error("extension は一つしかサポートしません"); }
                this.BaseType = GXsd.FindSimpleType(baseTypeName);
                if (this.BaseType == null) { Error("simpleContent の基本型がありません"); }

                InitializeSimpleContentExtension(element);
            }
        }

        // simpleContent 拡張の初期化
        public void InitializeSimpleContentExtension(XmlElement extensionElement)
        {
            foreach (XmlNode node in extensionElement)
            {
                if (node.NodeType == XmlNodeType.Comment) { continue; }
                XmlElement element = (XmlElement)node;
                switch (element.LocalName)
                {
                    case "attributeGroup":
                        InitializeAttributeGroup(element);
                        break;
                    case "attribute":
                        InitializeAttribute(element);
                        break;
                    default:
                        Error("extension に未サポートの要素があります " + element.LocalName);
                        break;
                }
            }
        }

        // attributeGroup の初期化
        public void InitializeAttributeGroup(XmlElement element)
        {
            string attributeGroupName = element.GetAttribute("ref");
            if (attributeGroupName == string.Empty)
            {
                Error("ref 属性が見つかりません " + element.LocalName);
            }
            XsdAttributeGroup attributeGroup = GXsd.FindAttributeGroup(attributeGroupName);
            if (attributeGroup == null)
            {
                Error("attributeGroup が見つかりません " + attributeGroupName);
            }
            // attributeGroup は attribute に展開するので、ここで使用フラグを立てる
            attributeGroup.Use();
            this.Attributes.AddRange(attributeGroup.Attributes);
        }

        // attribute の初期化
        public void InitializeAttribute(XmlElement element)
        {
            XsdAttribute attribute = new XsdAttribute(element, this.FilePath);
            attribute.Initialize();
            this.Attributes.Add(attribute);
        }

        // 使用フラグを立てる
        public override void Use()
        {
            if (this.IsUsed) { return; }
            base.Use();
            if (this.BaseType != null) { this.BaseType.Use(); }
            if (this.Content != null) { this.Content.Use(); }
            foreach (XsdAttribute attribute in this.Attributes)
            {
                attribute.Use();
            }
        }
        #endregion

        // 文字列化
        public override string ToString()
        {
            return string.Format("C  {0}", this.Name);
        }

        // extension されていなければ null
        public IXsdType BaseType { get; private set; }
        // simpleContent を extension していれば null
        public XsdContent Content { get; private set; }
        // 属性
        public List<XsdAttribute> Attributes { get { return _attributes; } }
        private readonly List<XsdAttribute> _attributes = new List<XsdAttribute>();
    }
}
