﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;

namespace IFFRefGen.Src
{
    // 要素の属性説明
    public class HtmlElementAttribute
    {
        // 属性 XPath
        private readonly string AttributeXPath = "//div[@id='attribute']";
        // 属性テーブル XPath
        private readonly string AttributeTableXPath = "//tbody[@id='attribute_table']";
        // DCC テーブルヘッダ XPath
        private readonly string DccTableHeaderXPath = "//th[@id='attribute_dcc']";
        // 属性無し
        private readonly string AttributeEmpty = "属性はありません。";
        // 属性省略可能
        private readonly string AttributeOptional = "この属性は省略できます。";

        // コンストラクタ
        public HtmlElementAttribute(XsdElement element, DocElement document, XmlDocument template)
        {
            this.Element = element;
            this.Document = document;
            this.Template = template;
        }

        // 生成
        public void Generate()
        {
            // 属性無し対応
            XsdComplexType complexType = this.Element.Type as XsdComplexType;
            if ((complexType == null) || (complexType.Attributes.Count == 0))
            {
                XmlNode attributeNode = this.Template.SelectSingleNode(this.AttributeXPath);
                attributeNode.RemoveAll();
                XmlElement p = this.Template.CreateElement("p");
                attributeNode.AppendChild(p);
                p.InnerText = this.AttributeEmpty;
                return;
            }

            // DCC 出力無効対応
            if ((this.Document != null) && this.Document.DccExportDisable)
            {
                XmlNode dccNode = this.Template.SelectSingleNode(this.DccTableHeaderXPath);
                dccNode.ParentNode.RemoveChild(dccNode);
            }

            // アトリビュートテーブル生成
            XmlNode attributeTableNode = this.Template.SelectSingleNode(this.AttributeTableXPath);
            foreach (XsdAttribute attribute in complexType.Attributes)
            {
                GenerateAttribute(attributeTableNode, attribute);
            }
        }

        // 属性の生成
        private void GenerateAttribute(XmlNode attributeTableNode, XsdAttribute attribute)
        {
            // 行の追加
            XmlElement tr = this.Template.CreateElement("tr");
            XmlAttribute id = this.Template.CreateAttribute("id");
            id.InnerText = attribute.Name;
            tr.Attributes.Append(id);
            attributeTableNode.AppendChild(tr);

            // 属性ドキュメントの取得
            DocAttribute docAttribute = GetDocAttribute(attribute);

            // 名前
            GenerateAttributeName(tr, attribute);

            // 型と値
            GenerateAttributeTypeAndValue(tr, attribute);

            // 説明
            GenerateAttributeDescription(tr, attribute, docAttribute);

            // GX
            //GenerateAttributeGx(tr, attribute, docAttribute);

            // DCC
            GenerateAttributeDcc(tr, attribute, docAttribute);
        }

        // 属性ドキュメントの取得
        private DocAttribute GetDocAttribute(XsdAttribute attribute)
        {
            DocAttribute docAttribute = null;

            if (this.Document != null)
            {
                docAttribute = this.Document.FindAttribute(attribute.Name);
            }

            if (docAttribute != null) { docAttribute.IsUsed = true; }
            return docAttribute;
        }

        #region 名前
        // 属性の名前生成
        private void GenerateAttributeName(XmlElement tr, XsdAttribute attribute)
        {
            XmlElement tdName = this.Template.CreateElement("td");
            tr.AppendChild(tdName);
            XmlAttribute nameClass = this.Template.CreateAttribute("class");
            nameClass.Value = "nowrap_right";
            tdName.Attributes.Append(nameClass);
            tdName.InnerXml = "<strong>" + attribute.Name + "</strong>";
        }
        #endregion

        #region 型と値
        // 属性の型と値の生成
        private void GenerateAttributeTypeAndValue(XmlElement tr, XsdAttribute attribute)
        {
            XmlElement td = this.Template.CreateElement("td");
            tr.AppendChild(td);
            XmlAttribute typeClass = this.Template.CreateAttribute("class");
            typeClass.Value = "nowrap_center";
            td.Attributes.Append(typeClass);

            GenerateAttributeType2(td, attribute);
            td.AppendChild(this.Template.CreateElement("br"));
            GenerateAttributeValue2(td, attribute);
        }
        // アトリビュート値の生成
        private void GenerateAttributeType2(XmlElement td, XsdAttribute attribute)
        {
            string typeName = string.Empty;
            if (attribute.Type.IsEnumeration)
            {
                typeName = "enum";
            }
            else
            {
                switch (attribute.Type.PrimitiveType.Name)
                {
                    case "xsd:boolean":
                        typeName = "bool";
                        break;
                    case "xsd:byte":
                        typeName = "byte";
                        break;
                    case "xsd:unsignedByte":
                        typeName = "ubyte";
                        break;
                    case "xsd:short":
                        typeName = "short";
                        break;
                    case "xsd:unsignedShort":
                        typeName = "ushort";
                        break;
                    case "xsd:int":
                        typeName = "int";
                        break;
                    case "xsd:unsignedInt":
                        typeName = "uint";
                        break;
                    case "xsd:float":
                        typeName = "float";
                        break;
                    case "xsd:string":
                        typeName = "string";
                        break;
                    case "xsd:dateTime":
                        typeName = "dateTime";
                        break;
                    default:
                        throw new Exception(
                            "未対応の基本型です " + attribute.Type.PrimitiveType.Name);
                }
            }
            if (attribute.Type.IsList)
            {
                typeName += string.Format("[{0}]", attribute.Type.ListLength);
            }
            XmlElement strong = this.Template.CreateElement("strong");
            td.AppendChild(strong);
            strong.InnerText = typeName;
        }

        // アトリビュート値の生成
        private void GenerateAttributeValue2(XmlElement td, XsdAttribute attribute)
        {
            XsdSimpleType valueType = attribute.Type;
            switch (valueType.PrimitiveType.Name)
            {
                case "xsd:boolean":
                    GenerateBooleanAttributeValue(td, valueType);
                    break;
                case "xsd:byte":
                    GenerateNumberAttributeValue(td, valueType, "-128", "127");
                    break;
                case "xsd:unsignedByte":
                    GenerateNumberAttributeValue(td, valueType, "0", "255");
                    break;
                case "xsd:short":
                    GenerateNumberAttributeValue(td, valueType, "-32768", "32767");
                    break;
                case "xsd:unsignedShort":
                    GenerateNumberAttributeValue(td, valueType, "0", "65535");
                    break;
                case "xsd:int":
                    GenerateNumberAttributeValue(td, valueType, "MIN", "MAX");
                    break;
                case "xsd:unsignedInt":
                    GenerateNumberAttributeValue(td, valueType, "0", "MAX");
                    break;
                case "xsd:float":
                    GenerateNumberAttributeValue(td, valueType, "MIN", "MAX");
                    break;
                case "xsd:string":
                    if (valueType.IsEnumeration)
                    {
                        GenerateEnumAttributeValue(td, valueType);
                    }
                    else
                    {
                        GenerateStringAttributeValue(td, valueType);
                    }
                    break;
                case "xsd:dateTime":
                    td.InnerXml += "ISO8601";
                    break;
                default:
                    throw new Exception("未対応の基本型です " + attribute.Type.PrimitiveType.Name);
            }
        }

        // boolean アトリビュート値の生成
        private void GenerateBooleanAttributeValue(XmlElement tdValue, XsdSimpleType valueType)
        {
            tdValue.AppendChild(this.Template.CreateTextNode("true false"));
        }

        // 数値アトリビュート値の生成
        private void GenerateNumberAttributeValue(
            XmlElement tdValue, XsdSimpleType valueType, string min, string max)
        {
            if (valueType.HasMinInclusive)
            {
                tdValue.AppendChild(this.Template.CreateTextNode(valueType.MinInclusive));
            }
            else if (valueType.HasMinExclusive)
            {
                tdValue.AppendChild(this.Template.CreateTextNode(valueType.MinExclusive + " <"));
            }
            else
            {
                tdValue.AppendChild(this.Template.CreateTextNode(min));
            }
            tdValue.AppendChild(this.Template.CreateTextNode(" ～ "));
            if (valueType.HasMaxInclusive)
            {
                tdValue.AppendChild(this.Template.CreateTextNode(valueType.MaxInclusive));
            }
            else if (valueType.HasMaxExclusive)
            {
                tdValue.AppendChild(this.Template.CreateTextNode("< " + valueType.MaxExclusive));
            }
            else
            {
                tdValue.AppendChild(this.Template.CreateTextNode(max));
            }
        }

        // enum アトリビュート値の生成
        private void GenerateEnumAttributeValue(XmlElement tdValue, XsdSimpleType valueType)
        {
            int count = valueType.Enumerations.Count;
            for (int i = 0; i < count; i++)
            {
                tdValue.AppendChild(this.Template.CreateTextNode(valueType.Enumerations[i]));
                if (i < (count - 1))
                {
                    tdValue.AppendChild(this.Template.CreateElement("br"));
                }
            }
        }

        // 文字列アトリビュート値の生成
        private void GenerateStringAttributeValue(XmlElement tdValue, XsdSimpleType valueType)
        {
            if (valueType.HasPattern)
            {
                tdValue.AppendChild(this.Template.CreateTextNode(valueType.Pattern));
            }
            else
            {
                tdValue.AppendChild(this.Template.CreateTextNode("-"));
            }
        }
        #endregion

        #region 説明
        // アトリビュート説明の生成
        private void GenerateAttributeDescription(
            XmlElement tr, XsdAttribute attribute, DocAttribute docAttribute)
        {
            XmlElement tdDesc = this.Template.CreateElement("td");
            tr.AppendChild(tdDesc);
            XmlAttribute descClass = this.Template.CreateAttribute("class");
            descClass.Value = "description";
            tdDesc.Attributes.Append(descClass);

            if ((docAttribute != null) && (docAttribute.Detail != null))
            {
                string detail = docAttribute.Detail;
                detail = GenerateTable(attribute, detail);
                tdDesc.InnerXml = GHtml.LinkElement(detail);

                tdDesc.AppendChild(this.Template.CreateElement("br"));
            }

            // 省略可能
            if (!attribute.IsRequired)
            {
                tdDesc.AppendChild(this.Template.CreateTextNode(this.AttributeOptional));
                tdDesc.AppendChild(this.Template.CreateElement("br"));
            }

            if (attribute.Type.HasWhiteSpace)
            {
                throw new Exception("WhiteSpace には未対応です。");
            }
        }

        // テーブルの生成
        private string GenerateTable(XsdAttribute attribute, string detail)
        {
            string result = detail;

            int headerDefine = detail.IndexOf("{{");
            int firstDefine = detail.IndexOf("[[");
            if (firstDefine == -1) { return result; }
            int lastDefine = detail.LastIndexOf("]]");
            if (lastDefine == -1) { return result; }

            result = result.Insert(lastDefine + 2, "</tbody></table>");
            if (headerDefine != -1)
            {
                result = result.Insert(firstDefine, "</thead><tbody>");
                result = result.Insert(headerDefine, "<table><thead>");
                result = result.Replace("{{", "<tr><th>");
                result = result.Replace("}{", "</th><th>");
                result = result.Replace("}}", "</th></tr>");
            }
            else
            {
                result = result.Insert(firstDefine, "<table><tbody>");
            }
            result = result.Replace("[[", "<tr><td>");
            result = result.Replace("][", "</td><td>");
            result = result.Replace("]]", "</td></tr>");
            return result;
        }
        #endregion

        #region GX
        // 属性の GX 生成
        private void GenerateAttributeGx(
            XmlElement tr, XsdAttribute attribute, DocAttribute docAttribute)
        {
            XmlElement tdGx = this.Template.CreateElement("td");
            tr.AppendChild(tdGx);
            XmlAttribute gxClass = this.Template.CreateAttribute("class");
            gxClass.Value = "gx";
            tdGx.Attributes.Append(gxClass);

            if ((docAttribute != null) && (docAttribute.Gx != null))
            {
                tdGx.InnerText = docAttribute.Gx;
            }
            else
            {
                tdGx.InnerText = "-";
            }
        }
        #endregion

        #region DCC
        // 属性の DCC 生成
        private void GenerateAttributeDcc(
            XmlElement tr, XsdAttribute attribute, DocAttribute docAttribute)
        {
            // DCC 出力無効対応
            // 共通アトリビュートがあるので、DccExport 指定の有無はチェックしない
            if ((this.Document != null) && this.Document.DccExportDisable) { return; }

            XmlElement tdDcc = this.Template.CreateElement("td");
            tr.AppendChild(tdDcc);
            XmlAttribute dccClass = this.Template.CreateAttribute("class");
            dccClass.Value = "nowrap_center";
            tdDcc.Attributes.Append(dccClass);
            if (docAttribute != null)
            {
                switch (docAttribute.DccExport)
                {
                    case DocDccExport.Enable:
                        tdDcc.InnerText = "設定";
                        break;
                    case DocDccExport.Disable:
                        tdDcc.InnerText = "不可";
                        break;
                    case DocDccExport.Fixed:
                        tdDcc.InnerText = "固定";
                        break;
                    case DocDccExport.Default:
                        throw new Exception("DccExport が指定されていません " + attribute.Name);
                    default:
                        throw new Exception("サポートしていない DocDccExport です");
                }
            }
            else
            {
                tdDcc.InnerText = "-";
            }
        }
        #endregion

        // 要素
        public XsdElement Element { get; private set; }
        // 要素ドキュメント
        public DocElement Document { get; private set; }
        // テンプレート
        public XmlDocument Template { get; private set; }
    }
}
