﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using System.Diagnostics;

namespace IFFRefGen.Src
{
    // HTML 属性リスト
    public class HtmlAttributeList
    {
        // テンプレートファイルの名前
        private readonly string TemplateFileName = "TmplAttributeList.html";
        // 保存パスのフォーマット文字列
        private readonly string SavePathFormat = "html/reference/attribute_list.html";

        // 要素リストノードの XPath
        private readonly string ElementListNodeXPath = "//div[@id='contents_list']";
        // 要素のフォーマット文字列
        private readonly string ElementFormat = "<{0}>";
        // リンクのフォーマット文字列
        private readonly string HrefFormat = "elements/{0}.html#{1}";

        // コンストラクタ
        public HtmlAttributeList()
        {
            this.Template = new XmlDocument();
        }

        // 初期化
        public void Initialize()
        {
            // テンプレートの読み込み
            string loadPath = GApp.Config.HtmlOutputConfig.TemplateElementFolder + this.TemplateFileName;
            this.Template.Load(loadPath);

            // 属性リストの初期化
            foreach (XsdElement xsdElement in GXsd.Elements)
            {
                XsdComplexType complexType = xsdElement.Type as XsdComplexType;
                if (complexType == null) { continue; }
                foreach (XsdAttribute xsdAttribute in complexType.Attributes)
                {
                    List<XsdElement> elementList = null;
                    if (this.Attributes.ContainsKey(xsdAttribute.Name))
                    {
                        elementList = this.Attributes[xsdAttribute.Name];
                    }
                    else
                    {
                        elementList = new List<XsdElement>();
                        this.Attributes.Add(xsdAttribute.Name, elementList);
                    }
                    elementList.Add(xsdElement);
                    //xsdAttribute.Name
                }
            }
        }

        // 生成
        public void Generate()
        {
            string[] attributeNames = this.Attributes.Keys.ToArray<string>();
            Array.Sort<string>(attributeNames,
                delegate(string lhs, string rhs) { return lhs.CompareTo(rhs); });

            XmlNode contentsNode = this.Template.SelectSingleNode(this.ElementListNodeXPath);

            // インデックステーブル
            XmlElement indexTable = this.Template.CreateElement("table");
            contentsNode.AppendChild(indexTable);
            indexTable.SetAttribute("id", "index_table");
            XmlElement indexThead = this.Template.CreateElement("thead");
            indexTable.AppendChild(indexThead);
            XmlElement indexTr = this.Template.CreateElement("tr");
            indexThead.AppendChild(indexTr);

            XmlElement rootUl = this.Template.CreateElement("ul");
            contentsNode.AppendChild(rootUl);

            XmlElement letterUl = null;
            char firstLetter = ' ';
            int indexTableCount = 0;
            foreach (string attributeName in attributeNames)
            {
                // 頭文字処理
                if (attributeName[0] != firstLetter)
                {
                    firstLetter = attributeName[0];

                    XmlElement li = this.Template.CreateElement("li");
                    string letter = firstLetter.ToString().ToUpper();
                    li.InnerXml = "<strong>" + letter +
                        "　<a href=\"#index_table\" class=\"disable_chm\">↑</a></strong>";
                    string id = "id_" + letter;
                    li.SetAttribute("id", id);
                    rootUl.AppendChild(li);

                    letterUl = this.Template.CreateElement("ul");
                    li.AppendChild(letterUl);

                    // インデックステーブルへの追加
                    XmlElement indexTh = this.Template.CreateElement("th");
                    indexTr.AppendChild(indexTh);
                    indexTh.InnerXml =
                        "<strong><a href=\"#" + id + "\">" + letter + "</a></strong>";
                    indexTableCount++;
                    if ((indexTableCount % 8) == 0)
                    {
                        indexTr = this.Template.CreateElement("tr");
                        indexThead.AppendChild(indexTr);
                    }
                }

                GenerateAttribute(attributeName, letterUl);
            }

            // 保存
            string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, this.SavePathFormat);

            GHtml.Save(this.Template, savePath);
        }

        // 属性の生成
        private void GenerateAttribute(string attributeName, XmlNode parentNode)
        {
            XmlElement attributeLi = this.Template.CreateElement("li");
            parentNode.AppendChild(attributeLi);
            XmlElement strong = this.Template.CreateElement("em");
            strong.InnerText = attributeName;
            attributeLi.AppendChild(strong);

            XmlElement ul = this.Template.CreateElement("ul");
            attributeLi.AppendChild(ul);

            List<XsdElement> elements = this.Attributes[attributeName];
            foreach (XsdElement element in elements)
            {
                XmlElement li = this.Template.CreateElement("li");
                ul.AppendChild(li);

                XmlElement a = this.Template.CreateElement("a");
                a.InnerText = string.Format(this.ElementFormat, element.Name);
                {
                    XmlAttribute href = this.Template.CreateAttribute("href");
                    href.InnerText = string.Format(
                        this.HrefFormat, element.Name, attributeName);
                    a.Attributes.Append(href);
                    XmlAttribute target = this.Template.CreateAttribute("target");
                    target.InnerText = "main";
                    a.Attributes.Append(target);
                }
                li.AppendChild(a);
            }
        }

        // テンプレート
        public XmlDocument Template { get; private set; }
        // 属性リスト
        public readonly Dictionary<string, List<XsdElement>> Attributes =
            new Dictionary<string, List<XsdElement>>();
    }
}
