﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Xml;
using System.Text.RegularExpressions;

using LOG = Common.Logger;
using DOXYGEN_XML = Common.ParseDoxygenXml;
using FILEIO = Common.FileIOWrapper;

namespace MakeOverloadHd
{
    // @overloadlisti コマンドを内部的に用いることでドキュメント作成者の入力を不要にしたクラス
    public class ParseAndMakeHd2
    {
        private enum WordType
        {
            OVERLOAD_LIST_PAGE_NAME,
            OVERLOAD_LIST_PAGE_EXPLAIN1,
            OVERLOAD_LIST_PAGE_EXPLAIN2,
            OVERLOAD_API_EXPLAIN1_1,
            OVERLOAD_API_EXPLAIN1_2,
            OVERLOAD_API_EXPLAIN2_1,
            OVERLOAD_API_EXPLAIN2_2,
        }
        private static Dictionary<WordType, string> wordJaJp =
            new Dictionary<WordType, string>()
            {
                { WordType.OVERLOAD_LIST_PAGE_NAME,     "オーバーロードAPI一覧" },
                { WordType.OVERLOAD_LIST_PAGE_EXPLAIN1, "オーバーロードしているAPIの一覧です。" },
                { WordType.OVERLOAD_LIST_PAGE_EXPLAIN2, "これらのAPIはオーバーロードしています。" },
                { WordType.OVERLOAD_API_EXPLAIN1_1,     "これらのAPIはオーバーロードしています。" },
                { WordType.OVERLOAD_API_EXPLAIN1_2,     "全て派生したAPIですので、以下の派生元クラスのAPIを参照して下さい。" },
                { WordType.OVERLOAD_API_EXPLAIN2_1,     "これらのAPIはオーバーロードしています。" },
                { WordType.OVERLOAD_API_EXPLAIN2_2,     "このクラスで定義されたもの以外に派生したAPIもあります。以下の派生元クラスのAPIも参照して下さい。" },
            };
        private static Dictionary<WordType, string> wordEnUs =
            new Dictionary<WordType, string>()
            {
                { WordType.OVERLOAD_LIST_PAGE_NAME,     "List of overloaded API" },
                { WordType.OVERLOAD_LIST_PAGE_EXPLAIN1, "This is the list of overloaded APIs." },
                { WordType.OVERLOAD_LIST_PAGE_EXPLAIN2, "These API functions are overloaed." },
                { WordType.OVERLOAD_API_EXPLAIN1_1,     "These API functions are overloaed." },
                { WordType.OVERLOAD_API_EXPLAIN1_2,     "They are all inherited, so please refer to below APIs whose classs is inerited." },
                { WordType.OVERLOAD_API_EXPLAIN2_1,     "These API functions are overloaed." },
                { WordType.OVERLOAD_API_EXPLAIN2_2,     "Some of them are inherited, so please refer to below APIs whose class is inherited." },
            };
        private static Dictionary<CmdlineOption.LanguageModeType, Dictionary<WordType, string>> word =
            new Dictionary<CmdlineOption.LanguageModeType, Dictionary<WordType, string>>()
            {
                { CmdlineOption.LanguageModeType.LANGUAGE_JA_JP, wordJaJp },
                { CmdlineOption.LanguageModeType.LANGUAGE_EN_US, wordEnUs },
            };
        private static string headerBasePath = null;
        private static string documentBasePath = null;

        private struct FuncDefinitionInfo
        {
            public string FunctionName;
            public string ArgumentName;
            public bool IsDerived;
            public bool IsVirtual;
        }

        private class FuncCountInfo
        {
            private List<string> __refIdList = new List<string>();
            private List<FuncDefinitionInfo> __definitionList = new List<FuncDefinitionInfo>();
            private List<FuncCountInfo> __relateInfo = new List<FuncCountInfo>();
            private string __namespaceRefid;
            private string __functionName;
            private string __namespaceName;
            private bool __hasVirtualFunction;
            private bool __hasDerivedFunction;

            public List<string> RefIdList
            {
                get { return __refIdList; }
                set { __refIdList = value; }
            }
            public List<FuncDefinitionInfo> DefinitionList
            {
                get { return __definitionList; }
                set { __definitionList = value; }
            }
            // public List<FuncCountInfo> RelateInfo
            // {
            //     get { return __relateInfo; }
            //     set { __relateInfo = value; }
            // }
            public string NamespaceRefid
            {
                get { return __namespaceRefid; }
                set { __namespaceRefid  = value; }
            }
            public string FunctionName
            {
                get { return __functionName; }
                set { __functionName = value; }
            }
            public string NamespaceName
            {
                get { return __namespaceName; }
                set { __namespaceName = value; }
            }
            public bool HasVirtualFunction
            {
                get { return __hasVirtualFunction; }
                set { __hasVirtualFunction = value; }
            }
            public bool HasDerivedFunction
            {
                get { return __hasDerivedFunction; }
                set { __hasDerivedFunction = value; }
            }

            public void RegistRelateInfo(FuncCountInfo info)
            {
                __relateInfo.Add(info);
            }
            public IEnumerator<FuncCountInfo> GetEnumerator()
            {
                //まずinfo自身を返す (これが特徴)
                yield return this;
                //次にRelateInfoに登録されているものを返す
                foreach (FuncCountInfo relateInfo in this.__relateInfo)
                {
                    yield return relateInfo;
                }
            }
        }
        private static List<FuncCountInfo> functionCountList = new List<FuncCountInfo>();

        private static void AddFunctionCountInfo(string namespaceName, string functionName, string namespaceRefId, string refId)
        {
            string convertedFunctionName = FilterFunctionObjectName(functionName);
            foreach (FuncCountInfo inf in functionCountList)
            {
                if (inf.NamespaceName == namespaceName && inf.FunctionName == convertedFunctionName)
                {
                    inf.RefIdList.Add(refId);
                    return;
                }
            }
            // 関数オブジェクトの場合は()演算子を省略して解析する
            FuncCountInfo inf2 = new FuncCountInfo();
            inf2.NamespaceName = namespaceName;
            inf2.FunctionName = convertedFunctionName;
            inf2.NamespaceRefid = namespaceRefId;
            inf2.HasVirtualFunction = false;
            inf2.HasDerivedFunction = false;
            inf2.RefIdList.Add(refId);
            functionCountList.Add(inf2);
        }

        private static Dictionary<string, string> specialCharacterDef = new Dictionary<string, string>()
                                                                            { { "&lt;", "<" },
                                                                              { "&gt;", ">" },
                                                                            };

        private static string ConvertSpecalCharacterOfFunction(string functionName)
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder(functionName);
            foreach (KeyValuePair<string, string> def in specialCharacterDef)
            {
                sb.Replace(def.Key, def.Value);
            }
            return sb.ToString();
        }

        private static string FilterFunctionObjectName(string functionName)
        {
            string[] operatorChars = { "(", "&", "++" };
            foreach (string operatorString in operatorChars)
            {
                if (functionName.Contains(operatorString))
                {
                    int index = functionName.IndexOf(operatorString);
                    return functionName.Substring(0, index);
                }
            }
            return functionName;
        }

        /// <summary>
        /// ドキュメントパスを指定します
        /// </summary>
        /// <param name="doxygenpath">ドキュメントパス(%SIGLO_ROOT%\Documents\Outputs\Api 相当)</param>
        /// <returns>パスが存在して設定できたら true</returns>
        public static bool SetPath(string doxygenPath)
        {
            if (!Directory.Exists(doxygenPath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "directory not found : {0}", doxygenPath);
                return false;
            }
            headerBasePath = Common.UtilFs.ConvertFullPath(doxygenPath + "\\..\\..\\..\\Programs");
            documentBasePath = doxygenPath;
            if (!Directory.Exists(headerBasePath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "header base directory not found : {0}", headerBasePath);
                headerBasePath = null;
                documentBasePath = null;
                return false;
            }
            Common.UtilFs.SetVsHelpBaseDirName(CmdlineOption.InputDirectoryName);
            Common.UtilFs.SetVsHelpDirName(CmdlineOption.OutputDirectoryName);
            return true;
        }

        /// <summary>
        /// Doxygen XMLからオーバーロードAPIを検索します
        /// </summary>
        public static void SearchOverloadFunctions()
        {
            if (headerBasePath == null)
            {
                LOG.Log(LOG.Level.LOG_INFO, "SetPath() is not set\n");
                return;
            }

            // オーバーロードAPIをindex.xmlから検索します
            ReadOverloadApiFromXml();
            //CheckOverloadFunctions(); // デバッグ用

            // 各オーバーロードAPIの情報をそれぞれのXMLから取得します
            ReadOverloadApiInformationFromSeveralXml();

            // テンプレート違いのAPIを FuncCountInfo.RelateInfo にまとめます
            ListRelateFunctions();
        }

        private static void ReadOverloadApiFromXml()
        {
            string indexxmlpath = documentBasePath + Common.UtilFs.GetXmlPreDirName() + "\\" + Common.UtilFs.DoxygenIndexXmlName;
            if (!File.Exists(indexxmlpath))
            {
                LOG.Log(LOG.Level.LOG_ERROR, "index.xml not found!\n");
                return;
            }

            string fileData;
            FILEIO.GetData(out fileData, indexxmlpath);
            using (StringReader sr = new StringReader(fileData))
            using (XmlTextReader reader = new XmlTextReader(sr))
            {
                // doxygenが生成したXMLの解析
                DOXYGEN_XML.XmlReadLoop(reader,
                                            NamespaceHandler, FunctionHandler, NoneHandler, NoneHandler,
                                            NoneHandler, NoneHandler, NoneHandler, NoneHandler);
            }
        }

        private static void ReadOverloadApiInformationFromSeveralXml()
        {
            foreach (FuncCountInfo inf in functionCountList)
            {
                if (inf.RefIdList.Count <= 1)
                {
                    continue;
                }

#if false
                // MEMO: とりあえずNintendoWare側は未対応とする
                if (inf.NamespaceName.Contains("nw::"))
                {
                    inf.RefIdList.Clear();
                    continue;
                }
#endif

                if (Common.UtilDoxygen.CheckSkipApi(inf.FunctionName))
                {
                    // 対象外と判定された場合はRefIdListおよびDefinitionListをクリアする(DefinitionListはまだ触ってないので省略する)
                    inf.RefIdList.Clear();
                }

                System.Diagnostics.Debug.Assert(!inf.FunctionName.Contains("::"), "inf.FunctionName must not contains ::");
                foreach (string refId in inf.RefIdList)
                {
                    string namespaceXmlFilename = inf.NamespaceRefid;
                    if (DOXYGEN_XML.GetNamespaceRefIdFromRefId(refId) != inf.NamespaceRefid)
                    {
                        inf.HasDerivedFunction = true;
                        namespaceXmlFilename = DOXYGEN_XML.GetNamespaceRefIdFromRefId(refId);
                    }
                    string path = documentBasePath + Common.UtilFs.GetXmlPreDirName() + "\\" + namespaceXmlFilename + ".xml";
                    string fileData;
                    FILEIO.GetData(out fileData, path);
                    using (StringReader sr = new StringReader(fileData))
                    using (XmlTextReader reader = new XmlTextReader(sr))
                    {
                        if (Common.ParseDoxygenXml.CheckClassXml(reader, inf.NamespaceName, refId) == false)
                        {
                            // 対象外と判定された場合はRefIdListおよびDefinitionListをクリアする
                            inf.RefIdList.Clear();
                            inf.DefinitionList.Clear();
                            break;
                        }
                    }

                    using (StringReader sr = new StringReader(fileData))
                    using (XmlTextReader reader = new XmlTextReader(sr))
                    {
                        ClassXmlFunctionDefinitionName = null;
                        ClassXmlFunctionDefinitionArgument = null;
                        ClassXmlIsVirtual = false;
                        ClassXmlIsDocumented = false;
                        if (Common.ParseDoxygenXml.ClassXmlReadLoop(reader, refId, ClassXml_MemberRefHandler,
                                                                            ClassXml_DefinitionHandler, ClassXml_ArgsstringHandler,
                                                                            ClassXml_BriefdescriptionHandler, ClassXml_DetaileddescriptionFuncHandler,
                                                                            ClassXml_InbodydescriptionFuncHandler) == true)
                        {
                            if (ClassXmlIsDocumented == true)
                            {
                                // 定義情報を保存する
                                FuncDefinitionInfo definitionInfo = new FuncDefinitionInfo();
                                definitionInfo.FunctionName = ClassXmlFunctionDefinitionName;
                                definitionInfo.ArgumentName = ClassXmlFunctionDefinitionArgument;
                                //definitionInfo.FunctionName;
                                definitionInfo.IsDerived = DOXYGEN_XML.GetNamespaceRefIdFromRefId(refId) == inf.NamespaceRefid ? false : true;
                                definitionInfo.IsVirtual = ClassXmlIsVirtual;
                                inf.DefinitionList.Add(definitionInfo);
                                if (ClassXmlIsVirtual == true)
                                {
                                    inf.HasVirtualFunction = true;
                                }
                            }
                            else
                            {
                                // ドキュメント対象外のAPIはRefIdListおよびDefinitionListをクリアする
                                inf.RefIdList.Clear();
                                inf.DefinitionList.Clear();
                                break;
                            }
                        }
                        else
                        {
                            // 対象外と判定された場合はRefIdListおよびDefinitionListをクリアする
                            inf.RefIdList.Clear();
                            inf.DefinitionList.Clear();
                            break;
                        }
                    }
                }
            }
        }

        private static bool NoneHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string noneName)
        {
            return true;
        }

        private static bool NamespaceHandler(DOXYGEN_XML.CompoundMode cmode, string refid, string namespaceName, string functionName)
        {
            return true;
        }

        private static bool FunctionHandler(DOXYGEN_XML.CompoundMode cmode, string member_refid, string refid, string namespaceName, string functionName)
        {
            if (cmode == DOXYGEN_XML.CompoundMode.COMPOUND_FILE)
            {
                return true;
            }

            // 関数名情報を保存します
            AddFunctionCountInfo(namespaceName, functionName, refid, member_refid);

            return true;
        }

        private static string ClassXmlFunctionDefinitionName = null;
        private static string ClassXmlFunctionDefinitionArgument = null;
        private static bool ClassXmlIsVirtual = false;
        private static bool ClassXmlIsDocumented = false;

        private static bool ClassXml_MemberRefHandler(string name, string value)
        {
            if (name == "prot")
            {
                // 非公開扱いAPIはドキュメント対象外
                if (value != "public")
                {
                    return false;
                }
            }
            else if (name == "virt")
            {
                //TODO: <reimplements> がなければ対象とする
                //if (value == "virtual" || value == "pure-virtual")
                if (value == "virtual")
                {
                    ClassXmlIsVirtual = true;
                }
                else
                {
                    ClassXmlIsVirtual = false;
                }
            }
            return true;
        }

        private static bool ClassXml_DefinitionHandler(string value)
        {
            ClassXmlFunctionDefinitionName = value;
            return true;
        }

        private static bool ClassXml_ArgsstringHandler(string value)
        {
            string argsName = value;

            ClassXmlFunctionDefinitionArgument = argsName;
            return true;
        }

        private static bool ClassXml_BriefdescriptionHandler(string value)
        {
            string trimValue = value.Trim();
            if (trimValue != null && trimValue != string.Empty)
            {
                ClassXmlIsDocumented = true;
            }
            return true;
        }

        private static bool ClassXml_DetaileddescriptionFuncHandler(string value)
        {
            string trimValue = value.Trim();
            if (trimValue != null && trimValue != string.Empty)
            {
                ClassXmlIsDocumented = true;
            }
            return true;
        }

        private static bool ClassXml_InbodydescriptionFuncHandler(string value)
        {
            string trimValue = value.Trim();
            if (trimValue != null && trimValue != string.Empty)
            {
                ClassXmlIsDocumented = true;
            }
            return true;
        }

        private static void CheckOverloadFunctions()
        {
            foreach (FuncCountInfo inf in functionCountList)
            {
#if false
                if(inf.NamespaceName.Contains("nw::"))
                {
                    continue;
                }
#endif
                if (inf.RefIdList.Count > 1)
                {
                    LOG.Log("overload functions: {0}::{1} ({2} times)\n", inf.NamespaceName, inf.FunctionName, inf.RefIdList.Count);
                    foreach (FuncDefinitionInfo defs in inf.DefinitionList)
                    {
                        LOG.Log("    definition : {0}::{1}\n", defs.FunctionName, defs.ArgumentName);
                    }
                }
            }
        }

        private static void ListRelateFunctions()
        {
            List<FuncCountInfo> tmpInfoList = new List<FuncCountInfo>();
            Dictionary<int, bool> checkUsed = new Dictionary<int, bool>();
            tmpInfoList.AddRange(functionCountList);
            functionCountList.Clear();
            for (int i = 0; i < tmpInfoList.Count; i++)
            {
                checkUsed[i] = false;
            }
            for (int i = 0; i < tmpInfoList.Count; i++)
            {
                if (checkUsed[i] == true)
                {
                    continue;
                }
                FuncCountInfo info1 = tmpInfoList[i];
                for (int j = i + 1; j < tmpInfoList.Count; j++)
                {
                    if (checkUsed[j] == true)
                    {
                        continue;
                    }
                    FuncCountInfo info2 = tmpInfoList[j];
                    if (info1.FunctionName == info2.FunctionName)
                    {
                        string templateLessNamespaceName1 = SkipTemplateName(info1.NamespaceName);
                        string templateLessNamespaceName2 = SkipTemplateName(info2.NamespaceName);
                        if (templateLessNamespaceName1 == templateLessNamespaceName2)
                        {
                            info1.RegistRelateInfo(info2);
                            checkUsed[j] = true;
                        }
                    }
                }
                functionCountList.Add(info1);

                checkUsed[i] = true;
            }
        }

        private static void MakeOverloadlistHD()
        {
            if (documentBasePath == null)
            {
                LOG.Log(LOG.Level.LOG_INFO, "SetPath() is not set\n");
                return;
            }

            // 保存先のパスを作成する
            string outputpath = documentBasePath + Common.UtilFs.GetGeneratedSourcesDirName() + "\\overloadlist.hd";

            Common.UtilFs.FileCheckDelete(outputpath);
            FileCheckParentDirectoryExist(outputpath);

            // 場所情報リストから保存先の内容を作成する
            MakeOverloadHdCore(outputpath);
        }

        private static bool MakeOverloadHdCore(string outputpath)
        {
            Encoding encode = Encoding.GetEncoding("UTF-8");
            using (StringWriter sw = new StringWriter())
            {
                sw.NewLine = "\r\n";

                // まずコピーライトヘッダを出力する
                string resourceBytes = Properties.Resources.headerTxt;
                sw.Write(resourceBytes);

                // 次に以下のようなコメントを出力する
                /**
                 * @fn nn::Result nn::os::CreateThread( ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority);
                 *
                 * @overloadlisti{nn_os_CreateThread}
                 */

                // virtual なAPIの場合はdefinitionNameをfunctionNameに変換する
                foreach (FuncCountInfo info in functionCountList)
                {
                    foreach (FuncCountInfo relateInfo in info)
                    {
                        WriteOverloadlisti(sw, relateInfo);
                    }
                }

                File.WriteAllText(outputpath, sw.ToString(), encode);
            }

            return true;
        }

        private static void WriteOverloadlisti(StringWriter sw, FuncCountInfo relateInfo)
        {
            foreach (FuncDefinitionInfo defInfo in relateInfo.DefinitionList)
            {
                if (defInfo.IsDerived == false && defInfo.IsVirtual == false)
                {
                    string apiDefinitionName = defInfo.FunctionName + defInfo.ArgumentName;
                    string overloadclistiName = GetOverloadcApiName(relateInfo);
                    sw.WriteLine(string.Empty);
                    sw.Write("/**" + Environment.NewLine + " * @fn " + apiDefinitionName + ";" + Environment.NewLine);
                    sw.Write(" * " + Environment.NewLine + " * ");
                    sw.Write("@overloadlisti{" + overloadclistiName + "}" + Environment.NewLine + " */" + Environment.NewLine);
                }
            }
        }

        private static string FilterTemplateName(string name)
        {
            if (name.Contains('<') && name.Contains('>'))
            {
                int index = name.IndexOf('<');
                return name.Substring(0, index);
            }
            return name;
        }

        /// <summary>
        /// overload.hd, overloadlist.hd ファイルを生成します
        /// </summary>
        public static void MakeHdFiles()
        {
            MakeOverloadlistHD();
            MakeOverloadHD();
        }

        /// <summary>
        /// overload.hd ファイルを生成します
        /// </summary>
        private static void MakeOverloadHD()
        {
            if (documentBasePath == null)
            {
                LOG.Log(LOG.Level.LOG_INFO, "SetPath() is not set\n");
                return;
            }

            // 保存先のパスを作成する
            string outputpath = documentBasePath + Common.UtilFs.GetGeneratedSourcesDirName() + "\\overload.hd";

            Common.UtilFs.FileCheckDelete(outputpath);
            FileCheckParentDirectoryExist(outputpath);

            // 場所情報リストから保存先の内容を作成する
            MakeOverloadlistHdCore(outputpath);
        }

        private static void FileCheckParentDirectoryExist(string path)
        {
            // 親ディレクトリの存在をチェックしてないならば作る
            FileInfo fInfo = new FileInfo(path);
            if (!File.Exists(fInfo.DirectoryName))
            {
                Directory.CreateDirectory(fInfo.DirectoryName);
            }
        }

        private static bool MakeOverloadlistHdCore(string outputpath)
        {
            Encoding encode = Encoding.GetEncoding("UTF-8");
            //using (StreamWriter sw = new StreamWriter(outputpath, false, encode))
            using (StringWriter sw = new StringWriter())
            {
                sw.NewLine = "\r\n";

                // ヘッダ部と最初の@pageを出力する
                WriteHdHeaderAndPageSection1(sw);

                foreach (FuncCountInfo info in functionCountList)
                {
                    if (info.RefIdList.Count <= 1)
                    {
                        continue;
                    }

                    bool isInheritanceFunc = info.HasDerivedFunction | info.HasVirtualFunction;
                    bool isAllInheritanceFunc = true;
                    FuncDefinitionInfo firstParentInfo = new FuncDefinitionInfo();
                    bool isFirstParent = false;

                    if (isInheritanceFunc == true)
                    {
                        foreach (FuncCountInfo relateinfo in info)
                        {
                            foreach (FuncDefinitionInfo defInfo in relateinfo.DefinitionList)
                            {
                                if (defInfo.IsDerived == true || defInfo.IsVirtual == true)
                                {
                                    if (isFirstParent == false)
                                    {
                                        firstParentInfo = defInfo;
                                        isFirstParent = true;
                                    }
                                }
                                else
                                {
                                    isAllInheritanceFunc = false;
                                }
                            }
                        }

                        if (isAllInheritanceFunc == true)
                        {
                            WriteHdPageSection2_Inherit(sw, info, firstParentInfo);
                        }
                        else
                        {
                            WriteHdPageSection2_withInherit(sw, info, firstParentInfo);
                        }
                    }
                    else
                    {
                        // API毎の@pageを出力する
                        WriteHdPageSection2(sw, info);
                    }
                }

                File.WriteAllText(outputpath, sw.ToString(), encode);
            }

            return true;
        }

        private static void WriteHdHeaderAndPageSection1(StringWriter sw)
        {
            // コピーライトヘッダを出力する
            string resourceBytes = Properties.Resources.headerTxt;
            sw.Write(resourceBytes);

            // 次に以下のようなコメントを出力する
            /**
             *
             * @page overloadlist オーバーロードAPI一覧
             * @par
             *
             * オーバーロードしているAPIの一覧です。@n@n
             *
             */
            sw.WriteLine(string.Empty);
            sw.Write("/**" + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write("@page overloadlist " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_NAME] + Environment.NewLine);
            sw.Write(" * @par" + Environment.NewLine + " * " + Environment.NewLine + " * " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_EXPLAIN1] + "@n@n" + Environment.NewLine);
            sw.Write(" * " + Environment.NewLine + " */" + Environment.NewLine);
        }

        private static void WriteHdPageSection2(StringWriter sw, FuncCountInfo info)
        {
            string overloadcApiName = GetOverloadcApiName(info);
            string cppApiName = GetCppApiName(info);

            // 以下のようなコメントを出力する
            /**
             *
             * @page overloadlist オーバーロードAPI一覧
             * @subpage overloadc_nn_os_CreateThread
             *
             *
             * @page overloadc_nn_os_CreateThread nn::os::CreateThread
             * @par
             *
             * これらのAPIはオーバーロードしています。
             *
             */
            sw.WriteLine(string.Empty);
            sw.Write("/**" + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write("@page overloadlist " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_NAME] + Environment.NewLine + " * ");
            sw.Write(string.Format("@subpage overloadc_{0}" + Environment.NewLine, overloadcApiName));
            sw.Write(" * " + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write(string.Format("@page overloadc_{0} {1}" + Environment.NewLine, overloadcApiName, cppApiName));
            sw.Write(" * @par" + Environment.NewLine + " * " + Environment.NewLine + " * " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_EXPLAIN2] + Environment.NewLine);
            sw.Write(" * " + Environment.NewLine + " */" + Environment.NewLine);
        }

        private static void WriteHdPageSection2_Inherit(StringWriter sw, FuncCountInfo info, FuncDefinitionInfo parentDefInfo)
        {
            string overloadcApiName = GetOverloadcApiName(info);
            string cppApiName = GetCppApiName(info);
            string parentCppApiName = GetParentCppApiName(parentDefInfo, info);

            // 以下のようなコメントを出力する
            /**
             *
             * @page overloadlist オーバーロードAPI一覧
             * @subpage overloadc_nw_lyt_Animator_SetResource
             *
             *
             * @page overloadc_nw_lyt_Animator_SetResource nw::lyt::Animator::SetResource
             * @par
             *
             * このAPIはオーバーロードしています。
             * 全て派生したAPIですので、以下の派生元クラスのAPIを参照して下さい。
             * @li virtual void nw::lyt::AnimTransformBasic::SetResource(const res::AnimationBlock *pRes, ResourceAccessor* pResAccessor)
             *
             */
            sw.WriteLine(string.Empty);
            sw.Write("/**" + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write("@page overloadlist " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_NAME] + Environment.NewLine + " * ");
            sw.Write(string.Format("@subpage overloadc_{0}" + Environment.NewLine, overloadcApiName));
            sw.Write(" * " + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write(string.Format("@page overloadc_{0} {1}" + Environment.NewLine, overloadcApiName, cppApiName));
            sw.Write(" * @par" + Environment.NewLine + " * " + Environment.NewLine + " * " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_API_EXPLAIN1_1] + Environment.NewLine);
            sw.Write(" * " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_API_EXPLAIN1_2] + Environment.NewLine);
            foreach (FuncCountInfo relateInfo in info)
            {
                foreach (FuncDefinitionInfo definfo in relateInfo.DefinitionList)
                {
                    if (definfo.IsDerived == true || definfo.IsVirtual == true)
                    {
                        sw.Write(" * @li " + definfo.FunctionName + definfo.ArgumentName + Environment.NewLine);
                    }
                }
            }
            sw.Write(" * " + Environment.NewLine + " */" + Environment.NewLine);
        }

        private static void WriteHdPageSection2_withInherit(StringWriter sw, FuncCountInfo info, FuncDefinitionInfo parentDefInfo)
        {
            string overloadcApiName = GetOverloadcApiName(info);
            string cppApiName = GetCppApiName(info);
            string parentCppApiName = GetParentCppApiName(parentDefInfo, info);

            // 以下のようなコメントを出力する
            /**
             *
             * @page overloadlist オーバーロードAPI一覧
             * @subpage overloadc_nw_lyt_TextBox_GetMaterial
             *
             *
             * @page overloadc_nw_lyt_TextBox_GetMaterial nw::lyt::TextBox::GetMaterial
             * @par
             *
             * このAPIはオーバーロードしています。
             * このクラスで定義されたもの以外に派生したAPIもあります。以下の派生元クラスのAPIも参照して下さい。
             * @li Material* nw::lyt::Pane::GetMaterial() const
             *
             */
            sw.WriteLine(string.Empty);
            sw.Write("/**" + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write("@page overloadlist " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_LIST_PAGE_NAME] + Environment.NewLine + " * ");
            sw.Write(string.Format("@subpage overloadc_{0}" + Environment.NewLine, overloadcApiName));
            sw.Write(" * " + Environment.NewLine + " * " + Environment.NewLine + " * ");
            sw.Write(string.Format("@page overloadc_{0} {1}" + Environment.NewLine, overloadcApiName, cppApiName));
            sw.Write(" * @par" + Environment.NewLine + " * " + Environment.NewLine + " * " + word[CmdlineOption.CountryMode][WordType.OVERLOAD_API_EXPLAIN2_1] + Environment.NewLine);
            sw.Write(word[CmdlineOption.CountryMode][WordType.OVERLOAD_API_EXPLAIN2_2] + Environment.NewLine);
            foreach (FuncCountInfo relateInfo in info)
            {
                foreach (FuncDefinitionInfo definfo in relateInfo.DefinitionList)
                {
                    if (definfo.IsDerived == true || definfo.IsVirtual == true)
                    {
                        sw.Write(" * @li " + definfo.FunctionName + definfo.ArgumentName + Environment.NewLine);
                    }
                }
            }
            sw.Write(" * " + Environment.NewLine + " */" + Environment.NewLine);
        }

        private static bool CheckOverloadlistArgument(string line, string paramString)
        {
            if (paramString != null)
            {
                // LOG.Log(LOG.Level.LOG_INFO, "found string = {0}\n", paramString);
                if (paramString.Contains(":"))
                {
                    // @overloadlist の引数がC++スタイルの場合は警告を出力する
                    LOG.Log(LOG.Level.LOG_WARN, "@overloadlist doesn't support delimiter colon ({0})", paramString);
                    return false;
                }
            }
            else
            {
                // {}がない場合はエラー
                LOG.Log(LOG.Level.LOG_WARN, "Argument of @overloadlist is NOT FOUND : {0}\n", line);
                return false;
            }
            return true;
        }

        private static string GetOverloadcApiName(FuncCountInfo info)
        {
            return SkipTemplateName(info.NamespaceName).Replace("::", "_") + "_" + info.FunctionName;
        }
        private static string GetCppApiName(FuncCountInfo info)
        {
            return SkipTemplateName(info.NamespaceName) + "::" + info.FunctionName;
        }
        private static string GetParentCppApiName(FuncDefinitionInfo parentDefInfo, FuncCountInfo info)
        {
            return FilterFunctionObjectName(SkipReturnType(SkipTemplateName(parentDefInfo.FunctionName))).Replace("::", "_");
        }

        private static string SkipTemplateName(string namespaceName)
        {
            if (namespaceName.Contains('<') == false)
            {
                return namespaceName;
            }
            int left = namespaceName.IndexOf('<');
            int right = namespaceName.LastIndexOf('>');
            string returnString = namespaceName.Substring(0, left);
            if (right + 1 < namespaceName.Length)
            {
                returnString += namespaceName.Substring(right + 1);
            }
            return returnString;
        }

        private static string SkipReturnType(string definitionName)
        {
            char[] splitChar = { ' ' };
            string[] name = definitionName.Split(splitChar, StringSplitOptions.RemoveEmptyEntries);
            return name[name.Count() - 1];
        }
    }
}
