﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Markup;
using System.Windows.Media;
using System.Windows.Media.Imaging;

namespace Nintendo.DotNetLocalizer
{
    public class SnapshotGenerator
    {
        private string outputFolder = string.Empty;

        public SnapshotGenerator(string outputFolder)
        {
            this.outputFolder = outputFolder;
        }

        public IDictionary<string, string> Execute(FileInfo asmFile, bool japanese)
        {
            var imgFileDic = new Dictionary<string, string>();

            CultureInfo keepCultureInfo = null;

            try
            {
                Output.DebugWriteLine(" SnapshotGenerator {0} - ", japanese);

                string assemblyName = Path.GetFileNameWithoutExtension(asmFile.Name);

                if (japanese == false)
                {
                    keepCultureInfo = Thread.CurrentThread.CurrentUICulture;
                    Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", false);
                }

                var assembly = Assembly.Load(assemblyName);

                // Applicationを先に読み込ませたいため、app.bamlを先に持ってきます。
                // Application が App.xaml というファイル名で定義されている必要があります。
                // (現状の制限)
                var sortedBamls =
                    from resourcesName in AssemblyUtility.EnumerateResourceName(assembly)
                    from entry in AssemblyUtility.EnumerateResouces(assembly.GetManifestResourceStream(resourcesName))
                    where DllGenerator.IsResourceEntryBamlStream(entry.Key, entry.Value)
                    let bamlPath = entry.Key
                    orderby Path.GetFileName(bamlPath).Equals("app.baml", StringComparison.OrdinalIgnoreCase) ? 0 : 1
                    select bamlPath;
                foreach (var bamlPath in sortedBamls)
                {
                    var xamlPath = Path.ChangeExtension(bamlPath.Replace('/', '\\'), ".xaml");
                    string snapshotPath =
                        CreateSnapshot(assemblyName, xamlPath, japanese);

                    if (snapshotPath != string.Empty)
                    {
                        Output.DebugWriteLine(" snapshotPath {0} - ", snapshotPath);
                        imgFileDic.Add(bamlPath, snapshotPath);
                    }
                }
            }
            finally
            {
                if (keepCultureInfo != null)
                {
                    Thread.CurrentThread.CurrentUICulture = keepCultureInfo;
                }
            }

            return imgFileDic;
        }

        /// <summary>
        /// xmal 画像作成
        /// </summary>
        private string CreateSnapshot(string assemblyName, string xamlPath, bool japanese)
        {
            Window window = null;
            bool createWindow = false;
            var xamlImagePath =
                Path.Combine(
                    "images",
                    japanese ? "ja" : "en",
                    assemblyName,
                    xamlPath + ".jpg");

            var uriString = "/" + assemblyName + ";component/" + Utility.BackSlashToSlash(xamlPath);
            Uri uri = new Uri(uriString, UriKind.Relative);
            try
            {
                object obj = Application.LoadComponent(uri);
                if (obj is UIElement == true)
                {
                    window = obj as Window;
                    if (window == null)
                    {
                        window = new Window();
                        createWindow = true;
                    }
                    else
                    {
                        obj = window.Content;
                        window.Content = null;
                    }
                    var userControl = new UserControl();
                    userControl.Background = Brushes.White;
                    userControl.Content = obj;
                    window.Content = userControl;
                    window.Show();
                    int width = (int)(userControl.ActualWidth);
                    int height = (int)(userControl.ActualHeight);
                    var bmp =
                        new RenderTargetBitmap(
                            width,
                            height,
                            96.0,
                            96.0,
                            PixelFormats.Pbgra32);
                    bmp.Render(userControl as UIElement);
                    var fullPath = Path.Combine(this.outputFolder, xamlImagePath);
                    var outputImageFolder = Path.GetDirectoryName(fullPath);
                    Directory.CreateDirectory(outputImageFolder);
                    using (var stream = new FileStream(fullPath, FileMode.Create))
                    {
                        JpegBitmapEncoder encoder = new JpegBitmapEncoder();
                        encoder.QualityLevel = 95;
                        encoder.Frames.Add(BitmapFrame.Create(bmp));
                        encoder.Save(stream);
                    }
                    window.Hide();
                }
                else
                {
                    xamlImagePath = string.Empty;
                }
            }
            catch (XamlParseException)
            {
                Output.DebugWriteLine("Error snapshot - {0}", xamlPath);
                xamlImagePath = string.Empty;
            }
            finally
            {
                if (createWindow == true)
                {
                    window.Close();
                }
            }

            return xamlImagePath;
        }
    }
}
