﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;

namespace Nintendo.DotNetLocalizer
{
    [Serializable]
    public class LocalizationItem
    {
        static readonly LocalizationItem[] EmptyArrayField = new LocalizationItem[0];

        private string assemblyName;

        private string resourceFile;

        private string resourceName;

        private string text;

        private bool isNew;

        public static LocalizationItem[] EmptyArray { get { return EmptyArrayField; } }

        [XmlAttribute]
        public string AssemblyName
        {
            get { return assemblyName; }
            set { assemblyName = value; }
        }

        [XmlAttribute]
        public string ResourceFile
        {
            get { return resourceFile; }
            set { resourceFile = value; }
        }

        [XmlAttribute]
        public string ResourceName
        {
            get { return resourceName; }
            set { resourceName = value; }
        }

        public string Text
        {
            get { return text; }
            set { text = value; }
        }

        [XmlAttribute]
        [DefaultValue(false)]
        public bool IsNew
        {
            get { return isNew; }
            set { isNew = value; }
        }

        public LocalizationItem()
        {
        }

        public LocalizationItem(string assemblyName, string resourceFile, string resourceName, string text, bool isNew = false)
        {
            AssemblyName = assemblyName;
            ResourceFile = resourceFile;
            ResourceName = resourceName;
            Text = text;
            IsNew = isNew;
        }

        public static IEnumerable<LocalizationItemPair> JoinLocalizeItemPair(IEnumerable<LocalizationItem> locItemsJa, IEnumerable<LocalizationItem> locItemsEn)
        {
            return
                from locItemJa in locItemsJa
                join locItemEn in locItemsEn
                    on new { AssemblyName = locItemJa.AssemblyName.ToUpperInvariant(), ResourceFile = locItemJa.ResourceFile.ToUpperInvariant(), locItemJa.ResourceName }
                    equals new { AssemblyName = locItemEn.AssemblyName.ToUpperInvariant(), ResourceFile = locItemEn.ResourceFile.ToUpperInvariant(), locItemEn.ResourceName }
                select new LocalizationItemPair(locItemJa, locItemEn);
        }

        public static IEnumerable<LocalizationItem> UpdateLocalizeItem(IEnumerable<LocalizationItem> locItemsJa, IEnumerable<LocalizationItem> locItemsEn, IEnumerable<LocalizationItem> refLocItemsJa, bool checkNew)
        {
            var enumLocItemPairs = JoinLocalizeItemPair(locItemsJa, locItemsEn);

            if (!checkNew)
            {
                return
                    from newItemJa in refLocItemsJa
                    join pair in enumLocItemPairs
                        on new { AssemblyName = newItemJa.AssemblyName.ToUpperInvariant(), ResourceFile = newItemJa.ResourceFile.ToUpperInvariant(), newItemJa.ResourceName, newItemJa.Text }
                        equals new { AssemblyName = pair.Ja.AssemblyName.ToUpperInvariant(), ResourceFile = pair.Ja.ResourceFile.ToUpperInvariant(), pair.Ja.ResourceName, pair.Ja.Text }
                        into g
                    let innerPair = g.SingleOrDefault()
                    select new LocalizationItem(newItemJa.AssemblyName, newItemJa.ResourceFile, newItemJa.ResourceName, null != innerPair ? innerPair.En.Text : null);
            }
            else
            {
                return
                    from pair in enumLocItemPairs
                    join oldItemJa in refLocItemsJa
                        on new { AssemblyName = pair.Ja.AssemblyName.ToUpperInvariant(), ResourceFile = pair.Ja.ResourceFile.ToUpperInvariant(), pair.Ja.ResourceName, pair.Ja.Text }
                        equals new { AssemblyName = oldItemJa.AssemblyName.ToUpperInvariant(), ResourceFile = oldItemJa.ResourceFile.ToUpperInvariant(), oldItemJa.ResourceName, oldItemJa.Text }
                        into g
                    let isNew = null == g.SingleOrDefault()
                    select new LocalizationItem(pair.Ja.AssemblyName, pair.Ja.ResourceFile, pair.Ja.ResourceName, pair.En.Text, isNew);
            }
        }
    }

    public class LocalizationItems
    {
        private LocalizationItem[] localizationItem;

        [XmlElement]
        public LocalizationItem[] LocalizationItem
        {
            get { return localizationItem; }
            set { localizationItem = value; }
        }
    }

    public class LocalizationItemPair
    {
        public LocalizationItemPair(LocalizationItem ja, LocalizationItem en)
        {
            Ja = ja;
            En = en;
        }

        public LocalizationItem Ja { get; private set; }

        public LocalizationItem En { get; private set; }
    }

    public class LocalizeItemXmlSerializer
    {
        public static LocalizationItem[] ReadXml(string path)
        {
            //書き込むファイルを開く（UTF-8 BOM無し）
            using (var sr = new StreamReader(path, Encoding.UTF8))
            {
                return ReadXml(sr);
            }
        }

        public static void WriteXml(string path, IEnumerable<LocalizationItem> items)
        {
            using (var sr = new StreamWriter(path, false, Encoding.UTF8))
            {
                WriteXml(sr, items);
            }
        }

        private static LocalizationItem[] ReadXml(System.IO.TextReader tr)
        {
            //オブジェクトの型を指定する
            var serializer = new XmlSerializer(typeof(LocalizationItems));

            //書き込むファイルを開く（UTF-8 BOM無し）
            var locItems = (LocalizationItems)serializer.Deserialize(tr);
            return locItems.LocalizationItem ?? new LocalizationItem[0];
        }

        private static void WriteXml(TextWriter sw, IEnumerable<LocalizationItem> strRess)
        {
            var stringResources = new LocalizationItems();
            stringResources.LocalizationItem = strRess.ToArray();

            // メモ:
            // .NET 4.5 以前では、XmlSerializer は既定では一時フォルダにシリアライズ用のcsファイルを生成する。
            // それを回避するためには、シリアル化アセンブリ(～.XmlSerializers.dll)を作る。
            // そのための設定は次の通り。
            //     1. プロジェクトの「ビルド」->「シリアル化アセンブリの生成」を「ON」
            //     2. csproj を直接編集して、次のプロパティを追加。
            //         <SGenUseProxyTypes>false</SGenUseProxyTypes>    # sgen.exe オプションの /proxytypes を有効にしない。
            //         <SGenSerializationTypes>Nintendo.DotNetLocalizer.LocalizeItems</SGenSerializationTypes>    # sgen.exe の /type オプション

            var serializer = new XmlSerializer(typeof(LocalizationItems));

            // 明示的にXmlWriterを作成する。
            // TextWriterを直接渡すと、内部で作られるXmlWriterでは改行の正規化が行われない。
            var settings = new XmlWriterSettings
            {
                Indent = true,
            };
            var xw = XmlWriter.Create(sw, settings);
            //書き込むファイルを開く（UTF-8 BOM無し）
            serializer.Serialize(xw, stringResources);
        }
    }
}
