﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Resources;
using System.Text.RegularExpressions;
using System.Windows.Markup.Localizer;

namespace Nintendo.DotNetLocalizer
{
    internal class AssemblyUtility
    {
        public static IEnumerable<string> EnumerateResourceName(Assembly assembly)
        {
            return
                from resourcesName in assembly.GetManifestResourceNames()
                where resourcesName.EndsWith(".resources", StringComparison.OrdinalIgnoreCase)
                select resourcesName;
        }

        public static IEnumerable<KeyValuePair<string, object>> EnumerateResouces(Stream stream)
        {
            using (var resReader = new ResourceReader(stream))
            {
                // 全エントリを順に処理する
                foreach (DictionaryEntry entry in resReader)
                {
                    object value = null;
                    try
                    {
                        value = entry.Value;
                    }
                    catch (FileNotFoundException)
                    {
                        Output.DebugWriteLine("Exception: A file cannot be found. - {0}", entry.Key);
                    }
                    catch (FormatException)
                    {
                        Output.DebugWriteLine("Exception: Corrupted data. - {0}", entry.Key);
                    }
                    catch (TypeLoadException)
                    {
                        Output.DebugWriteLine("Exception: Cannot load the data type. - {0}", entry.Key);
                    }

                    if (null != value)
                    {
                        yield return
                            new KeyValuePair<string, object>((string)entry.Key, value);
                    }
                }
            }
        }

        public static string GetKeyString(BamlLocalizableResourceKey key)
        {
            return
                string.Format(
                                "{0}:{1}.{2}",
                                key.Uid,
                                key.ClassName,
                                key.PropertyName);
        }

        public static FileInfo[] GetAsmFiles(string applicationBase, string[] excludes, string privateBinPath)
        {
            var excludeAsmRegex =
                null != excludes && excludes.Length > 0 ?
                    new Regex(
                        GetExcludesString(excludes),
                        RegexOptions.IgnoreCase
                        | RegexOptions.CultureInvariant
                        | RegexOptions.Compiled) :
                    null;

            // ApplicationBase 直下のアセンブリとなるファイル(.exe,.dll)を
            // 列挙します。
            // スクリーンショットを取る際に App.bamlを先に読み込む必要があるため、
            // exe を最初の方に持ってきます。
            var rootFiles =
                from fileInfo in new DirectoryInfo(applicationBase).EnumerateFiles()
                let fileExt = fileInfo.Extension.ToUpperInvariant()
                where (fileExt == ".EXE" || fileExt == ".DLL")
                    && (null == excludeAsmRegex || !excludeAsmRegex.IsMatch(Path.GetFileNameWithoutExtension(fileInfo.Name)))
                    && IsDotNetAssembly(fileInfo.FullName)
                orderby // exe を最初に持ってきます。
                    fileExt == ".EXE" ? 0 : 1,
                    fileInfo.Name
                select fileInfo;

            // プライベートパスを列挙します。
            var enumPrivatePath =
                string.IsNullOrEmpty(privateBinPath) ?
                    Enumerable.Empty<string>() :
                    privateBinPath.Split(';');

            // プライベートパスにあるアセンブリとなるファイル(.dll)を列挙します。
            var priFiles =
                from dir in enumPrivatePath
                let di = new DirectoryInfo(Path.Combine(applicationBase, dir))
                from fileInfo in di.EnumerateFiles("*.dll")
                where (null == excludeAsmRegex || !excludeAsmRegex.IsMatch(Path.GetFileNameWithoutExtension(fileInfo.Name)))
                    && IsDotNetAssembly(fileInfo.FullName)
                select fileInfo;

            // ApplicationBase 直下のアセンブリとプライベートパスにあるアセンブリの列挙を
            // 繋げて固定化(配列)します。
            return rootFiles.Concat(priFiles).ToArray();
        }

        /// <summary>
        /// アセンブリかどうかを判定します。
        /// </summary>
        /// <param name="filePath">反転するファイルのパス。</param>
        /// <returns>アセンブリの場合は真。</returns>
        private static bool IsDotNetAssembly(string filePath)
        {
            // アセンブリかどうかテスト
            try
            {
                AssemblyName.GetAssemblyName(filePath);
            }
            catch (BadImageFormatException)
            {
                return false;
            }
            return true;
        }

        /// <summary>
        /// 除外アセンブリファイルの指定を1つの文字列につなげたものを取得します。
        /// </summary>
        /// <returns></returns>
        private static string GetExcludesString(string[] excludes)
        {
            return
                null != excludes && excludes.Length > 0 ?
                    string.Join(
                        "|",
                        (from asmName in excludes
                         select "(" + asmName + ")")) :
                    null;
        }
    }
}
