﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;

namespace MakeDictionarySource
{
    /// <summary>
    /// DictionaryCpp ファイルの文字列データを生成するためのクラスです。
    /// </summary>
    class DictionaryCpp
    {
        private readonly bool m_Dummy;
        private readonly string m_ModuleName;
        private readonly string m_AdditionalInclude;
        private readonly string m_InputPath;

        /// <summary>
        /// 辞書データがダミーの DictionaryCpp オブジェクトを生成
        /// </summary>
        /// <param name="moduleName">シンボル名</param>
        /// <param name="additionalInclude">追加のインクルードファイル</param>
        public DictionaryCpp(string moduleName, string additionalInclude)
        {
            if (moduleName == null)
            {
                throw new ArgumentNullException();
            }

            m_Dummy = true;
            m_ModuleName = moduleName;
            m_AdditionalInclude = additionalInclude;
            m_InputPath = null;
        }

        /// <summary>
        /// inputPath の .mo ファイルを辞書データとする DictionaryCpp オブジェクトを生成
        /// </summary>
        /// <param name="moduleName">シンボル名</param>
        /// <param name="additionalInclude">追加のインクルードファイル</param>
        /// <param name="inputPaths">mo ファイルのパスの配列</param>
        public DictionaryCpp(string moduleName, string additionalInclude, string inputPath)
        {
            if (moduleName == null || inputPath == null)
            {
                throw new ArgumentNullException();
            }

            m_Dummy = false;
            m_ModuleName = moduleName;
            m_AdditionalInclude = additionalInclude;
            m_InputPath = inputPath;
        }

        /// <summary>
        /// クラスの中身を C++ ソースコードの形で返します。
        /// </summary>
        /// <returns>C++ ソースコード</returns>
        public override string ToString()
        {
            string includeExpression = Utils.GetIncludeExpression(m_AdditionalInclude);
            string arrayName = Utils.GetArrayName(m_ModuleName);
            string functionName = Utils.GetFunctionName(m_ModuleName);

            string implementation;
            if (m_Dummy)
            {
                implementation = String.Format(DummyImplementationFormat, functionName);
            }
            else
            {
                string arrayBody = ConvertBinaryToArrayBody(m_InputPath);
                implementation = String.Format(ImplementationFormat, arrayName, arrayBody, functionName);
            }

            return String.Format(CppFileFormat, includeExpression, implementation);
        }

        /// <summary>
        /// 辞書データを C++ 配列定義に使える形に変換します。
        /// </summary>
        /// <param name="binaryPath">バイナリファイルへのパス</param>
        /// <returns>C++ 配列定義で {} 内に記述する部分</returns>
        private string ConvertBinaryToArrayBody(string binaryPath)
        {
            StringBuilder builder = new StringBuilder();

            using (var fs = new FileStream(binaryPath, FileMode.Open))
            {
                int data;
                while ((data = fs.ReadByte()) != -1)
                {
                    builder.AppendLine(String.Format("        0x{0:x2},", data));
                }
            }

            return builder.ToString();
        }

        // ソースコードのテンプレート文字列
        // {0}: 追加のインクルード
        // {1}: 実装
        private const string CppFileFormat = @"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/diag/detail/diag_DetailTranslation.h>
{0}

/*
* @file
* @brief    アサート・ログメッセージ翻訳用の辞書にアクセスする API の実装です。
*           このファイルは自動生成されたものなので、変更しないでください。
*/

namespace nn {{
namespace diag {{
namespace detail {{
    {1}
}}
}}
}}
";

        // 実装のテンプレート文字列
        // {0}: 配列名
        // {1}: 配列のデータ
        // {2}: メソッド名
        private const string ImplementationFormat = @"const uint8_t {0}[] =
    {{
{1}
    }};

    const char* {2}(const char* string)
    {{
        return nn::diag::detail::Translate({0}, string);
    }}";

        // ダミー実装のテンプレート文字列
        // {0}: メソッド名
        private const string DummyImplementationFormat = @"const char* {0}(const char* string)
    {{
        return string;
    }}";
    }
}
