﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading.Tasks;
using CredentialRegister.Exceptions;

namespace CredentialRegister
{
    public class Program
    {
        public static void Main(string[] args)
        {
#if DEBUG
            Execute();
#else
            try
            {
                Execute(args);
            }
            catch (CredentialRegisterException e)
            {
                Console.Error.WriteLine("Error: {0}", e.Message);

                Environment.Exit(1);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(string.Empty);
                Console.Error.WriteLine("Internal error: {0}", e.Message);
                Console.Error.WriteLine(string.Empty);
                Console.Error.WriteLine("== Exception ==");
                Console.Error.WriteLine(string.Empty);
                Console.Error.WriteLine(e.ToString());

                Environment.Exit(1);
            }
#endif
        }

        private static void Execute(string[] args)
        {
            if (args.Length == 2)
            {
                var username = args[0];
                var password = args[1];

                NonInteractive(username, password);
            }
            else
            {
                Interactive();
            }
        }

        private static void NonInteractive(string username, string password)
        {
            string targetname = GetCredentialTargetName();
            string sourceUri = GetSourceUri();

            var authenticationInfo = new Credential.AuthenticationInfo
            {
                Username = username,
                Password = password
            };

            if (IsAuthenticationInfoValid(authenticationInfo, sourceUri))
            {
                // 認証に成功しました。
                Console.WriteLine("Succeeded in authentication using the account.");

                // 登録済みのアカウント情報を用いていない場合は、アカウント情報を保存します。
                Console.WriteLine("Save the account.");

                Credential.SaveAuthenticationInfo(targetname, authenticationInfo);
            }
            else
            {
                Console.WriteLine("Failed to authentication using the account.");
            }
        }

        private static void Interactive()
        {
            string targetname = GetCredentialTargetName();
            string sourceUri = GetSourceUri();

            Credential.AuthenticationInfo authenticationInfo;

            Console.WriteLine("Find an account.");
            var found = Credential.FindAuthenticationInfo(targetname, out authenticationInfo);
            if (!found)
            {
                Console.WriteLine("Could not find an account.");
            }

            while (true)
            {
                if (!found)
                {
                    Console.WriteLine("Please input your account.");
                    if (!Credential.RequireAuthenticationInfo(out authenticationInfo, "Siglo account info", "Please input your account for Siglo JIRA/Confluence/TeamCity."))
                    {
                        // キャンセルされました。
                        Console.WriteLine("Cancel an operation.");
                        break;
                    }
                }

                if (authenticationInfo.Username != string.Empty && authenticationInfo.Password != string.Empty)
                {
                    if (IsAuthenticationInfoValid(authenticationInfo, sourceUri))
                    {
                        // 認証に成功しました。
                        Console.WriteLine("Succeeded in authentication using the account.");

                        if (!found)
                        {
                            // 登録済みのアカウント情報を用いていない場合は、アカウント情報を保存します。
                            Console.WriteLine("Save the account.");

                            Credential.SaveAuthenticationInfo(targetname, authenticationInfo);
                        }

                        break;
                    }
                    else
                    {
                        Console.WriteLine("Failed to authentication using the account.");
                    }
                }

                // 認証に失敗しました。
                // アカウント情報を要求するため、クレデンシャルから読みだした情報は使いません。
                found = false;
            }
        }

        private static string GetCredentialTargetName()
        {
            return "Nintendo/Siglo Server";
        }

        private static string GetSourceUri()
        {
            return "http://devsedci01.ncl.nintendo.co.jp/TeamCity/";
        }

        private static bool IsAuthenticationInfoValid(Credential.AuthenticationInfo authenticationInfo, string sourceUri)
        {
            using (var client = new WebClient())
            {
                client.Credentials = new System.Net.NetworkCredential(authenticationInfo.Username, authenticationInfo.Password);

                try
                {
                    var response = client.DownloadString(sourceUri);
                    if (response == string.Empty)
                    {
                        return false;
                    }

                    return true;
                }
                catch (WebException e)
                {
                    throw new CredentialRegisterException(string.Format("Failed to access '{0}'.", sourceUri), e);
                }
            }
        }
    }
}
